﻿/**
 *	Bitmap structure.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.bitmap;

import drt.stream;

struct Bitmap {
	
	/// construct bitmap.
	static Bitmap opCall(void* buf, size_t len) {
		Bitmap bmp;
		bmp.buffer_ = (cast(ubyte*)buf)[0 .. len];
		bmp.buffer_[] = 0x00;
		return bmp;
	}
	
	bool opIndex(size_t i) {
		size_t bits = i & 0b0111;
		i >>= 3;
		return (buffer_[i] & (1U<<bits)) != 0;
	}
	
	bool opIndexAssign(bool b, size_t i) {
		size_t bits = i & 0b0111;
		i >>= 3;
		if(b) {
			buffer_[i] |= cast(ubyte)(1U<<bits);
		} else {
			buffer_[i] &= ~cast(ubyte)(1U<<bits);
		}
		return b;
	}
	
	size_t length() {return buffer_.length * 8;}
	
	int opApply(int delegate(inout size_t, inout bool) dg) {
		size_t len = length();
		for(size_t i = 0; i < len; ++i) {
			bool b = opIndex(i);
			int result = dg(i, b);
			opIndexAssign(b, i);
			if(result) {
				return result;
			}
		}
		return 0;
	}
	
	int opApply(int delegate(inout bool) dg) {
		size_t len = length();
		for(size_t i = 0; i < len; ++i) {
			bool b = opIndex(i);
			int result = dg(b);
			opIndexAssign(b, i);
			if(result) {
				return result;
			}
		}
		return 0;
	}
	
	int opApplyReverse(int delegate(inout size_t, inout bool) dg) {
		size_t len = length();
		for(size_t i = this.length; i >= 1; --i) {
			size_t idx = i - 1;
			bool b = opIndex(idx);
			int result = dg(idx, b);
			opIndexAssign(b, idx);
			if(result) {
				return result;
			}
		}
		return 0;
	}
	
	int opApplyReverse(int delegate(inout bool) dg) {
		size_t len = length();
		for(size_t i = this.length; i >= 1; --i) {
			bool b = opIndex(i - 1);
			int result = dg(b);
			opIndexAssign(b, i - 1);
			if(result) {
				return result;
			}
		}
		return 0;
	}
	
	void clear() {fill(false);}
	
	void fill(bool b) {buffer_[] = cast(ubyte)(b ? 0xff : 0x00);}
	
	size_t find(bool b) {return b ? findTrue() : findFalse();}
	
	void print(OutputPrintStream dest) {
		for(size_t i = 0; i < length(); ++i) {
			dest.write(opIndex(i) ? '1' : '0');
		}
	}
	
private:
	
	size_t findTrue() {
		size_t bi = buffer_.length;
		foreach(i, v; buffer_) {
			if(v != 0x00) {
				bi = i;
				break;
			}
		}
		
		if(bi == buffer_.length) {
			return length;
		}
		
		for(size_t i = 0; i < 8; ++i) {
			if(buffer_[bi] & 1U<<i) {
				return bi * 8 + i;
			}
		}
		
		//assert(false);
		return length;
	}
	
	size_t findFalse() {
		size_t bi = buffer_.length;
		foreach(i, v; buffer_) {
			if(v != 0xff) {
				bi = i;
				break;
			}
		}
		
		if(bi == buffer_.length) {
			return length;
		}
		
		for(size_t i = 0; i < 8; ++i) {
			if((buffer_[bi] & 1U<<i) == 0) {
				return bi * 8 + i;
			}
		}
		
		//assert(false);
		return length;
	}
	
	/// buffer.
	ubyte[] buffer_;
}
