﻿/**
 *	interrupt descriptor.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.interrupt;

import std.metastrings;
import std.stdint;

import os.i386.io;
import os.i386.segment;

const size_t INTERRUPT_DESCRIPTOR_TABLE_LENGTH = 256;

enum Interrupt : uint8_t {
	DIV_ERROR,
	DEBUG,
	NMI,
	BREAK_POINT,
	OVERFLOW,
	OUT_OF_BOUND,
	INVALID_OPECODE,
	MATH_COPROCESSOR,
	DOUBLE_FAULT,
	COPROCESSOR_SEGMENT_OVERRUN,
	INVALID_TSS,
	SEGMENT_NOT_PRESENT,
	STACK_SEGMENT_FAULT,
	GENERAL_PROTECTION,
	PAGE_FAULT,
	//RESERVED15,
	MATH_FAULT = 16,
	ALIGNMENT_CHECK,
	MACHINE_CHECK,
	SIMD_FAULT,
	
	PROCESSOR_RESERVED_BEGIN = 0,
	PROCESSOR_RESERVED_END = 32,
	
	IRQ_BEGIN = 32,
	
	// irq numbers.
	IRQ_SYSTEM_TIMER = 32,
	IRQ_KEYBOARD,
	IRQ_CASCADE,
	IRQ_COM2,
	IRQ_COM1,
	IRQ_LPT2,
	IRQ_FLOPPY,
	IRQ_LPT1,
	IRQ_REAL_TIME_CLOCK,
	IRQ_USABLE9,
	IRQ_USABLE10,
	IRQ_USABLE11,
	IRQ_MOUSE,
	IRQ_MATH_COPROCESSOR,
	IRQ_PRIMARY_IDE,
	IRQ_SECONDARY_IDE,
	
	IRQ_END,
	
	SYSTEM_CALL = 128,
}

struct InterruptContext {
	uint32_t edi;
	uint32_t esi;
	uint32_t ebp;
	uint32_t esp;
	uint32_t ebx;
	uint32_t edx;
	uint32_t ecx;
	uint32_t eax;
	uint32_t es;
	uint32_t ds;
	uint32_t n;
	uint32_t err;
	uint32_t eip;
	uint32_t cs;
	uint32_t eflags;
}

/// IRQ programmable interrupt controller
interface InterruptController {
	void disable();
	void enable();
	void ackInterrupt(uint n);
	void maskInterrupt(uint n);
}

/// interrupt handler function type.
alias void function(inout InterruptContext) InterruptHandler;

void loadInterruptDescriptorTable(InterruptController pic, InterruptHandler defHandler) {
	pic_ = pic;
	initInterruptDescriptorTable(defHandler);
	asm {
		lidt IDT_LIMIT;
	}
}

void enableInterrupt() {asm {sti;}}
void disableInterrupt() {asm {cli;}}

bool isEnableInterrupt() {	
	// check eflags register.
	int result = 0;
	asm {
		pushf;
		pop EAX;
		and EAX, 0x200;
		mov result, EAX;
	}
	return result != 0;
}

void duringDisableInterrupt(void delegate() dg) {
	bool enable = isEnableInterrupt();
	disableInterrupt();
	scope(exit) if(enable) enableInterrupt();
	dg();
}

enum HandlerType {
	INTERRUPT,
	TRAP,
}

enum HandlerPrivilege {
	KERNEL,
	USER,
}

void setInterruptHandler(size_t n, HandlerType t, HandlerPrivilege p, InterruptHandler h) {
	IDT[n].privilege = (p == HandlerPrivilege.KERNEL) ? InterruptDescriptor.Privilege.KERNEL : InterruptDescriptor.Privilege.USER;
	IDT[n].type = (t == HandlerType.INTERRUPT) ? InterruptDescriptor.Type.INTERRUPT_GATE : InterruptDescriptor.Type.TRAP_GATE;
	INTERRUPT_HANDLER_TABLE[n] = h;
}

InterruptController getInterruptController() {return pic_;}

private:

/// inner interrupt handler function type.
extern(C) alias void function() InnerInterruptHandler;

struct InterruptDescriptor {
	
	enum Type : uint8_t {		
		/// task gate flag
		TASK_GATE = 0b0101,
		
		/// interrupt gate flag
		INTERRUPT_GATE = 0b1110,
		
		/// trap gate flag
		TRAP_GATE = 0b1111,
		
		/// present flag
		PRESENT = 0b1000_0000,
		
		/// privilege level bits mask
		PRIVILEGE_MASK = 0b0110_0000,
		
		/// gate type mask
		TYPE_MASK = 0b0000_1111,
	}
	
	/// descriptor privilege level (DPL)
	enum Privilege {
		KERNEL = 0b0000_0000,
		SERVICE1 = 0b0010_0000,
		SERVICE2 = 0b0100_0000,
		USER = 0b0110_0000,
	}
	
	/// constructor.
	static InterruptDescriptor opCall(Type t, Privilege p, InnerInterruptHandler h) {
		InterruptDescriptor desc;
		desc.typeDPL = cast(uint8_t)(t | p | Type.PRESENT);
		desc.codeSegment = cast(uint16_t)((p == Privilege.USER) ? GdtSelector.USER_CODE : GdtSelector.KERNEL_CODE);
		desc.offset = h;
		return desc;
	}
	
	bool present() {return (typeDPL & Type.PRESENT) != 0;}
	
	void present(bool b) {
		if(b) {
			typeDPL |= Type.PRESENT;
		} else {
			typeDPL &= ~Type.PRESENT;
		}
	}
	
	Privilege privilege() {return cast(Privilege)(typeDPL & Type.PRIVILEGE_MASK);}
	
	void privilege(Privilege p) {
		typeDPL = cast(uint8_t)((typeDPL & ~Type.PRIVILEGE_MASK) | p);
	}
	
	Type type() {return cast(Type)(typeDPL & Type.TYPE_MASK);}
	
	void type(Type t) {typeDPL = cast(uint8_t)((typeDPL & ~Type.TYPE_MASK) | t);}
	
	void offset(uint32_t off) {
		offsetLow = cast(uint16_t)(off & 0xffff);
		offsetHigh = cast(uint16_t)(off >> 16);
	}
	
	void offset(InnerInterruptHandler h) {offset = cast(uint32_t) h;}
	
	uint32_t offset() {return offsetLow | (offsetHigh << 16);}
	
	align(1) {
		uint16_t offsetLow;
		uint16_t codeSegment;
		uint8_t reserved;
		uint8_t typeDPL;
 		uint16_t offsetHigh;
	}
}

extern(C) void trampolineInterrupt();

/// set interrupt number and call trampoline function.
extern(C) void interruptHandlerWithErrorCode(uint N)() {
	asm {
		naked;
		push N;
		jmp trampolineInterrupt;
	}
}

/// set interrupt number and call trampoline function.
extern(C) void interruptHandlerNoErrorCode(uint N)() {
	asm {
		naked;
		push 0U;
		push N;
		jmp trampolineInterrupt;
	}
}

/// generic interrupt handler function.
template interruptHandler(uint N) {
	// interrupts with error code 
	static if(N == Interrupt.DOUBLE_FAULT
			|| N == Interrupt.INVALID_TSS
			|| N == Interrupt.SEGMENT_NOT_PRESENT
			|| N == Interrupt.STACK_SEGMENT_FAULT
			|| N == Interrupt.GENERAL_PROTECTION
			|| N == Interrupt.PAGE_FAULT
			|| N == Interrupt.ALIGNMENT_CHECK) {
		alias interruptHandlerWithErrorCode!(N) interruptHandler;
	} else {
		alias interruptHandlerNoErrorCode!(N) interruptHandler;
	}
}

void initIDT(size_t N)() {
	static if(N < INTERRUPT_DESCRIPTOR_TABLE_LENGTH) {
		IDT[N] = InterruptDescriptor(
			InterruptDescriptor.Type.INTERRUPT_GATE,
			InterruptDescriptor.Privilege.KERNEL,
			&interruptHandler!(N));
		initIDT!(N + 1)();
	}
}

void initInterruptDescriptorTable(InterruptHandler defHandler) {
	initIDT!(0)();
	INTERRUPT_HANDLER_TABLE[] = defHandler;
}

align(8) const Gdtr IDT_LIMIT = {
	cast(uint16_t)(IDT.sizeof - 1)
	cast(uint32_t) IDT.ptr,
};

/// call D function handler.
extern(C) void dispatchInterrupt(InterruptContext ctx) {
	if(ctx.n < INTERRUPT_HANDLER_TABLE.length && INTERRUPT_HANDLER_TABLE[ctx.n] !is null) {
		INTERRUPT_HANDLER_TABLE[ctx.n](ctx);		
		asm {nop;} // disable function return optimize.
	}
}

/// IDT variable.
extern(C) align(8) InterruptDescriptor[INTERRUPT_DESCRIPTOR_TABLE_LENGTH] IDT;

/// D function handlers.
InterruptHandler[INTERRUPT_DESCRIPTOR_TABLE_LENGTH] INTERRUPT_HANDLER_TABLE;

/// IRQ PIC object
InterruptController pic_;
