﻿/**
 *	Memory map.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.memorymap;

import std.stdint;

import drt.stream;

extern(C) void* getKernelBegin();
extern(C) void* getKernelEnd();

const void* DYNAMIC_MEMORY_BEGIN = cast(void*) 0x0010_0000;

void initializeMemoryMap(MemoryMapEntry[] map) {
	memoryMap_ = map;
	
	foreach(MemoryMapEntry e; memoryMap_) {
		if(e.type == MemoryMapEntry.Type.MEMORY
				&& e.attr == MemoryMapEntry.Attribute.ENABLED
				&& e.end > memoryEnd_) {
			memoryEnd_ = e.end;
		}
	}
}

uint64_t getMemoryEnd() {return memoryEnd_;}

MemoryMapEntry[] getMemoryMap() {return memoryMap_;}

/**
 *	memory map structure.
 *
 *	from ACPI int 15h E820h
 */
struct MemoryMapEntry {
	
	/// memory range types.
	enum Type : uint32_t {
		MEMORY = 1,
		RESERVED,
		ACPI,
		NVS,
		UNUSUABLE,
	}

	/// memory range attributes.
	enum Attribute : uint32_t {
		ENABLED,
		NON_VOLATILE,
	}
	
	uint64_t base;		/// base address
	uint64_t length;	/// memory length.
	uint64_t end() {return base + length;}
	Type type;			/// memory range type
	Attribute attr;		/// extended attribute
	
	/// print out.
	void print(OutputPrintStream dest)
	in {
		assert(dest !is null);
	} body {
		with(dest) {
			write(base);
			write("-");
			write(end);
			write(" ");
			switch(type) {
			case Type.MEMORY:		write("MEMORY"); break;
			case Type.RESERVED:		write("RESERVED"); break;
			case Type.ACPI:			write("ACPI"); break;
			case Type.NVS:			write("NVS"); break;
			case Type.UNUSUABLE:	write("UNUSUABLE"); break;
			default:				write("UNDEFINED"); break;
			}
			write(" ");
			switch(attr) {
			case Attribute.ENABLED:			write("ENABLED"); break;
			case Attribute.NON_VOLATILE:	write("NON_VOLATILE"); break;
			default:						write("UNDEFINED"); break;
			}
			write("\r\n");
		}
	}
}

private:

MemoryMapEntry[] memoryMap_;
uint64_t memoryEnd_;
