﻿/**
 *	programable interrupt controller interface.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.pic8259a;

import std.stdint;

import os.i386.cache;
import os.i386.interrupt;
import os.i386.io;
import os.i386.sync;

class Pic8259A : KernelObject, InterruptController {
	
	this() {
		// begin initialize
		outp(MASTER_ICW1, ICW1_BIT | ICW4_NEEDED | CASCADE_MODE);
		
		// interrupt handler vector
		outp(MASTER_ICW2, cast(ubyte) Interrupt.IRQ_BEGIN);
		
		// cascade port
		outp(MASTER_ICW3, cast(ubyte)(1U << (Interrupt.IRQ_CASCADE - Interrupt.IRQ_BEGIN)));
		
		// x86 mode
		outp(MASTER_ICW4, X86_MODE | NORMAL_EOI | NO_BUFFERED);
		
		// slave initialize
		outp(SLAVE_ICW1, ICW1_BIT | ICW4_NEEDED | CASCADE_MODE);
		outp(SLAVE_ICW2, cast(ubyte)(Interrupt.IRQ_BEGIN + PIC2_BEGIN));
		outp(SLAVE_ICW3, cast(ubyte)(Interrupt.IRQ_CASCADE - Interrupt.IRQ_BEGIN));
		outp(SLAVE_ICW4, X86_MODE | NORMAL_EOI | NO_BUFFERED);
		
		// wait complete initialize
		wait(1000);
	}
	
	void disable() {
		oldMask1_ = inp(MASTER_OCW1);
		oldMask2_ = inp(SLAVE_OCW1);
		outp(MASTER_OCW1, 0xff);
		outp(SLAVE_OCW1, 0xff);
	}
	
	void enable() {
		outp(MASTER_OCW1, oldMask1_);
		outp(SLAVE_OCW1, oldMask2_);
	}
	
	void ackInterrupt(uint n) {
		auto irq = n - Interrupt.IRQ_BEGIN;
		if(PIC1_BEGIN <= irq && irq < PIC1_END) {
			inp(MASTER_ICW1); // dummy?
			outp(MASTER_ICW1, cast(ubyte)(ACK_INTERRUPT + irq));
		} else if(PIC2_BEGIN <= irq && irq < PIC2_END) {
			inp(SLAVE_ICW1); // dummy?
			outp(SLAVE_ICW1, cast(ubyte)(ACK_INTERRUPT + (irq & PIC_MASK)));
			outp(MASTER_ICW1, cast(ubyte)(ACK_INTERRUPT + (Interrupt.IRQ_CASCADE - Interrupt.IRQ_BEGIN)));
		}
	}
	
	void maskInterrupt(uint n) {
		auto irq = n - Interrupt.IRQ_BEGIN;
		if(PIC1_BEGIN <= irq && irq < PIC1_END) {
			outp(MASTER_OCW1, cast(ubyte)(1U << irq));
		} else if(PIC2_BEGIN <= irq && irq < PIC2_END) {
			outp(SLAVE_OCW1, cast(ubyte)(1U << (irq & PIC_MASK)));
		}
	}
	
private:
	
	/// io ports.
	enum : uint16_t {
		MASTER_ICW1 = 0x20,
		MASTER_ICW2 = 0x21,
		MASTER_ICW3 = 0x21,
		MASTER_ICW4 = 0x21,
		MASTER_OCW1 = 0x21,
		
		SLAVE_ICW1 = 0xA0,
		SLAVE_ICW2 = 0xA1,
		SLAVE_ICW3 = 0xA1,
		SLAVE_ICW4 = 0xA1,
		SLAVE_OCW1 = 0xA1,
	}
	
	/// IRQ nums.
	enum : uint8_t {
		PIC1_BEGIN = 0,
		PIC1_END = 8,
		PIC2_BEGIN = 8,
		PIC2_END = 16,
		
		PIC_MASK = 0b111,
	}
	
	/// instructions
	enum {
		ACK_INTERRUPT = 0b01100000,
	}
	
	/// initialize mask bits
	enum {
		// ICW1
		ICW4_NEEDED = 0b0000_0001,
		SINGLE_MODE = 0b0000_0010,
		CALL_ADDRESS_INTERVAL4 = 0b0000_0010,
		LEVEL_TRIGGER = 0b0000_1000,
		ICW1_BIT = 0b0001_0000,
		
		// ICW4
		X86_MODE = 0b0000_0001,
		AUTO_EOI = 0b0000_0010,
		BUFFERED = 0b0000_1000,
		BUFFERED_MASTER = 0b0000_1100,
		SPECIAL_FULLY_NESTED = 0b0001_0000,
		
		// default
		CASCADE_MODE = 0,
		CALL_ADDRESS_INTERVAL8 = 0,
		EDGE_TRIGGER = 0,
		NORMAL_EOI = 0,
		NO_BUFFERED = 0,
		BUFFERED_SLAVE = 0,
		NOT_SPECIAL_FULLY_NESTED = 0,
	}
	
	uint8_t oldMask1_;
	uint8_t oldMask2_;
}
