﻿/**
 *	Segment descriptor.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.segment;

import std.stdint;

struct SegmentDescriptor {
	
	enum Type : uint8_t {
		R = 0b0000,		/// readable
		RW = 0b0010,	/// read/writeable
		RD = 0b0100,	/// readable down
		RWD = 0b0110,	/// read/writeable down
		
		E = 0b1000,		/// executable
		ER = 0b1010,	/// execute/readable
		EC = 0b1100,	/// executable confirming
		ERC = 0b1110,	/// execute/readable confirming
		
		/// accessed flag
		ACCESS = 0b0001,
		
		/// present flag
		PRESENT = 0b1000_0000,
		
		/// privilege level bits mask
		PRIVILEGE_MASK = 0b0110_0000,
	}
	
	/// descriptor privilege level (DPL)
	enum Privilege {
		KERNEL,
		SERVICE1,
		SERVICE2,
		USER,
	}
	
	bool present() {return (type & Type.PRESENT) != 0;}
	
	void present(bool b) {
		if(b) {
			type |= Type.PRESENT;
		} else {
			type &= ~Type.PRESENT;
		}
	}
	
	void access(bool b) {
		if(b) {
			type |= Type.ACCESS;
		} else {
			type &= ~Type.ACCESS;
		}
	}
	
	bool access() {return (type & Type.ACCESS) != 0;}
	bool checkAccessAndClear() {
		bool result = access();
		type &= ~Type.ACCESS;
		return result;
	}
	
	void limit(uint32_t val) {
		limitLow = cast(uint16_t) val;
		limitHigh = cast(uint8_t)((limitHigh & 0xf0) | ((val >> 16) & 0x0f));
	}
	
	uint32_t limit() {return ((limitHigh & 0x0f) << 16) | limitLow;}
	
	uint32_t base() {return (base3 << 24) | (base2 << 16) | base01;}
	
	void base(uint32_t val) {
		base01 = cast(uint16_t) val;
		base2 = cast(uint8_t)((val >> 16) & 0x00ff);
		base3 = cast(uint8_t)(val >> 24);
	}
	
	Privilege privilege() {
		return cast(Privilege)((type & Type.PRIVILEGE_MASK) >> 5);
	}
	
	void privilege(Privilege p) {
		type = cast(uint8_t)((type & ~Type.PRIVILEGE_MASK) | (p << 5));
	}
	
	align(1) {
		uint16_t limitLow;
		uint16_t base01;
		uint8_t base2;
		uint8_t type;
		uint8_t limitHigh;
		uint8_t base3;
	}
}

void loadGlobalDescriptorTable(Gdtr* gdtr) {
	asm {
		lgdt gdtr;
		jmp reset_pipeline;
	reset_pipeline:;
	}
}

void loadLocalDescriptorTable(uint16_t i) {
	asm {
		lldt i;
		jmp reset_pipeline;
	reset_pipeline:;
	}
}

/// Global descriptor table register
struct Gdtr {
	align(1) {
		uint16_t limit;
		uint32_t base;
	}
}

/// Global descriptor table indexes
enum GdtIndex {
	NULL,			/// empty segment descriptor
	KERNEL_CODE,	/// kernel mode code segment index
	KERNEL_DATA,	/// kernel mode data segment index
	USER_CODE,		/// user mode code segment index
	USER_DATA,		/// user mode data segment index
	TSS,			/// task state segment index
}

/// Global descriptor table selectors
enum GdtSelector {
	NULL = GdtIndex.NULL * 8,				/// empty segment descriptor
	KERNEL_CODE = GdtIndex.KERNEL_CODE * 8,	/// kernel mode code segment selector
	KERNEL_DATA = GdtIndex.KERNEL_DATA * 8,	/// kernel mode data segment selector
	USER_CODE = GdtIndex.USER_CODE * 8,		/// user mode code segment selector
	USER_DATA = GdtIndex.USER_DATA * 8,		/// user mode data segment selector
	TSS = GdtIndex.TSS * 8,					/// task state segment selector
}

// global valiables.
extern(C):

align(8) const Gdtr GDT_LIMIT = {
	cast(uint16_t)(GDT.sizeof - 1)
	cast(uint32_t) GDT.ptr,
};

align(32) SegmentDescriptor[GdtIndex.max + 1] GDT = [
	{0, 0, 0, 0, 0, 0},
	{0xffff, 0, 0, 0x98, 0xdf, 0x00},	// kernel code
	{0xffff, 0, 0, 0x92, 0xdf, 0x00},	// kernel data
	{0xffff, 0, 0, 0xf8, 0xdf, 0x00},	// user code
	{0xffff, 0, 0, 0xf2, 0xdf, 0x00},	// user data
	{0, 0, 0, 0b1000_1001, 0, 0},		// TSS
];
