﻿/**
 *	serial controler code.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.serial;

import os.i386.io;

import drt.stream;

/// COM ports.
enum Port {
	COM1 = 0x300,
	COM2 = 0x200,
}

/// baudrate values.
enum Baudrate : ushort {
	BPS_115_2K = 0x0001,
	BPS_57_6K = 0x0002,
	BPS_38_4K = 0x0003,
	BPS_19_2K = 0x0006,
	BPS_9600 = 0x000c,
	BPS_4800 = 0x0018,
	BPS_2400 = 0x0030,
	BPS_1200 = 0x0060,
	BPS_600 = 0x00c0,
	BPS_300 = 0x0180,
}

/// interrupt enable masks.
enum InterruptEnable : ubyte {
	RX_DATA	= 0b0001,
	TX_DATA	= 0b0010,
	RX_LINE	= 0b0100,
	MODEM	= 0b1000,
	NONE	= 0b0000,
	ALL		= 0b1111,
}

/// line status.
enum LineStatus : ubyte {
	RX_ERROR			= 0b1000_0000,
	COMPLETE_TRANSMIT	= 0b0100_0000,
	EMPTY_REGISTER		= 0b0010_0000,
	RECEIVE_BREAK		= 0b0001_0000,
	FRAMING_ERROR		= 0b0000_1000,
	PARITY_ERROR		= 0b0000_0100,
	RECEIVE_OVERFLOW	= 0b0000_0010,
	RECEIVE_DATA		= 0b0000_0001,
}

/// modem status.
enum ModemStatus : ubyte {
	DCD_ASSERT	= 0b1000_0000,
	RI_ASSERT	= 0b0100_0000,
	DSR_ASSERT	= 0b0010_0000,
	CTS_ASSERT	= 0b0001_0000,
	CHANGE_DCD	= 0b0000_1000,
	RI_NEGATE	= 0b0000_0100,
	CHANGE_DSR	= 0b0000_0010,
	CHANGE_CTS	= 0b0000_0001,
}

/// line control.
private enum LineControl {
	ACCESS_BAUDRATE = 0b1000_0000,
	BREAK			= 0b0100_0000,
	FIXED_PARITY	= 0b0010_0000,
	EVEN_PARITY		= 0b0001_0000,
	ENABLE_PARITY	= 0b0000_1000,
	STOP_BITS_2		= 0b0000_0100,
	DATA_BITS_8		= 0b0000_0011,
}

/// modem control.
private enum ModemControl {
	LOOP_BACK			= 0b0001_0000,
	INTERRUPT_ENABLE	= 0b0000_1000,
	LOOP_BACK2			= 0b0000_0100,
	ASSERT_RTS			= 0b0000_0010,
	ASSERT_DTS			= 0b0000_0001,
}

/// serial functions.
template Serial(Port P) {
	
	/// control registers.
	private enum {
		RECEIVE_DATA = P + 0xf8,
		TRANSMIT_DATA = P + 0xf8,
		BAUDRATE_LOW = P + 0xf8,
		BAUDRATE_HIGH = P + 0xf9,
		INTERRUPT_ENABLE = P + 0xf9,
		INTERRUPT_ID = P + 0xfa,
		FIFO_CONTROL = P + 0xfa,
		LINE_CONTROL = P + 0xfb,
		MODEM_CONTROL = P + 0xfc,
		LINE_STATUS = P + 0xfd,
		MODEM_STATUS = P + 0xfe,
		SCRATCH = P + 0xff,
	}
	
	/// setup baudrate and interrupt.
	void setup(Baudrate bps, ubyte line = LineControl.DATA_BITS_8, InterruptEnable itr = InterruptEnable.NONE) {
		outp(LINE_CONTROL, LineControl.ACCESS_BAUDRATE);
		if(bps & 0xff00 != 0) {
			outp(BAUDRATE_HIGH, cast(ubyte)(bps >> 8u));
		}
		outp(BAUDRATE_LOW, cast(ubyte)(bps & 0x00ff));
		outp(LINE_CONTROL, line);
		outp(MODEM_CONTROL, ModemControl.INTERRUPT_ENABLE | ModemControl.ASSERT_RTS | ModemControl.ASSERT_DTS);
		outp(INTERRUPT_ENABLE, itr);
	}
	
	/// put data.
	void put(ubyte val) {
		outp(TRANSMIT_DATA, val);
		while(!isCompleteTransmit()) {}
	}
	
	/// get data.
	ubyte get() {return inp(RECEIVE_DATA);}
	
	/// line status.
	ubyte lineStatus() {return inp(LINE_STATUS);}
	
	/// ditto
	bool lineStatus(LineStatus s) {return (lineStatus() & s) != 0;}
	
	/// modem status.
	ubyte modemStatus() {return inp(MODEM_STATUS);}
	
	/// ditto
	bool modemStatus(ModemStatus s) {return (modemStatus() & s) != 0;}
	
	/// interrupt id.
	ubyte interruptId() {return cast(ubyte)(inp(INTERRUPT_ID) >> 1U);}
	
	/// transmit completed?
	bool isCompleteTransmit() {return (lineStatus() & LineStatus.COMPLETE_TRANSMIT) != 0;}
}

class ComStream(Port P) : OutputPrintStream {	
	override void writeImpl(char c) {Serial!(P).put(c);}
}
