﻿/**
 *  文字パーサのモジュール。
 *
 *  Version:
 *      $Revision$
 *  Date:
 *      $Date$
 *  License:
 *      MIT/X Consortium License
 *  History:
 *      $Log$
 */

module outland.parser.charparser;

import std.ctype;
import std.string;
import std.utf;
import std.conv;

import outland.parser.parser;

/// 文字パーサ。
struct UtfCharParser {
    
    /// 戻り値の型。
    struct ResultType(I) {
        I begin;        /// 開始位置。
        size_t length;  /// マッチした長さ。
        bool match;     /// マッチしたかどうか。
        dchar ch;       /// 取得した文字。
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) r;
        r.begin = i.clone;
        if(!i.hasMore) {
            return r;
        }
        
        // 最初の1要素を取得。
        auto c = i.next;
        
        // 文字型に応じてバッファを用意。
        static if(is(typeof(c) == char)) {
            char buf[4];
        } else static if(is(typeof(c) == wchar)) {
            wchar buf[2];
        } else {
            dchar buf[1];
        }
        
        buf[0] = c;
        
        // 最初の要素から文字全体の要素数を得る。
        size_t len = stride(cast(typeof(c)[]) [c, '\0'], 0u);
        
        // UTFの開始要素でないか、バッファ長より長さが大きければ非マッチ。
        if(len == 0xFF || len > buf.length) {
            i = r.begin.clone;
            return r;
        }
        
        // 文字終端まで反復子を進める。
        for(size_t n = 1; n < len; ++n) {
            if(!i.hasMore) {
                i = r.begin.clone;
                return r;
            }
            buf[n] = i.next;
        }
        
        // 文字取得成功。
        try {
            size_t idx = 0;
            r.ch = decode(buf, idx);
        } catch(Exception e) {
            i = r.begin.clone;
            return r;
        }
        
        r.match = true;
        r.length = len;
        
        return r;
    }
}

/// UTF文字パーサの生成。
UtfCharParser utfChar() {
    UtfCharParser p;
    return p;
}

unittest {
    auto r = utfChar.parse(iterator("a"c));
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'a');
    
    r = utfChar.parse(iterator("あ"c));
    assert(r.match);
    assert(r.length == 3);
    assert(r.ch == 'あ');
    
    auto i = iterator("あいう3え"c);
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 3);
    assert(r.ch == 'あ');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 3);
    assert(r.ch == 'い');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 3);
    assert(r.ch == 'う');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == '3');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 3);
    assert(r.ch == 'え');
    
    r = utfChar.parse(iterator(""c));
    assert(!r.match);
    assert(r.length == 0);
    
    r = utfChar.parse(iterator("\xC2"c));
    assert(!r.match);
    assert(r.length == 0);
    
}

unittest {
    auto r = utfChar.parse(iterator("a"w));
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'a');
    
    r = utfChar.parse(iterator("あ"w));
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == cast(wchar) 'あ');
    
    auto i = iterator("あいう3え"w);
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'あ');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'い');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'う');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == '3');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'え');
    
    r = utfChar.parse(iterator(""w));
    assert(!r.match);
    assert(r.length == 0);
}

unittest {    
    auto r = utfChar.parse(iterator("a"d));
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'a');
    
    r = utfChar.parse(iterator("あ"d));
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'あ');
    
    auto i = iterator("あいう3え"d);
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'あ');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'い');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'う');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == '3');
    
    r = utfChar.parse(i);
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'え');
    
    r = utfChar.parse(iterator(""d));
    assert(!r.match);
    assert(r.length == 0);
}

/** 文字タイプパーサ。
 *
 *  ASCII限定。
 *
 *  Params:
 *      F   = 文字タイプ判別関数。
 */
struct CTypeParser(alias F) {
    /// 戻り値の型。
    struct ResultType(I) {
        I begin;        /// 開始位置。
        size_t length;  /// マッチした長さ。
        bool match;     /// マッチしたかどうか。
        dchar ch;       /// 取得した文字。
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) r;
        r.begin = i.clone;
        if(!i.hasMore) {
            return r;
        }
        
        auto c = i.next;
        if(!F(c)) {
            i = r.begin.clone;
        } else {
            r.length = 1;
            r.match = true;
            r.ch = c;
        }
        
        return r;
    }
}

/// 英数字パーサ。
CTypeParser!(isalnum) alnumChar() {
    CTypeParser!(isalnum) p;
    return p;
}

/// 英字パーサ。
CTypeParser!(isalpha) alphaChar() {
    CTypeParser!(isalpha) p;
    return p;
}

/// 制御文字パーサ。
CTypeParser!(iscntrl) cntrlChar() {
    CTypeParser!(iscntrl) p;
    return p;
}

/// 数字パーサ。
CTypeParser!(isdigit) digitChar() {
    CTypeParser!(isdigit) p;
    return p;
}

/// 小文字パーサ。
CTypeParser!(islower) lowerChar() {
    CTypeParser!(islower) p;
    return p;
}

/// 区切り文字パーサ。
CTypeParser!(ispunct) punctChar() {
    CTypeParser!(ispunct) p;
    return p;
}

/// 空白文字パーサ。
CTypeParser!(isspace) spaceChar() {
    CTypeParser!(isspace) p;
    return p;
}

/// 大文字パーサ。
CTypeParser!(isupper) upperChar() {
    CTypeParser!(isupper) p;
    return p;
}

/// 16進数字パーサ。
CTypeParser!(isxdigit) xdigitChar() {
    CTypeParser!(isxdigit) p;
    return p;
}

/// 印字可能文字パーサ。
CTypeParser!(isgraph) graphChar() {
    CTypeParser!(isgraph) p;
    return p;
}

/// 印字可能文字パーサ。
CTypeParser!(isprint) printChar() {
    CTypeParser!(isprint) p;
    return p;
}

/// ASCII文字パーサ。
CTypeParser!(isascii) asciiChar() {
    CTypeParser!(isascii) p;
    return p;
}

unittest {
    auto r = asciiChar.parse(iterator("a"));
    assert(r.match);
    assert(r.length == 1);
    assert(r.ch == 'a');

    r = asciiChar.parse(iterator("あ"));
    assert(!r.match);
    assert(r.length == 0);
}
    
unittest {
    auto r = digitChar.parse(iterator("a"));
    assert(!r.match);
    assert(r.length == 0);
    
    r = digitChar.parse(iterator("1"));
    assert(r.match);
    assert(r.length == 1);
}
