﻿/**
 *  パーサのモジュール。
 *
 *  Version:
 *      $Revision$
 *  Date:
 *      $Date$
 *  License:
 *      MIT/X Consortium License
 *  History:
 *      $Log$
 */

module outland.parser.parser;

import std.stdio;
import std.string;

/// 反復子のインターフェイス。各メンバが実際に存在すれば必ずしも実装する必要はない。
/// 構造体等を反復子とすることも可能。
interface Iterator(T) {
    alias T Element;    /// 要素の型。
    Iterator clone();   /// 反復子の複製。
    bool hasMore(); /// まだ文字があるか。
    T next();           /// 次の文字を取り出す。
}

/** 配列を巡回する反復子。
 *  Params:
 *      T   = 要素の型。
 */
struct ArrayIterator(T) {
    /// 要素の型。
    alias T Element;
    
    /// コンストラクタ。
    static ArrayIterator opCall(T[] src) {
        ArrayIterator iter;
        iter.array_ = src;
        return iter;
    }
    
    /// 反復子のクローンを新たに生成する。
    ArrayIterator clone() {return *this;}
    
    /// まだ要素があるか返す。
    bool hasMore() {return array_.length != 0;}
    
    /// 次の要素を得る。
    T next()
    in {
        assert(hasMore);
    } body {
        T v = array_[0];
        array_ = array_[1 .. $];
        return v;
    }
    
    /// 配列を返す。
    T[] array() {return array_;}
    
private:
    
    /// 反復する配列。
    T[] array_;
}

/// 配列の反復子を生成する。
ArrayIterator!(T) iterator(T)(T[] array) {return ArrayIterator!(T)(array);}

/** 戻り値の構造体。
 *  Params:
 *      I   = 反復子の型。
 */
struct Result(I) {
    I begin;        /// 開始位置。
    size_t length;  /// マッチした長さ。
    bool match;     /// マッチしたかどうか。
}

/** 戻り値の構造体。
 *  Params:
 *      I   = 反復子の型。
 */
struct Result(T:ArrayIterator!(T)) {
    ArrayIterator!(T) begin;    /// 開始位置。
    size_t length;              /// マッチした長さ。
    bool match;             /// マッチしたかどうか。
    
    /// マッチ部分を返す。
    T[] getMatch() {return begin.array[0 .. this.length];}
}

/// パーサの戻り値の型。
template ParserResultType(alias P, I) {
    alias P.ResultType!(I) ParserResultType;
}

/// ditto
template ParserResultType(P:P[], I) {
    alias Result!(I) ParserResultType;
}

/// パーサの呼び出し。
ParserResultType!(P, I) callParser(P, I)(P p, inout I i) {return p.parse(i);}

/// ditto
Result!(I) callParser(T:T[], I)(T[] lit, inout I i) {
    Result!(I) r;
    r.begin = i.clone;
    r.match = true;
    for(r.length = 0; r.length < lit.length; ++r.length) {
        if(!i.hasMore || (i.next != lit[r.length])) {
            // 不一致。
            i = r.begin.clone;
            r.match = false;
            r.length = 0;
            break;
        }
    }
    return r;
}

/// パーサの代入。
void assignParser(P)(out P dest, P src) {dest = src;}

/// ditto
void assignParser(P:P[])(P[] dest, P[] src) {dest[] = src[];}

/// 意味アクション。
struct Action(P, A) {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias ParserResultType!(P, I) ResultType;
    }
    
    /// パーサと付加するアクションを指定して生成する。
    static Action opCall(P p, A a) {
        Action act;
        assignParser!(P)(act.parser, p);
        act.action = a;
        return act;
    }
    
    /// 解析を行う。
    ResultType!(I) parse(I)(inout I i) {
        auto r = callParser!(P, I)(parser, i);
        if(r.match) action(r);
        return r;
    }
    
private:
    
    /// パーサ。
    P parser;
    
    /// アクション。
    A action;
}

/// 連続パーサ。
struct Sequence(T...) {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias Result!(I) ResultType;
    }
    
    /// パーサを指定して生成する。
    static Sequence opCall(T t) {
        Sequence seq;
        foreach(i, v; t) assignParser!(T[i])(seq.values[i], v);
        return seq;
    }
    
    /// 解析。
    ResultType!(I) parse(I)(inout I iter) {
        Result!(I) result;
        result.begin = iter.clone;
        foreach(i, v; values) {
            auto r = callParser!(T[i], I)(v, iter);
            if(!r.match) {
                iter = result.begin.clone;
                result.length = 0;
                result.match = false;
                break;
            }
            
            result.length += r.length;
            result.match = true;
        }
        if(!result.match) iter = result.begin.clone;
        return result;
    }
    
    /// パーサ。
    private T values;
}

/// 選択パーサ。
struct Choice(T...) {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias Result!(I) ResultType;
    }
    
    /// パーサを指定して生成する。
    static Choice opCall(T t) {
        Choice ch;
        foreach(i, v; t) assignParser!(T[i])(ch.values[i], v);
        return ch;
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I iter) {
        ResultType!(I) result;
        result.begin = iter.clone;
        foreach(i, v; values) {
            auto r = callParser!(T[i], I)(v, iter);
            if(r.match) {
                result.match = true;
                result.length = r.length;
                break;
            }
        }
        if(!result.match) iter = result.begin.clone;
        return result;
    }
    
    /// 格納しているパーサ。
    private T values;
}

/// オプションパーサ。
struct Option(P) {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias ParserResultType!(P, I) ResultType;
    }
    
    /// パーサを指定して生成する。
    static Option opCall(P p) {
        Option opt;
        opt.parser = p;
        return opt;
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        auto result = callParser!(P, I)(parser, i);
        result.match = true;
        return result;
    }
    
    /// パーサ。
    private P parser;
}

/// オプション繰り返しパーサ。
struct OptionRepeat(P) {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias ParserResultType!(P, I) ResultType;
    }
    
    /// パーサを指定して生成する。
    static OptionRepeat opCall(P p) {
        OptionRepeat opt;
        opt.parser = p;
        return opt;
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) result;
        result.match = true;
        result.begin = i.clone;
        for(;;) {
            auto r = callParser!(P, I)(parser, i);
            if(!r.match) {
                break;
            }
            result.length += r.length;
        }
        return result;
    }
    
    /// パーサ。
    private P parser;
}

/// 繰り返しパーサ。
struct Repeat(P) {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias ParserResultType!(P, I) ResultType;
    }
    
    /// パーサを指定して生成する。
    static Repeat opCall(P p) {
        Repeat rep;
        rep.parser = p;
        return rep;
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) result;
        result.match = false;
        result.begin = i.clone;
        for(;;) {
            auto r = callParser!(P, I)(parser, i);
            if(!r.match) {
                break;
            }
            result.match = true;
            result.length += r.length;
        }
        return result;
    }
    
    /// パーサ。
    private P parser;
}

/// ANDパーサ。
struct And(P) {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias ParserResultType!(P, I) ResultType;
    }
    
    /// パーサを指定して生成する。
    static And opCall(P p) {
        And and;
        assignParser!(P)(and.parser, p);
        return and;
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        I begin = i.clone;
        auto r = callParser!(P, I)(parser, i);
        r.length = 0;
        i = begin.clone;
        return r;
    }
    
    /// パーサ。
    private P parser;
}

/// NOTパーサ。
struct Not(P) {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias ParserResultType!(P, I) ResultType;
    }
    
    /// パーサを指定して生成する。
    static Not opCall(P p) {
        Not not;
        assignParser!(P)(not.parser, p);
        return not;
    }

    /// 解析。
    ResultType!(I) parse(I)(inout I i) {
        I begin = i.clone;
        auto r = callParser!(P, I)(parser, i);
        r.length = 0;
        r.match = !r.match;
        i = begin.clone;
        return r;
    }
    
    /// パーサ。
    private P parser;
}

/// どんな文字でも受け入れるパーサ。
struct Any {

    /// 戻り値の型。
    template ResultType(I) {
        alias Result!(I) ResultType;
    }
    
    /// 解析。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) r;
        r.begin = i.clone();
        if(i.hasMore) {
            i.next();
            r.length = 1;
            r.match = true;
        }
        return r;
    }
}

/// 0文字にマッチするパーサ。
struct Empty {

    /// 戻り値の型。
    template ResultType(I) {
        alias Result!(I) ResultType;
    }
    
    /// 解析。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) r;
        r.begin = i.clone();
        r.match = true;
        return r;
    }
}

/// 意味アクションの生成。
Action!(P, A) act(P, A)(P p, A a) {return Action!(P, A)(p, a);}

/// シーケンスパーサの生成。
Sequence!(T) seq(T...)(T t) {return Sequence!(T)(t);}

/// 選択パーサの生成。
Choice!(T) choice(T...)(T t) {return Choice!(T)(t);}

/// オプションパーサの生成。
Option!(P) opt(P)(P p) {return Option!(P)(p);}

/// オプション繰り返しパーサの生成。
OptionRepeat!(P) optr(P)(P p) {return OptionRepeat!(P)(p);}

/// 繰り返しパーサの生成。
Repeat!(P) rep(P)(P p) {return Repeat!(P)(p);}

/// andパーサの生成。
And!(P) and(P)(P p) {return And!(P)(p);}

/// notパーサの生成。
Not!(P) not(P)(P p) {return Not!(P)(p);}

/// anyパーサの生成。
Any any() {return Any.init;}

/// emptyパーサの生成。
Empty empty() {return Empty.init;}

/// 構文規則を保持するためのクラス。
class Rule(I) {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias Result!(I) ResultType;
    }
    
    /// アクションの型。
    alias void delegate(ResultType!(I)) Action;

private:
    
    /// パーサ保持クラス。
    static abstract class Holder {
        /// 解析する。
        ResultType!(I) parse(inout I i);
    }
    
    /// 具象パーサ保持クラス。
    static class ConcreateHolder(P) : Holder {
        /// パーサを指定して生成する。
        this(P p) {assignParser!(P)(parser_, p);}
        
        /// 解析する。
        ResultType!(I) parse(inout I i) {
            Result!(I) result;
            result.begin = i.clone;
            auto r = callParser!(P, I)(parser_, i);
            result.match = r.match;
            result.length = r.length;
            return result;
        }
        
        private P parser_;
    }
    
    /// 保持しているパーサ。
    Holder holder_;
    
    /// 後付のセマンティックアクション。
    Action action_;
    
    /// パーサ名。
    char[] name_;
    
    /// 例外オブジェクト。
    Exception exception_;
    
public:
    
    /// パーサ名を指定して生成する。
    this(char[] name = "") {name_ = name;}
    
    /// パーサの格納。
    void assign(P)(P p) {holder_ = new ConcreateHolder!(P)(p);}
    
    /// 解析。
    ResultType!(I) parse(inout I i) {
        ResultType!(I) r;
        if(holder_ !is null) {
            r = holder_.parse(i);
            if(r.match) {
                if(action_ !is null) {
                    action_(r);
                }
            } else if(exception_ !is null) {
                throw exception_;
            }
        } else {
            r.begin = i.clone;
        }
        return r;
    }
    
    /// 後付のアクション。
    void action(Action a) {action_ = a;}
    
    /// ditto
    Action action() {return action_;}
    
    /// マッチ失敗時の例外オブジェクト。
    void exception(Exception exp) {exception_ = exp;}
    
    /// ditto
    Exception exception() {return exception_;}
    
    /// パーサ名。
    char[] name() {return name_;}
}

unittest {
    alias typeof(iterator("test")) Iter;
    alias .Rule!(Iter) Rule;
    
    assert(seq("te").parse(iterator("test")).match);
    assert(seq("te").parse(iterator("test")).length == 2);
    assert(!seq("et").parse(iterator("test")).match);
    assert(seq("et").parse(iterator("test")).length == 0);
    assert(!seq("testt").parse(iterator("test")).match);
    assert(seq("testt").parse(iterator("test")).length == 0);
    assert(seq("te", "st").parse(iterator("test")).length == 4);
    assert(!seq("st", "te").parse(iterator("test")).match);
    assert(seq("st", "te").parse(iterator("test")).length == 0);
    assert(choice("st", "te", "tt").parse(iterator("test")).match);
    assert(choice("st", "te", "tt").parse(iterator("test")).length == 2);
    assert(!choice("st", "tt").parse(iterator("test")).match);
    assert(choice("st", "tt").parse(iterator("test")).length == 0);
    assert(any().parse(iterator("test")).match);
    assert(any().parse(iterator("test")).length == 1);
    assert(!any().parse(iterator("")).match);
    assert(any().parse(iterator("")).length == 0);
    
    int[] stack;
    
    void matchNumber(Result!(Iter) result) {
        int val;
        Iter iter = result.begin.clone;
        for(size_t i = 0; i < result.length && iter.hasMore; ++i) {
            val *= 10;
            switch(iter.next) {
            case '9': ++val;
            case '8': ++val;
            case '7': ++val;
            case '6': ++val;
            case '5': ++val;
            case '4': ++val;
            case '3': ++val;
            case '2': ++val;
            case '1': ++val;
            default:
            case '0': break;
            }
        }
        stack ~= val;
    }
    
    void matchMul(Result!(Iter) result) {
        stack[$-2] = stack[$-2] * stack[$-1];
        stack.length = stack.length - 1;
    }
    
    void matchDiv(Result!(Iter) result) {
        stack[$-2] = stack[$-2] / stack[$-1];
        stack.length = stack.length - 1;
    }
    
    void matchPlus(Result!(Iter) result) {
        stack[$-2] = stack[$-2] + stack[$-1];
        stack.length = stack.length - 1;
    }
    
    void matchMinus(Result!(Iter) result) {
        stack[$-2] = stack[$-2] - stack[$-1];
        stack.length = stack.length - 1;
    }
    
    auto number = new Rule();
    auto factor = new Rule();
    auto term = new Rule();
    auto exp = new Rule();
    
    number.assign(act(rep(choice("0", "1", "2", "3", "4", "5", "6", "7", "8", "9")), &matchNumber));
    factor.assign(choice(number, seq("(", exp, ")")));
    term.assign(seq(factor, opt(choice(act(seq("*", term), &matchMul), act(seq("/", term), &matchDiv)))));
    exp.assign(seq(term, opt(choice(act(seq("+", exp), &matchPlus), act(seq("-", exp), &matchMinus)))));
    
    assert(exp.parse(iterator("1")).match);
    assert(exp.parse(iterator("1+1")).match);
    assert(exp.parse(iterator("1-1")).match);
    assert(exp.parse(iterator("1*1")).match);
    assert(exp.parse(iterator("1/1")).match);
    assert(exp.parse(iterator("1*1+1")).match);
    assert(exp.parse(iterator("1+1*1")).match);
}
