﻿/**
 *  空白パーサのモジュール。
 *
 *  Version:
 *      $Revision$
 *  Date:
 *      $Date$
 *  License:
 *      MIT/X Consortium License
 *  History:
 *      $Log$
 */

module outland.parser.spaceparser;

import std.ctype;
import std.stdio;
import std.string;

import outland.parser.parser;

/// 空白文字。
const char[] whitespace = std.string.whitespace;

/// 改行文字パーサ。
struct NewLineParser {
    
    private const char CR = '\r';
    private const char LF = '\n';
    
    /// 戻り値の型。
    template ResultType(I) {
        alias Result!(I) ResultType;
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) r;
        r.begin = i.clone;
        if(!i.hasMore) {
            return r;
        }
        
        char c = i.next;
        if(c == LF) {
            r.match = true;
            r.length = 1;
        } else if(c == CR) {
            r.match = true;
            r.length = 1;
            if(i.hasMore) {
                I tmp = i.clone;
                if(i.next == LF) {
                    ++r.length;
                } else {
                    i = tmp.clone;
                }
            }
        } else {
            i = r.begin.clone;
        }
        
        return r;
    }
}

/// 改行文字パーサの生成。
NewLineParser newLine() {
    NewLineParser p;
    return p;
}

unittest {
    auto r = newLine.parse(iterator(""));
    assert(!r.match);
    assert(r.length == 0);
    
    r = newLine.parse(iterator("\r"));
    assert(r.match);
    assert(r.length == 1);
    
    r = newLine.parse(iterator("\n"));
    assert(r.match);
    assert(r.length == 1);
    
    r = newLine.parse(iterator("\r\n"));
    assert(r.match);
    assert(r.length == 2);
    
    r = newLine.parse(iterator("\n\r"));
    assert(r.match);
    assert(r.length == 1);
}

/// 空白パーサ。
struct SpaceParser {
    
    /// 戻り値の型。
    template ResultType(I) {
        alias Result!(I) ResultType;
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) r;
        r.begin = i.clone;
        if(!i.hasMore) {
            return r;
        }
        
        while(i.hasMore) {
            I tmp = i.clone;
            if(!std.ctype.isspace(i.next)) {
                i = tmp.clone;
                break;
            }
            r.match = true;
            ++r.length;
        }
        
        return r;
    }
}

/// 空白パーサの生成。
SpaceParser space() {
    SpaceParser p;
    return p;
}

unittest {
    auto r = space.parse(iterator(""));
    assert(!r.match);
    assert(r.length == 0);
    
    r = space.parse(iterator("\r"));
    assert(r.match);
    assert(r.length == 1);
    
    r = space.parse(iterator("\n"));
    assert(r.match);
    assert(r.length == 1);
    
    r = space.parse(iterator("\r\n"));
    assert(r.match);
    assert(r.length == 2);
    
    r = space.parse(iterator("\n\r"));
    assert(r.match);
    assert(r.length == 2);
    
    r = space.parse(iterator("  \t\t"));
    assert(r.match);
    assert(r.length == 4);
}
