﻿module outland.widget.event;

import outland.widget.shape;
import outland.tl.algorithm;

/// イベントコード。
typedef uint EventCode;

/// イベントID。
alias uint EventId;

/// イベントインターフェイス。
interface IEvent {
	
	/// イベントコード。
	EventCode code();
	
	/// イベントID。
	EventId id();
	
	/// イベントソース。
	Object source();
}

/// イベント基本クラス。
abstract class Event : IEvent {

	/** 生成する。
	 *	Params:
	 *		id		= イベントID。
	 *		code	= イベントコード。
	 */
	this(EventId id, EventCode code) {
		id_ = id;
		code_ = code;
	}
	
	/// イベントコード。
	EventCode code() {return code_;}
	
	/// イベントID。
	EventId id() {return id_;}
	
	/// イベントソース。
	Object source() {return source_;}
	
	/// ditto
	void source(Object src) {source_ = src;}
	
private:
	
	/// イベントID。
	final EventId id_;
	
	/// イベントコード。
	final EventCode code_;
	
	/// イベントソース。
	Object source_;
}

/// イベントターゲット。
interface IEventTarget {
	
	/**	イベントの受信。
	 *	Params:
	 *		e	= イベント。
	 *	Returns:
	 *		イベントを処理したかどうか。
	 */
	bool handle(IEvent e);
}

/// イベントマップ。
class EventMap : IEventTarget {
	
	// イベント受信。
	bool handle(IEvent e)
	in {
		assert(e !is null);
	} body {
		foreach(entry; entries_) {
			if(entry.code == e.code && entry.begin <= e.id && e.id <= entry.end) {
				if(entry.target.handle(e)) return true;
			}
		}
		return false;
	}
	
	/** ターゲットの追加。
	 *	Params:
	 *		code	= イベントコード。
	 *		min		= ID始点。
	 *		max		= ID終点。
	 *		t		= イベントターゲット。
	 *	Returns:
	 *		追加されたイベントターゲットオブジェクト。
	 */
	void addTarget(EventCode code, EventId min, EventId max, IEventTarget t)
	in {
		assert(t !is null);
	} body {
		Entry e;
		e.code = code;
		e.begin = min;
		e.end = max;
		e.target = t;
		entries_ ~= e;
	}
	
	/**	ターゲットの破棄。
	 *	Params:
	 *		t	= イベントターゲット。
	 */
	void removeTarget(IEventTarget t) {
		outland.tl.algorithm.removeIf(entries_, (Entry e) {return e.target is t;});
	}
	
	/// サイズを返す。
	size_t length() {return entries_.length;}
	
	/// 全項目の破棄。
	void clear() {entries_.length = 0;}
	
private:
	
	/// エントリの型。
	struct Entry {
		EventCode code;			/// イベントコード。
		EventId begin;			/// ID始点。
		EventId end;			/// ID終点。
		IEventTarget target;	/// ターゲット。
	}
	
	/// イベントマップ。
	Entry[] entries_;
}

/// デリゲートからイベントターゲットを生成する。
IEventTarget delegateToEventTarget(R, E)(R delegate(E) dg) {
	// IEventTarget実装クラス。
	static class EventTarget : IEventTarget {
		this(R delegate(E) dg) {dg_ = dg;}
		bool handle(IEvent e) {
			if(auto ev = cast(E) e) {
				// デリゲートの戻り型がvoidならば戻り値は常にtrue。そうでなければ実行結果を返す。
				static if(is(R : void)) {
					dg_(ev);
					return true;
				} else {
					return dg_(ev);
				}
			}
			// 型が違う。
			return false;
		}
		
		// 呼び出されるデリゲート。
		private R delegate(E) dg_;
	}
	
	// IEventTarget実装クラスを生成して返す。
	return new EventTarget(dg);
}

/// イベントマップハンドラ。
template MEventHandlerMap() {
	
	/// イベント受信。
	bool handle(IEvent e) {
		if(map_ !is null) return map_.handle(e);
		return false;
	}
	
	/** ターゲットの追加。
	 *	Params:
	 *		code	= イベントコード。
	 *		min		= ID始点。
	 *		max		= ID終点。
	 *		t		= イベントターゲット。
	 */
	void addTargetRange(EventCode code, EventId min, EventId max, IEventTarget t)
	in {
		assert(t !is null);
	} body {
		if(map_ is null) map_ = new EventMap;
		map_.addTarget(code, min, max, t);
	}
	
	/** ターゲットの追加。
	 *	Params:
	 *		code	= イベントコード。
	 *		id		= ID。
	 *		t		= イベントターゲット。
	 */
	void addTarget(EventCode code, EventId id, IEventTarget t) {addTargetRange(code, id, id, t);}
	
	/** ターゲットの追加。
	 *	Params:
	 *		code	= イベントコード。
	 *		t		= イベントターゲット。
	 */
	void addTargetAny(EventCode code, IEventTarget t) {addTargetRange(code, EventId.min, EventId.max, t);}
	
	/** ハンドラの追加。
	 *	Params:
	 *		code	= イベントコード。
	 *		min		= ID始点。
	 *		max		= ID終点。
	 *		dg		= デリゲート。
	 */
	IEventTarget addHandlerRange(R, E)(EventCode code, EventId min, EventId max, R delegate(E) dg)
	in {
		assert(dg !is null);
	} body {
		auto t = delegateToEventTarget(dg);
		addTargetRange(code, min, max, t);
		return t;
	}
	
	/** ハンドラの追加。
	 *	Params:
	 *		code	= イベントコード。
	 *		id		= ID。
	 *		dg		= デリゲート。
	 */
	IEventTarget addHandler(R, E)(EventCode code, EventId id, R delegate(E) dg) {
		return addHandlerRange!(R, E)(code, id, id, dg);
	}
	
	/** ハンドラの追加。
	 *	Params:
	 *		code	= イベントコード。
	 *		dg		= デリゲート。
	 */
	IEventTarget addHandlerAny(R, E)(EventCode code, R delegate(E) dg) {
		return addHandlerRange!(R, E)(code, EventId.min, EventId.max, dg);
	}
	
	/// イベントマップ。
	private EventMap map_;
}

/// イベントコードの定義。
enum : EventCode {
	INVALID_EVENT_CODE,	/// 無効なイベントコード。
	
	EVENT_CODE_MOUSE_CAPTURE,
	EVENT_CODE_MOUSE_RELEASE,
	EVENT_CODE_MOUSE_MOVE,
	EVENT_CODE_MOUSE_DOWN,
	EVENT_CODE_MOUSE_UP,
	EVENT_CODE_MOUSE_DOUBLE_CLICK,
	EVENT_CODE_MOUSE_WHEEL,
	EVENT_CODE_KEY_DOWN,
	EVENT_CODE_KEY_UP,
	EVENT_CODE_KEY_FOCUS,
	EVENT_CODE_KEY_UNFOCUS,
	
	USER_CODE_BEGIN = cast(EventCode) 0x10000,	/// ユーザ定義のコード始点。これ未満は予約。
	
	/// マウスイベント開始値。
	EVENT_CODE_MOUSE_BEGIN = EVENT_CODE_MOUSE_CAPTURE,
	
	/// マウスイベント終了値。
	EVENT_CODE_MOUSE_END = EVENT_CODE_MOUSE_WHEEL + 1,
	
	/// キーボードイベント開始値。
	EVENT_CODE_KEY_BEGIN = EVENT_CODE_KEY_DOWN,
	
	/// キーボードイベント終了値。
	EVENT_CODE_KEY_END = EVENT_CODE_KEY_UNFOCUS + 1,
}

/// イベントIDの定義。
enum : EventId {
	EVENT_ID_INVALID,	/// 無効なID。
	
	EVENT_ID_SYSTEM,	/// システムからのイベント。
	EVENT_ID_USER,		/// ユーザ操作によるイベント。
	
	USER_ID_BEGIN = cast(EventId) 0x10000,	/// ユーザ定義のID始点。これ未満は予約。
}
