﻿module outland.widget.graphics;

import outland.widget.shape;

/// 色構造体。
struct Color {

	/// 無効な色。
	const Color INVALID = {value:0xFFFFFFFF};
	
	/**	色の生成。
	 *	Params:
	 *		r	= 赤。
	 *		g	= 緑。
	 *		b	= 青。
	 *	Returns:
	 *		色の構造体。
	 */
	static Color opCall(ubyte r, ubyte g, ubyte b) {
		Color c;
		c.red = r;
		c.green = g;
		c.blue = b;
		return c;
	}
	
	/// OR。
	void opOrAssign(Color rhs) {value |= rhs.value;}
	
	/// ditto
	Color opOr(Color rhs) {
		Color c = *this;
		c |= rhs;
		return c;
	}
	
	/// AND
	void opAndAssign(Color rhs) {value &= rhs.value;}
	
	/// ditto
	Color opAnd(Color rhs) {
		Color c = *this;
		c &= rhs;
		return c;
	}
	
	/// XOR
	void opXorAssign(Color rhs) {value ^= rhs.value;}
	
	/// ditto
	Color opXor(Color rhs) {
		Color c = *this;
		c ^= rhs;
		return c;
	}
	
	/// NOT。
	Color opCom() {
		Color c = *this;
		c.value = ~c.value;
		return c;
	}
	
	// 色の値。
	union {
		/// 32ビット。
		uint value;
		
		struct {
			/// 赤。
			ubyte red;
			
			/// 緑。
			ubyte green;
			
			/// 青。
			ubyte blue;
		}
	}
}

/// フォント。
struct Font {
	
	/**	フォント名を指定して生成する。
	 *	Params:
	 *		face	= フォント名。
	 */
	static Font opCall(char[] face) {
		Font f;
		f.face_ = face;
		return f;
	}
	
	/**	フォント名とサイズを指定して生成する。
	 *	Params:
	 *		face	= フォント名。
	 *		size	= サイズ。
	 */
	static Font opCall(char[] face, float size) {
		Font f;
		f.face_ = face;
		f.size_ = size;
		return f;
	}
	
	/**	フォントを指定して生成する。
	 *	Params:
	 *		face	= フォント名。
	 *		size	= サイズ。
	 *		bold	= 強調。
	 *		under	= 下線。
	 *		strike	= 取消線。
	 *		italic	= 斜体。
	 */
	static Font opCall(char[] face, float size, bool bold, bool under, bool strike, bool italic) {
		Font f;
		f.face_ = face;
		f.size_ = size;
		f.bold_ = bold;
		f.under_ = under;
		f.strike_ = strike;
		f.italic_ = italic;
		return f;
	}
	
	/// フォント名。
	char[] face() {return face_;}
	
	/// ditto
	void face(char[] f) {face_ = f;}
	
	/// サイズ。
	float size() {return size_;}
	
	/// ditto
	void size(float s) {size_ = s;}
	
	/// 強調。
	bool bold() {return bold_;}
	
	/// ditto
	void bold(bool b) {bold_ = b;}
	
	/// 下線。
	bool under() {return under_;}
	
	/// ditto
	void under(bool u) {under_ = u;}
	
	/// 取消線。
	bool strike() {return strike_;}
	
	/// ditto
	void strike(bool s) {strike_ = s;}
	
	/// 斜体。
	bool italic() {return italic_;}
	
	/// ditto
	void italic(bool i) {italic_ = i;}
	
private:
	
	/// フォントフェイス名。
	char[] face_;
	
	/// サイズ。
	float size_;
	
	/// 強調。
	bool bold_;
	
	/// 下線。
	bool under_;
	
	/// 取消線。
	bool strike_;
	
	/// 斜体。
	bool italic_;
}

/// イメージインターフェイス。
interface IImage {
	
	/// ピクセルフォーマット。
	enum Format {
		/// 未知の形式。
		UNKNOWN,
		
		/// モノクロ。
		MONOCHLOME,
		
		/// 16色。
		PALETTE16,
		
		/// 256色。
		PALETTE256,
		
		/// 15ビット。R:5, G:5, B5
		RGB15,
		
		/// 16ビット。R:5, G:6, B5
		RGB16,
		
		/// 24ビット。
		RGB24,
		
		/// 32ビット。
		RGB32,
	}
	
	/// イメージサイズ。
	Size size();
	
	/// ピクセルフォーマット。
	Format format();
	
	/// イメージの破棄。
	void close();
}

/// グラフィックスインターフェイス。
interface IGraphics {
	
	/// 転送モード。
	enum BltMode {
		COPY,	/// 通常のコピー。
		NOT,	/// 反転。
		OR,		/// OR合成。
		AND,	/// AND合成。
		XOR,	/// XOR合成。
	}
	
	/// 描画モード。
	enum DrawMode {
		COPY,	/// 通常のコピー。
		NOT,	/// 反転。
		OR,		/// OR合成。
		AND,	/// AND合成。
		XOR,	/// XOR合成。
	}
	
	/// 座標軸。
	enum Axis {
		X,	/// X軸。
		Y,	/// Y軸。
	}
	
	/// 1ピクセル当たりのポイント数。
	const uint POINT_PER_INCH = 72;
	
	/// 文字色。
	Color textColor();
	
	/// ditto
	void textColor(Color c);
	
	/// 背景色。
	Color backColor();
	
	/// ditto
	void backColor(Color c);
	
	/// 線色。
	Color lineColor();
	
	/// ditto
	void lineColor(Color c);
	
	/// 塗りつぶし色。
	void fillColor(Color c);
	
	/// ditto
	Color fillColor();
	
	/// 塗りつぶしイメージ。
	void fillImage(IImage img);
	
	/// ditto
	IImage fillImage();
	
	/// フォント。
	Font font();
	
	/// ditto
	void font(Font f);
	
	/// 点の描画。
	void dot(Point p, Color c);
	
	/// 線の描画。予めlineColorを設定していないと動作は未定義。
	void line(Point p1, Point p2);
	
	/// 矩形の描画。予めfillColorを設定していないと動作は未定義。
	void colorFill(Rect r);
	
	/// イメージ矩形の描画。予めfillImageを設定していないと動作は未定義。
	void imageFill(Rect r);
	
	/// 枠の描画。予めlineColorを設定していないと動作は未定義。
	void frame(Rect r);
	
	/// テキストの描画。予めfontを設定していないと動作は未定義。
	void text(Point p, char[] str);
	
	/// テキストサイズの取得。
	Size getTextSize(char[] str);
	
	/// 使用可能なフォントフェイス名。
	char[][] getFontFaces();
	
	/// ビットブロック転送。
	void transfer(Point dest, IGraphics g, Rect src, BltMode mode);
	
	/// 伸縮を行うビットブロック転送。
	void transfer(Rect dest, IGraphics g, Rect src, BltMode mode);
	
	/// イメージ転送。
	void transfer(Point dest, IImage img, Rect src, BltMode mode);
	
	/// 伸縮を行うメージ転送。
	void transfer(Rect dest, IImage img, Rect src, BltMode mode);
	
	/**	イメージの生成。
	 *	Params:
	 *		size	= イメージのサイズ。
	 *		fmt		= ピクセルフォーマット。
	 *		pixels	= ピクセルデータ。ピクセルフォーマットに合っていなければならない。
	 *		palette	= パレット。RGBフォーマット時は無視。
	 */
	IImage makeImage(Size size, IImage.Format fmt, void[] pixels, Color[] palette);
	
	/// 座標原点。
	void origin(Point org);
	
	/// ditto
	Point origin();
	
	/// クリップ領域。
	void clip(Rect r);
	
	/// ditto
	Rect clip();
	
	/// 描画モード。
	void drawMode(DrawMode mode);
	
	/// ditto
	DrawMode drawMode();
	
	/// ポイント値をピクセルに変換。
	uint pointToPixel(float pt, Axis axis);
	
	/// ピクセルをポイント値に変換。
	float pixelToPoint(uint p, Axis axis);
	
	/// 状態を記憶して処理を行う。
	void duringBackup(void delegate() dg);
	
	/// イメージを描画対象とする。
	void duringDrawImage(IImage img, void delegate() dg);
	
	/// グラフィックスの破棄。
	void close();
}
