﻿module outland.widget.shape;

import std.string;

/// 加算。
template MAddOperator(T, U) {
	T opAdd(U rhs) {
		T tmp = *this;
		tmp += rhs;
		return tmp;
	}
}

/// 減算。
template MSubOperator(T, U) {
	/// ditto
	T opSub(U rhs) {
		T tmp = *this;
		tmp -= rhs;
		return tmp;
	}
}

/// 乗算。
template MMulOperator(T, U) {
	/// ditto
	T opMul(U rhs) {
		T tmp = *this;
		tmp *= rhs;
		return tmp;
	}
}

/// 除算。
template MDivOperator(T, U) {
	/// ditto
	T opDiv(U rhs) {
		T tmp = *this;
		tmp /= rhs;
		return tmp;
	}
}

/// ビット積。
template MAndOperator(T, U) {
	/// ditto
	T opAnd(U rhs) {
		T tmp = *this;
		tmp &= rhs;
		return tmp;
	}
}

/// ビット和。
template MOrOperator(T, U) {
	/// ditto
	T opOr(U rhs) {
		T tmp = *this;
		tmp |= rhs;
		return tmp;
	}
}

/// 座標構造体。
struct Point {
	
	/// 最小の座標。
	const Point min = {int.min, int.min};
	
	/// 空の座標。
	const Point empty = {0, 0};
	
	/// 最大の座標。
	const Point max = {int.max, int.max};
	
	/**	コンストラクタ。
	 *	Params:
	 *		x	= X座標。
	 *		y	= Y座標。
	 */
	static Point opCall(int x, int y) {
		Point p;
		p.x = x;
		p.y = y;
		return p;
	}
	
	/// 加算。
	void opAddAssign(Point rhs) {
		x += rhs.x;
		y += rhs.y;
	}
	
	/// 減算。
	void opSubAssign(Point rhs) {
		x -= rhs.x;
		y -= rhs.y;
	}
	
	/// 乗算。
	void opMulAssign(Point rhs) {
		x *= rhs.x;
		y *= rhs.y;
	}
	
	/// 乗算。
	void opMulAssign(int val) {
		x *= val;
		y *= val;
	}
	
	/// 除算。
	void opDivAssign(Point rhs) {
		x /= rhs.x;
		y /= rhs.y;
	}
	
	/// 除算。
	void opDivAssign(int val) {
		x /= val;
		y /= val;
	}
	
	/// 移動。
	void offset(int x, int y) {
		this.x += x;
		this.y += y;
	}
	
	mixin MAddOperator!(Point, Point);
	mixin MSubOperator!(Point, Point);
	mixin MMulOperator!(Point, Point);
	mixin MDivOperator!(Point, Point);
	mixin MMulOperator!(Point, int);
	mixin MDivOperator!(Point, int);
	
	/// 文字列に変換する。
	char[] toString() {return format("(%d,%d)", x, y);}
	
	/// 比較する。
	int opCmp(Point rhs) {return std.string.memcmp(this, &rhs, Point.sizeof);}
	
	/// X座標。
	int x;
	
	/// Y座標。
	int y;
}

/// サイズ構造体。
struct Size {
	
	/// 最小のサイズ。
	const Size min = {int.min, int.min};
	
	/// 空のサイズ。
	const Size empty = {0, 0};
	
	/// 最大のサイズ。
	const Size max = {int.max, int.max};
	
	/**	コンストラクタ。
	 *	Params:
	 *		w	= 幅。
	 *		h	= 高さ。
	 */
	static Size opCall(int w, int h) {
		Size s;
		s.width = w;
		s.height = h;
		return s;
	}
	
	/// 面積を返す。
	int area() {return this.width * this.height;}
	
	/// 乗算。
	void opMulAssign(int val) {
		width *= val;
		height *= val;
	}
	
	/// 除算。
	void opDivAssign(int val) {
		width /= val;
		height /= val;
	}
	
	mixin MMulOperator!(Size, int);
	mixin MDivOperator!(Size, int);
	
	/// 文字列に変換する。
	char[] toString() {return format("[%d,%d]", width, height);}
	
	/// 比較する。
	int opCmp(Size rhs) {return this.area - rhs.area;}

	/// 幅。
	int width;
	
	/// 高さ。
	int height;
}

/// 矩形構造体。
struct Rect {
	
	/// 最小の矩形。
	const Rect min = {Point.min, Size.min};
	
	/// 空の矩形。
	const Rect empty = {Point.empty, Size.empty};
	
	/// 最大の矩形。
	const Rect max = {Point.empty, Size.max};
	
	/**	コンストラクタ。
	 *	Params:
	 *		p	= 位置。
	 *		s	= サイズ。
	 */
	static Rect opCall(Point p, Size s) {
		Rect r;
		r.position = p;
		r.size = s;
		return r;
	}
	
	/**	コンストラクタ。
	 *	Params:
	 *		x	= X座標。
	 *		y	= Y座標。
	 *		w	= 幅。
	 *		h	= 高さ。
	 */
	static Rect opCall(int x, int y, int w, int h) {return opCall(Point(x, y), Size(w, h));}
	
	/// X座標。
	int x() {return position.x;}
	
	/// ditto
	void x(int x) {position.x = x;}
	
	/// 左。
	alias x left;
	
	/// Y座標。
	int y() {return position.y;}
	
	/// ditto
	void y(int y) {position.y = y;}
	
	/// 上。
	alias y top;
	
	/// 幅。
	int width() {return size.width;}
	
	/// ditto
	void width(int w) {size.width = w;}
	
	/// 右。
	int right() {return x + width;}
	
	/// ditto
	void right(int r) {width = r - left;}
	
	/// 高さ。
	int height() {return size.height;}
	
	/// ditto
	void height(int h) {size.height = h;}
	
	/// 下。
	int bottom() {return top + height;}
	
	/// ditto
	void bottom(int b) {height = top + b;}
	
	/// 左上。
	Point leftTop() {return Point(left, top);}
	
	/// 右上。
	Point rightTop() {return Point(right, top);}
	
	/// 右下。
	Point rightBottom() {return Point(right, bottom);}
	
	/// 左下。
	Point leftBottom() {return Point(left, bottom);}
	
	/// 移動。
	void offset(int x, int y) {position.offset(x, y);}
	
	/// 加算。
	void opAddAssign(Point rhs) {
		x = x + rhs.x;
		y = y + rhs.y;
	}
	
	/// 減算。
	void opSubAssign(Point rhs) {
		x = x - rhs.x;
		y = y - rhs.y;
	}
	
	/// 乗算。
	void opMulAssign(int val) {
		width = width * val;
		height = height * val;
	}
	
	/// 除算。
	void opDivAssign(int val) {
		width = width / val;
		height = height / val;
	}
	
	/// ビット和。
	void opOrAssign(Rect rhs) {
		Point lt = (this.leftTop < rhs.leftTop) ? this.leftTop : rhs.leftTop;
		Point rb = (this.rightBottom < rhs.rightBottom) ? rhs.rightBottom : this.rightBottom;
		this.position = lt;
		this.size = Size(rb.x - lt.x, rb.y - lt.y);
	}
	
	/// ビット積。
	void opAndAssign(Rect rhs) {
		int t = (this.top < rhs.top) ? rhs.top : this.top;
		int b = (this.bottom < rhs.bottom) ? this.bottom : rhs.bottom;
		
		// 重なっているかチェック。
		if(b < t) {
			*this = Rect.empty;
			return;
		}
		
		int l = (this.left < rhs.left) ? rhs.left : this.left;
		int r = (this.right < rhs.right) ? this.right : rhs.right;
		
		// 重なっているかチェック。
		if(r < l) {
			*this = Rect.empty;
			return;
		}
		
		this.position = Point(l, t);
		this.size = Size(r - l, b - t);
	}
	
	/// 衝突判定。
	bool opIn_r(Point pos) {
		return this.left <= pos.x && pos.x < this.right && this.top <= pos.y && pos.y < this.bottom;
	}
	
	mixin MAddOperator!(Rect, Point);
	mixin MSubOperator!(Rect, Point);
	mixin MMulOperator!(Rect, int);
	mixin MDivOperator!(Rect, int);
	mixin MOrOperator!(Rect, Rect);
	mixin MAndOperator!(Rect, Rect);
	
	/// 文字列に変換する。
	char[] toString() {return format("%s%s", position, size);}
	
	/// 比較する。
	int opCmp(Rect rhs) {
		if(int result = this.position.opCmp(rhs.position)) {
			return result;
		}
		return this.size.opCmp(rhs.size);
	}
	
	/// ditto
	int opCmp(Point rhs) {return this.position.opCmp(rhs);}
	
	/// ditto
	int opCmp(Size rhs) {return this.size.opCmp(rhs);}
	
	/// 座標。
	Point position;
	
	/// サイズ。
	Size size;
}
