﻿module outland.widget.widget;

import std.stdio;

import outland.tl.algorithm;

import outland.widget.event;
import outland.widget.graphics;
import outland.widget.mouse;
import outland.widget.keyboard;
import outland.widget.shape;

/// ウィジェットインターフェイス。
interface IWidget : IEventTarget {
	
	/// 位置。
	Point position();
	
	/// ditto
	void position(Point p);
	
	/// サイズ。
	Size size();
	
	/// ditto
	void size(Size s);
	
	/// 矩形。
	Rect rect();
	
	/// ditto
	void rect(Rect r);
	
	/// 最小サイズ。
	Size minimumSize();
	
	/// 推奨サイズ。
	Size preferredSize();
	
	/// 最大サイズ。
	Size maximumSize();

	/// クライアント座標をスクリーン座標にする。
	Point clientToScreen(Point p);
	
	/// スクリーン座標をクライアント座標にする。
	Point screenToClient(Point p);
	
	/// 描画。
	void draw(IGraphics g);
	
	/// 再描画領域の指定。
	void update(Rect r);
	
	/// 再描画。
	void update();
	
	/// 表示されているかどうか。
	bool visible();
	
	/// ditto
	void visible(bool b);
	
	/// 有効かどうか。
	bool enable();
	
	/// ditto
	void enable(bool e);
	
	/// 親ウィジェット。
	IWidget parent();
	
	/// ditto
	void parent(IWidget p);
	
	/// フレーム。
	IFrame frame();
}

/// レイアウトサイズ情報。
template MLayoutSizeProperty() {
	
	/// 最小サイズ。
	Size minimumSize() {return minSize_;}
	
	/// 最小サイズ。
	void minimumSize(Size s) {minSize_ = s;}
	
	/// 推奨サイズ。
	Size preferredSize() {return preferredSize_;}
	
	/// 推奨サイズ。
	void preferredSize(Size s) {preferredSize_ = s;}
	
	/// 最大サイズ。
	Size maximumSize() {return maxSize_;}
	
	/// 最大サイズ。
	void maximumSize(Size s) {maxSize_ = s;}
	
private:
	
	/// 最小サイズ。
	Size minSize_ = Size.min;
	
	/// 推奨サイズ。
	Size preferredSize_;
	
	/// 最大サイズ。
	Size maxSize_ = Size.max;
}

/// 抽象ウィジェットクラス。
abstract class Widget : IWidget {
	mixin MLayoutSizeProperty;
	
	/// 位置。
	Point position() {return rect_.position;}
	
	/// サイズ。
	Size size() {return rect_.size;}
	
	/// 矩形。
	Rect rect() {return rect_;}
	
	/// クライアント座標をスクリーン座標にする。
	Point clientToScreen(Point p) {
		if(parent_ !is null) {
			return parent_.clientToScreen(p + position);
		} else {
			return p + position;
		}
	}
	
	/// スクリーン座標をクライアント座標にする。
	Point screenToClient(Point p) {
		if(parent_ !is null) {
			return parent_.screenToClient(p) - position;
		} else {
			return p - position;
		}
	}

	/// 再描画領域の指定。
	void update(Rect r) {
		if(frame_ && visible_) {
			r.position = frame_.screenToClient(clientToScreen(r.position));
			frame_.update(r);
		}
	}
	
	/// 再描画。
	void update() {if(frame_ && visible_) frame_.update(this.rect);}
	
	/// 位置。(ローカル座標)
	void position(Point p) {
		if(rect_.position == p) return;
		
		// 以前の領域を再描画。
		update();
		
		rect_.position = p;
		
		// 次の領域を再描画。
		update();
	}
	
	/// サイズ。
	void size(Size s) {
		if(rect_.size == s) return;
		
		// 以前の領域を再描画。
		update();
		
		rect_.size = s;
		
		// 次の領域を再描画。
		update();
	}
	
	/// 矩形。
	void rect(Rect r) {
		if(rect_ == r) return;
		
		// 以前の領域を再描画。
		update();
		
		rect_ = r;
		
		// 次の領域を再描画。
		update();
	}
	
	/// 表示されているかどうか。
	bool visible() {return visible_;}
	
	/// ditto
	void visible(bool b) {
		bool old = visible_;
		
		// 消えるときは再描画。
		if(b != old && old) update();
		
		visible_ = b;
		
		// 表示されるときは再描画。
		if(b != old && !old) update();
	}
	
	/// 有効かどうか。
	bool enable() {return enable_;}
	
	/// ditto
	void enable(bool e) {
		// 再描画。
		if(e != enable_) update();
		enable_ = e;
	}
	
	/// 親ウィジェット。
	IWidget parent() {return parent_;}
	
	/// ditto
	void parent(IWidget p) {
		update();
		parent_ = p;
		frame_ = p ? p.frame : null;
		update();
	}
	
	/// フレーム。
	IFrame frame() {return frame_;}
	
	/// 破棄を行う。
	void close() {
		if(auto p = cast(Container) parent_) {
			p.remove(this);
			parent_ = null;
			frame_ = null;
		}
	}
	
private:
	
	/// 表示矩形。
	Rect rect_;
	
	/// 親ウィジェット。
	IWidget parent_;
	
	/// フレーム。
	IFrame frame_;
	
	/// 表示されているかどうか。
	bool visible_;
	
	/// 有効かどうか。
	bool enable_;
}

/// コントロールインターフェイス。
interface IControl : IWidget {
	
	/// イベントID。
	enum : EventId {
		/// イベントID終点。
		EVENT_ID_END = USER_ID_BEGIN,
	}
	
	/// イベントコード。
	enum : EventCode {		
		/// イベントコード終点。
		EVENT_CODE_END = USER_CODE_BEGIN,
	}
	
	/// イベントID。
	EventId eventId();
	
	/// ditto
	void eventId(EventId id);
}

/// イベントIDプロパティ。
template MEventIdProperty() {
	
	/// イベントID。
	EventId eventId() {return eventId_;}
	
	/// ditto
	void eventId(EventId id) {eventId_ = id;}
	
	/// ditto
	private EventId eventId_;
}

/// 抽象コントロールクラス。
abstract class Control : Widget, IControl {
	mixin MEventIdProperty;
	mixin MEventHandlerMap;
}

/// レイアウト戦略。
interface ILayoutStrategy {
	
	/**	項目を追加する。
	 *	既に指定項目があった場合はヒントだけ置き換える。
	 *
	 *	Params:
	 *		item	= 追加項目。
	 *		hint	= レイアウトのためのヒント。
	 */
	void add(IWidget item, Object hint);
	
	/// 項目を削除する。
	void remove(IWidget item);
	
	/// 全項目をクリアする。
	void clear();
	
	/// レイアウトを実行する。
	void layout(IContainer cont);
	
	/// 最小サイズ。
	Size minimumSize(IContainer cont);
	
	/// 推奨サイズ。
	Size preferredSize(IContainer cont);
	
	/// 最大サイズ。
	Size maximumSize(IContainer cont);
}

/// コンテナ。
interface IContainer : IWidget {
	
	/// 子ウィジェットの追加。
	void add(IWidget child);
	
	/// 子ウィジェットの削除。
	void remove(IWidget child);
	
	/// 子ウィジェット数。
	size_t childlenCount();
	
	/// 子ウィジェットの巡回。
	int opApply(int delegate(inout IWidget) dg);
	
	/// ditto
	int opApply(int delegate(uint, inout IWidget) dg);
	
	/// レイアウト戦略。
	void layoutStrategy(ILayoutStrategy st);
	
	/// ditto
	ILayoutStrategy layoutStrategy();
	
	/// 子ウィジェットを再配置する。
	void pack();
}

/// ウィジェットコンテナ実装。
template MWidgetContainer() {
	
	/// 描画。
	void draw(IGraphics g) {
		foreach(c; this) {
			if(c is null || !c.visible) continue;
			
			auto org = frame.screenToClient(clientToScreen(c.position));
			g.duringBackup({
				g.origin = org;
				g.clip = g.clip & Rect(org, c.size);
				c.draw(g);
			});
		}
	}
	
	/// 子ウィジェットの追加。
	void add(IWidget child)
	in {
		assert(child !is null);
	} body {
		children_ ~= child;
		child.parent = this;
	}
	
	/// 子ウィジェットの削除。
	void remove(IWidget child) {.outland.tl.algorithm.remove(children_, child);}
	
	/// 子ウィジェット数。
	size_t childlenCount() {return children_.length;}
	
	/// 子ウィジェットの巡回。
	int opApply(int delegate(inout IWidget) dg) {
		foreach(c; children_) if(int result = dg(c)) return result;
		return 0;
	}
	
	/// ditto
	int opApply(int delegate(uint, inout IWidget) dg) {
		foreach(i, c; children_) if(int result = dg(i, c)) return result;
		return 0;
	}
	
	/// ditto
	int opApplyReverse(int delegate(inout IWidget) dg) {
		foreach_reverse(c; children_) if(int result = dg(c)) return result;
		return 0;
	}
	
	/// ditto
	int opApplyReverse(int delegate(uint, inout IWidget) dg) {
		foreach_reverse(i, c; children_) if(int result = dg(i, c)) return result;
		return 0;
	}
	
	/// レイアウト戦略。
	void layoutStrategy(ILayoutStrategy st) {
		if(layoutStrategy_) layoutStrategy_.clear();
		layoutStrategy_ = st;
	}
	
	/// ditto
	ILayoutStrategy layoutStrategy() {return layoutStrategy_;}
	
	/// 子ウィジェットを再配置する。
	void pack() {if(layoutStrategy_) layoutStrategy_.layout(this);}
	
private:
	
	/// 子ウィジェット。
	IWidget[] children_;
	
	/// レイアウト戦略。
	ILayoutStrategy layoutStrategy_;
}

/// 基本的なコンテナクラス。
class Container : Widget, IContainer {
	mixin MWidgetContainer;
	mixin MEventHandlerMap;
	
	// 名称の隠蔽を防ぐ。
	alias Widget.size size;
	alias Widget.rect rect;
	
	/// デフォルトコンストラクタ。
	this() {
		auto t = delegateToEventTarget(&dispatchMouseEvent);
		foreach(code; [EVENT_CODE_MOUSE_MOVE,
					   EVENT_CODE_MOUSE_DOWN,
					   EVENT_CODE_MOUSE_UP,
					   EVENT_CODE_MOUSE_DOUBLE_CLICK,
					   EVENT_CODE_MOUSE_WHEEL]) {
			addTargetAny(code, t);
		}
	}
	
	/// サイズ変更。
	override void size(Size s) {
		if(s != super.size) {
			super.size(s);
			
			// 再配置。
			pack();
		}
	}
	
	/// 矩形変更。
	override void rect(Rect r) {
		auto old = this.size;
		super.rect(r);
		
		// サイズ変更時には再配置。
		if(r.size != old) pack();
	}
	
private:
	
	/// マウス移動イベント。
	bool dispatchMouseEvent(MouseEvent e)
	in {
		assert(e !is null);
	} body {
		foreach_reverse(c; this) if(e.position in c.rect) return c.handle(e);
		return false;
	}
}

/// フレームのインターフェイス。
interface IFrame : IContainer, IControl {
	
	/// フォーカスされているウィジェット。
	IWidget keyboardFocus();
	
	/// ditto
	void keyboardFocus(IWidget w);
	
	/// マウス入力を受け取るウィジェット。
	IWidget mouseCapture();
	
	/// ditto
	void mouseCapture(IWidget w);
	
	/// タイトル表示有無。
	void showTitle(bool b);
	
	/// ditto
	bool showTitle();
	
	/// タイトル文字列。
	char[] title();
	
	/// ditto
	void title(char[] str);
	
	/// 最大化有無。
	void maximizeable(bool b);
	
	/// ditto
	bool maximizeable();
	
	/// 最大化。
	void maximize(bool b);
	
	/// ditto
	bool maximize();
	
	/// 最小化有無。
	void minimizeable(bool b);
	
	/// ditto
	bool minimizeable();
	
	/// 最小化。
	void minimize(bool b);
	
	/// ditto
	bool minimize();
	
	/// サイズ変更有無。
	void resizeable(bool b);
	
	/// ditto
	bool resizeable();
	
	/// 閉じるボタン有無。
	void closeable(bool b);
	
	/// ditto
	bool closeable();
	
	/// 閉じられた時に終了するかどうか。
	void quitOnClose(bool b);
	
	/// ditto
	bool quitOnClose();
	
	/// 再描画領域の指定。
	void update(Rect r);
	
	/// グラフィックスに対する処理。
	void duringGraphics(void delegate(IGraphics g) dg);
	
	/// 破棄を行う。
	void close();
}
