﻿module outland.widget.win32.graphics;

import std.stdio;
import std.c.string;

import win32.windows;

import outland.widget.error;
import outland.widget.graphics;
import outland.widget.shape;

import outland.widget.win32.error;
import outland.widget.win32.string;

/// Win32イメージ。
class WinImage : IImage {
	
	/**	DCとイメージ情報を指定して生成する。
	 *	Params:
	 *		dc		= 元にするDC。
	 *		size	= ビットマップサイズ。
	 *		fmt		= ピクセルフォーマット。
	 *		pixels	= ピクセルデータ。
	 *		palette	= パレットデータ。RGBピクセルでは無視。
	 *	Throws:
	 *		NotSupportedException	ピクセルフォーマットが未対応だった場合に投げられる。
	 *		Win32Exception	処理失敗時に投げられる。
	 */
	this(HDC dc, Size size, Format fmt, void[] pixels, Color[] palette) {
		// ヘッダ設定。
		BITMAPINFOHEADER header;
		PaletteEntry[] bmpPalette;
		scope(exit) delete bmpPalette;
		makeBitmapInformationHeader(header, bmpPalette, size, fmt, palette);
		
		// ビットマップ情報構築。
		void[] buf;
		scope(exit) delete buf;
		buf.length = header.sizeof + PaletteEntry.sizeof * bmpPalette.length;
		memcpy(buf.ptr, &header, header.sizeof);
		memcpy(buf[header.sizeof .. $].ptr, bmpPalette.ptr, PaletteEntry.sizeof * bmpPalette.length);
		
		// ビットマップ設定。
		bitmap_ = CreateDIBitmap(dc, &header, CBM_INIT, pixels.ptr, cast(BITMAPINFO*) buf.ptr, DIB_RGB_COLORS);
		if(!bitmap_) throw new Win32Exception;
	}
	
	/// デストラクタ。
	~this() {close();}
	
	/// イメージサイズ。
	Size size() {
		BITMAP bmp;
		if(!GetObject(bitmap_, bmp.sizeof, &bmp)) throw new Win32Exception;
		return Size(bmp.bmWidth, bmp.bmHeight);
	}
	
	/// ピクセルフォーマット。
	Format format() {return format_;}
	
	/// イメージの破棄。
	void close() {
		if(bitmap_) {
			DeleteObject(bitmap_);
			bitmap_ = null;
		}
	}
	
private:

	/// パレット項目。
	struct PaletteEntry {
		/// コンストラクタ。
		static PaletteEntry opCall(DWORD val) {
			PaletteEntry e;
			e.value = val;
			return e;
		}
		
		/// コンストラクタ。
		static PaletteEntry opCall(Color c) {
			PaletteEntry e;
			e.color.rgbBlue = c.blue;
			e.color.rgbGreen = c.green;
			e.color.rgbRed = c.red;
			return e;
		}
		
		union {
			DWORD value;
			RGBQUAD color;
		}
	}
	
	/// フォーマットからビットマップ情報に変換。
	void makeBitmapInformationHeader(
			out BITMAPINFOHEADER header, out PaletteEntry[] bmpPalette, Size size, Format fmt, Color[] palette) {
		// サイズの設定。
		with(header) {
			biSize = BITMAPINFOHEADER.sizeof;
			biPlanes = 1;
			biWidth = size.width;
			biHeight = -size.height;
		}
		
		// パレットをコピーする。
	 	void copyPalette(size_t len) {
	 		bmpPalette.length = len;
	 		foreach(i, c; palette) {
				if(i < bmpPalette.length) {
					bmpPalette[i] = PaletteEntry(c);
				} else {
					break;
				}
			}
	 	}
									 	
		// ビットフィールドを設定する。
	 	void setBitFields(DWORD r, DWORD g, DWORD b) {
	 		bmpPalette ~= PaletteEntry(r);
			bmpPalette ~= PaletteEntry(g);
			bmpPalette ~= PaletteEntry(b);
	 	}
		
		// ビット数・ピクセル形式・パレット・マスクの設定。
		switch(fmt) {
		case Format.MONOCHLOME:
			header.biBitCount = 1;
			copyPalette(2);
			break;
		case Format.PALETTE16:
			header.biBitCount = 4;
			copyPalette(16);
			break;
		case Format.PALETTE256:
			header.biBitCount = 8;
			copyPalette(256);
			break;
		case Format.RGB15:
			header.biCompression = BI_BITFIELDS;
			header.biBitCount = 16;
			setBitFields(0x7C00, 0x03E0, 0x001F);
			break;
		case Format.RGB16:
			header.biCompression = BI_BITFIELDS;
			header.biBitCount = 16;
			setBitFields(0xF800, 0x07E0, 0x001F);
			break;
		case Format.RGB24:
			header.biCompression = BI_RGB;
			header.biBitCount = 24;
			break;
		case Format.RGB32:
			header.biCompression = BI_RGB;
			header.biBitCount = 32;
			break;
		default:
			throw new NotSupportedException("Not support pixel format!");
			break;
		}
		// 1ライン4バイト単位に切り上げたイメージサイズ。
		header.biSizeImage = (size.width * header.biBitCount + 7) / 8;
		header.biSizeImage = (header.biSizeImage + 3) / 4 * 4;
		header.biSizeImage *= size.height;
	}
	
	/// ビットマップ。
	HBITMAP bitmap_;
	
	/// ピクセルフォーマット。
	final Format format_;
}

/// Win32グラフィックスインターフェイス。
class WinGraphics : IGraphics {
	
	/// DCデストラクタ。
	alias void delegate(HDC) Destructor;
	
	/// DCとデストラクタを指定して生成する。
	this(HDC dc, Destructor dest = null)
	in {
		assert(dc !is null);
	} body {
		// 失敗時はDC破棄。
		scope(failure) if(dest) dest(dc);
		
		// 転送用DCの生成。
		transDC_ = CreateCompatibleDC(dc);
		if(!transDC_) throw new Win32Exception;
		scope(failure) DeleteDC(transDC_);
		
		// 現在の状態の記憶。
		oldState_ = SaveDC(dc);
		if(!oldState_) throw new Win32Exception;
		scope(failure) RestoreDC(dc, oldState_);
		
		// DCの設定。
		if(!SetBkMode(dc, TRANSPARENT)
				|| SetTextAlign(dc, TA_TOP | TA_LEFT | TA_NOUPDATECP) == GDI_ERROR) throw new Win32Exception;
		dc_ = dc;
		destructor_ = dest;
	}
	
	/// デストラクタ。
	~this() {close();}
	
	/// 文字色。
	Color textColor() {
		COLORREF c = GetTextColor(dc_);
		if(c == CLR_INVALID) throw new Win32Exception;
		return toColor(c);
	}
	
	/// ditto
	void textColor(Color c) {
		if(SetTextColor(dc_, fromColor(c)) == CLR_INVALID) throw new Win32Exception;
	}
	
	/// 背景色。
	Color backColor() {
		COLORREF c = GetBkColor(dc_);
		return (c == CLR_INVALID) ? Color.INVALID : toColor(c);
	}
	
	/// ditto
	void backColor(Color c) {
		if(SetBkColor(dc_, fromColor(c)) == CLR_INVALID) throw new Win32Exception;
	}
	
	/// 線色。
	Color lineColor() {
		LOGPEN log;
		log.lopnColor = CLR_INVALID;
		return (!GetObject(pen_, log.sizeof, &log)) ? Color.INVALID : toColor(log.lopnColor);
	}
	
	/// ditto
	void lineColor(Color c) {
		HPEN pen = CreatePen(PS_SOLID, 0, fromColor(c));
		if(!pen) throw new Win32Exception;
		deletePen();
		pen_ = pen;
	}
	
	/// 塗りつぶし色。
	void fillColor(Color c) {
		// ペンとブラシの生成。
		HBRUSH brush = CreateSolidBrush(fromColor(c));
		if(!brush) throw new Win32Exception;
		scope(failure) DeleteObject(brush);
		
		HPEN pen = CreatePen(PS_SOLID, 0, fromColor(c));
		if(!pen) throw new Win32Exception;
		
		// 以前のブラシの破棄。
		deleteFillBrush();
		
		// 新しいブラシとペンの設定。
		fillBrush_ = brush;
		fillPen_ = pen;
	}
	
	/// ditto
	Color fillColor() {
		LOGBRUSH log;
		log.lbColor = CLR_INVALID;
		return (!GetObject(fillBrush_, log.sizeof, &log)) ? Color.INVALID : toColor(log.lbColor);
	}
	
	/// 塗りつぶしイメージ。
	void fillImage(IImage img)
	in {
		assert(img !is null);
		assert(cast(WinImage) img);
	} body {
		auto wimg = cast(WinImage) img;
		if(wimg is null) throw new NotSupportedException("Not supported image format!");
		HBRUSH brush = CreatePatternBrush(wimg.bitmap_);
		if(!brush) throw new Win32Exception;
		deleteImageBrush();
		imageBrush_ = brush;
		fillImage_ = wimg;
	}
	
	/// ditto
	WinImage fillImage() {return fillImage_;}
	
	/// フォント。
	Font font() {
		LOGFONT log;
		if(!GetObject(font_, log.sizeof, &log)) return Font.init;
		return Font(fromTString(log.lfFaceName.ptr),
					pixelToPoint(log.lfHeight, Axis.Y),
					log.lfWeight > FW_NORMAL,
					log.lfUnderline != FALSE,
				    log.lfStrikeOut != FALSE,
					log.lfItalic != FALSE);
	}
	
	/// ditto
	void font(Font f) {
		HFONT hf = CreateFont(
			pointToPixel(f.size, Axis.Y),
			0,
			0,
			0,
			f.bold ? FW_BOLD : FW_NORMAL,
			f.italic ? TRUE : FALSE,
			f.under ? TRUE : FALSE,
			f.strike ? TRUE : FALSE,
			GetTextCharset(dc_),
			OUT_DEFAULT_PRECIS,
			CLIP_DEFAULT_PRECIS,
			DEFAULT_QUALITY,
			DEFAULT_PITCH | FF_DONTCARE,
			toTString(f.face));
		if(!hf) throw new Win32Exception;
		deleteFont();
		font_ = hf;
	}
	
	/// 点の描画。
	void dot(Point p, Color c) {
		if(!SetPixelV(dc_, p.x, p.y, fromColor(c))) throw new Win32Exception;
	}
	
	/// 線の描画。
	void line(Point p1, Point p2) {
		duringSelect(pen_, {
			if(!MoveToEx(dc_, p1.x, p1.y, null)) throw new Win32Exception;
			if(!LineTo(dc_, p2.x, p2.y)) throw new Win32Exception;
		});
	}
	
	/// 矩形の描画。
	void colorFill(Rect r) {
		duringSelect(fillBrush_, {
			duringSelect(fillPen_, {
				if(!Rectangle(dc_, r.left, r.top, r.right, r.bottom)) throw new Win32Exception;
			});
		});
	}
	
	/// イメージ矩形の描画。
	void imageFill(Rect r) {
		RECT rect;
		rect.left = r.left;
		rect.top = r.top;
		rect.right = r.right;
		rect.bottom = r.bottom;
		if(!FillRect(dc_, &rect, imageBrush_)) throw new Win32Exception;
	}
	
	/// 枠の描画。
	void frame(Rect r) {
		Point rt = r.rightTop;
		Point rb = r.rightBottom;
		Point lb = r.leftBottom;
		rt.offset(-1, 0);
		rb.offset(-1, -1);
		lb.offset(0, -1);
		line(r.leftTop, rt);
		line(rt, rb);
		line(rb, lb);
		line(lb, r.leftTop);
	}
	
	/// テキストの描画。
	void text(Point p, char[] str) {
		LPCTSTR s = toTString(str);
		size_t len = lstrlen(s);
		duringSelect(font_, {
			if(!TextOut(dc_, p.x, p.y, s, len)) throw new Win32Exception;
		});
	}
	
	/// テキストサイズの取得。
	Size getTextSize(char[] str) {
		LPCTSTR s = toTString(str);
		size_t len = lstrlen(s);
		SIZE size;
		duringSelect(font_, {
			if(!GetTextExtentPoint32(dc_, s, len, &size)) throw new Win32Exception;
		});
		return Size(size.cx, size.cy);
	}
	
	/// 使用可能なフォントフェイス名。
	char[][] getFontFaces() {
		extern(Windows) static int eachFont(LOGFONT* font, TEXTMETRIC* metric, uint type, LPARAM param) {
			if(font) *(cast(char[][]*) param) ~= fromTString((font.lfFaceName ~ '\0').ptr);
			return TRUE;
		}
		
		const LOGFONT log = {
			lfCharSet:DEFAULT_CHARSET,
			lfFaceName:"",
			lfPitchAndFamily:0,
		};
		char[][] names;
		EnumFontFamiliesEx(dc_, &log, cast(FONTENUMPROC) &eachFont, cast(LPARAM) &names, 0);
		return names;
	}
	
	/// ビットブロック転送。
	void transfer(Point dest, IGraphics g, Rect src, BltMode mode)
	in {
		assert(g !is null);
	} body {
		auto wg = cast(WinGraphics) g;
		if(!wg) throw new NotSupportedException("Not support Graphics type!");
		if(!BitBlt(dc_, dest.x, dest.y, src.width, src.height, wg.dc_, src.x, src.y, toROP(mode))) {
			throw new Win32Exception;
		}
	}
	
	/// 伸縮を行うビットブロック転送。
	void transfer(Rect dest, IGraphics g, Rect src, BltMode mode) {
		auto wg = cast(WinGraphics) g;
		if(!wg) throw new NotSupportedException("Not support Graphics type!");
		if(!StretchBlt(dc_, dest.x, dest.y, dest.width, dest.height,
					   wg.dc_, src.x, src.y, src.width, src.height, toROP(mode))) {
			throw new Win32Exception;
		}
	}
	
	/// イメージ転送。
	void transfer(Point dest, IImage img, Rect src, BltMode mode) {
		auto wimg = cast(WinImage) img;
		if(!wimg) throw new NotSupportedException("Not support Image type!");
		
		// ビットマップ選択。
		auto old = cast(HBITMAP) SelectObject(transDC_, wimg.bitmap_);
		if(!old) throw new Win32Exception;
		scope(exit) SelectObject(transDC_, old);
		
		// 転送。
		if(!BitBlt(dc_, dest.x, dest.y, src.width, src.height, transDC_, src.x, src.y, toROP(mode))) {
			throw new Win32Exception;
		}
	}
	
	/// 伸縮を行うメージ転送。
	void transfer(Rect dest, IImage img, Rect src, BltMode mode) {
		auto wimg = cast(WinImage) img;
		if(!wimg) throw new NotSupportedException("Not support Image type!");
		
		// ビットマップ選択。
		auto old = cast(HBITMAP) SelectObject(transDC_, wimg.bitmap_);
		if(!old) throw new Win32Exception;
		scope(exit) SelectObject(transDC_, old);
		
		// 転送。
		if(!StretchBlt(dc_, dest.x, dest.y, dest.width, dest.height,
					   transDC_, src.x, src.y, src.width, src.height, toROP(mode))) {
			throw new Win32Exception;
		}
	}
	
	/**	イメージの生成。
	 *	Params:
	 *		size	= イメージのサイズ。
	 *		fmt		= ピクセルフォーマット。
	 *		pixels	= ピクセルデータ。ピクセルフォーマットに合っていなければならない。
	 *		palette	= パレット。RGBフォーマット時は無視。
	 *	Throws:
	 *		WidgetException	エラー発生時に投げられる。
	 */
	WinImage makeImage(Size size, IImage.Format fmt, void[] pixels, Color[] palette) {
		return new WinImage(dc_, size, fmt, pixels, palette);
	}
	
	/// 座標原点。
	void origin(Point org) {
		if(!SetViewportOrgEx(dc_, org.x, org.y, null)) throw new Win32Exception;
	}
	
	/// ditto
	Point origin() {
		POINT point;
		if(!GetViewportOrgEx(dc_, &point)) throw new Win32Exception;
		return Point(point.x, point.y);
	}
	
	/// クリップ領域。
	void clip(Rect r) {
		HRGN rgn = CreateRectRgn(r.left, r.top, r.right, r.bottom);
		if(!rgn) throw new Win32Exception;
		scope(exit) DeleteObject(rgn);
		if(!SelectObject(dc_, rgn)) throw new Win32Exception;
	}
	
	/// ditto
	Rect clip() {
		auto org = this.origin;
		RECT rect;
		if(GetClipBox(dc_, &rect) == ERROR) throw new Win32Exception;
		return Rect(Point(rect.left + org.x, rect.top + org.y), Size(rect.right - rect.left, rect.bottom - rect.top));
	}
	
	/// 描画モード。
	void drawMode(DrawMode mode) {
		int m = R2_COPYPEN;
		switch(mode) {
		case DrawMode.NOT:	m = R2_NOT; break;
		case DrawMode.OR:	m = R2_MERGEPEN; break;
		case DrawMode.AND:	m = R2_MASKPEN; break;
		case DrawMode.XOR:	m = R2_XORPEN; break;
		default:			m = R2_COPYPEN; break;
		}
		if(!SetROP2(dc_, m)) throw new Win32Exception;
	}
	
	/// ditto
	DrawMode drawMode() {
		switch(GetROP2(dc_)) {
		case R2_NOT:		return DrawMode.NOT;
		case R2_MERGEPEN:	return DrawMode.OR;
		case R2_MASKPEN:	return DrawMode.AND;
		case R2_XORPEN:		return DrawMode.XOR;
		default:			return DrawMode.COPY;
		}
	}
	
	/// ポイント値をピクセルに変換。
	uint pointToPixel(float pt, Axis axis) {
		int ppi = GetDeviceCaps(dc_, (axis == Axis.X) ? LOGPIXELSX : LOGPIXELSY);
		return cast(uint)(pt * ppi / POINT_PER_INCH);
	}
	
	/// ピクセルをポイント値に変換。
	float pixelToPoint(uint p, Axis axis) {
		int ppi = GetDeviceCaps(dc_, (axis == Axis.X) ? LOGPIXELSX : LOGPIXELSY);
		return cast(float)(p * POINT_PER_INCH / ppi);
	}
	
	/// 状態を記憶して処理を行う。
	void duringBackup(void delegate() dg) {
		// 状態を記憶。
		Color tc = textColor();
		Color bc = backColor();
		Color lc = lineColor();
		Color fc = fillColor();
		WinImage fi = fillImage();
		Font f = font();
		Point org = origin();
		Rect cl = clip();
		DrawMode m = drawMode();
		
		// 処理終了時に戻す。
		scope(exit) {
			if(tc != Color.INVALID) textColor(tc);
			if(bc != Color.INVALID) backColor(bc);
			if(lc != Color.INVALID) lineColor(lc);
			if(fc != Color.INVALID) fillColor(fc);
			if(fi) fillImage(fi);
			font(f);
			origin(org);
			clip(cl);
			drawMode(m);
		}
		
		// 処理実行。
		dg();
	}
	
	/// イメージを描画対象とする。
	void duringDrawImage(IImage img, void delegate() dg) {
		auto wimg = cast(WinImage) img;
		if(!wimg) throw new NotSupportedException("Not support Image type!");
		
		// イメージ選択。
		auto old = SelectObject(dc_, wimg.bitmap_);
		if(!old) throw new Win32Exception;
		scope(exit) SelectObject(dc_, old);
		
		// 処理実行。
		dg();
	}
	
	/// グラフィックスの破棄。
	void close() {
		if(dc_) {
			RestoreDC(dc_, oldState_);
			deletePen();
			deleteFillBrush();
			deleteImageBrush();
			if(destructor_) destructor_(dc_);
			dc_ = null;
			DeleteDC(transDC_);
			transDC_ = null;
		}
	}
	
private:
	
	/// 転送モードをフラグに変換する。
	DWORD toROP(BltMode mode) {
		switch(mode) {
		case BltMode.NOT:	return NOTSRCCOPY;
		case BltMode.OR:	return SRCPAINT;
		case BltMode.AND:	return SRCAND;
		case BltMode.XOR:	return SRCINVERT;
		default:			return SRCCOPY;
		}
	}
	
	/// COLORREFをColorに変換する。
	static Color toColor(COLORREF c) {
		return (c == CLR_INVALID) ? Color.INVALID : Color(GetRValue(c), GetGValue(c), GetBValue(c));
	}
	
	/// ColorをCOLORREFに変換する。
	static COLORREF fromColor(Color c) {
		return (c == Color.INVALID) ? CLR_INVALID : RGB(c.red, c.green, c.blue);
	}
	
	/// ペンの破棄。
	void deletePen() {
		if(pen_) {
			DeleteObject(pen_);
			pen_ = null;
		}
	}
	
	/// 塗りつぶしブラシの破棄。
	void deleteFillBrush() {
		if(fillBrush_) {
			DeleteObject(fillBrush_);
			DeleteObject(fillPen_);
			fillBrush_ = null;
			fillPen_ = null;
		}
	}
	
	/// イメージブラシの破棄。
	void deleteImageBrush() {
		if(imageBrush_) {
			DeleteObject(imageBrush_);
			imageBrush_ = null;
			fillImage_ = null;
		}
	}
	
	/// フォントの破棄。
	void deleteFont() {
		if(font_) {
			DeleteObject(font_);
			font_ = null;
		}
	}
	
	/// オブジェクト選択中に処理を行う。
	void duringSelect(HGDIOBJ obj, void delegate() dg) {
		auto old = SelectObject(dc_, obj);
		if(!old) throw new Win32Exception;
		scope(exit) SelectObject(dc_, old);
		dg();
	}
	
	/// デバイスコンテキスト。
	HDC dc_;
	
	/// 以前の状態。
	int oldState_;
	
	/// 破棄用の関数。
	Destructor destructor_;
	
	/// 生成されたペン。
	HPEN pen_;
	
	/// 塗りつぶし用ブラシ。
	HBRUSH fillBrush_;
	
	/// 塗りつぶし用ペン。
	HBRUSH fillPen_;
	
	/// イメージブラシ。
	HBRUSH imageBrush_;
	
	/// 生成されたフォント。
	HFONT font_;
	
	/// ブラシ用イメージ。
	WinImage fillImage_;
	
	/// イメージ転送用DC。
	HDC transDC_;
}
