﻿// Copyright (C) 2010 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: BrandListConfig.cs 355 2010-04-16 03:47:53Z panacoran $

using System.Collections.Generic;
using System.IO;

namespace Protra.Lib.Config
{
    /// <summary>
    /// 銘柄リストを格納するクラス。
    /// </summary>
    public class BrandList
    {
        /// <summary>
        /// 銘柄リストの名前を取得または設定する。
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// 銘柄リストを取得または設定する。
        /// </summary>
        public List<string> List { get; set; }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public BrandList()
        {
            List = new List<string>();
        }

        /// <summary>
        /// 文字列形式を返す。
        /// </summary>
        /// <returns>銘柄リストの名前。</returns>
        public override string ToString()
        {
            return Name;
        }
    }

    /// <summary>
    /// 銘柄リストの設定を読み書きするクラス。
    /// </summary>
    public class BrandListConfig : ConfigBase
    {
        /// <summary>
        /// 銘柄リストリストを取得または設定する。
        /// </summary>
        public List<BrandList> List { get; set; }

        /// <summary>
        /// 設定ファイルの名前を取得する。
        /// </summary>
        protected override string ConfigName
        {
            get { return "brandlist"; }
        }

        /// <summary>
        /// 銘柄リストを読み込む。5桁の銘柄IDを証券コードに変換して、既定の銘柄リストを挿入する。
        /// </summary>
        public override void Load()
        {
            List = new List<BrandList>();
            base.Load();
            foreach (var bl in List)
                for (int i = 0; i < bl.List.Count; i++)
                    if (bl.List[i].Length > 4)
                        bl.List[i] = bl.List[i].Substring(0, 4);
            SetDefaultBrandList();
        }

        /// <summary>
        /// 株価指数、日経平均構成銘柄、売買代金上位500位の銘柄リストを挿入する。
        /// すでに存在する場合には作り直す。
        /// </summary>
        public void SetDefaultBrandList()
        {
            if (List.Count >= 3 && List[2].Name == "売買代金上位500位")
                List.RemoveRange(0, 3);

            var indexes = new BrandList();
            indexes.Name = "株価指数";
            indexes.List.AddRange(new string[] { "1001", "1002" });
            List.Insert(0, indexes);

            var n225 = new BrandList();
            n225.Name = "日経平均構成銘柄";
            List.Insert(1, n225);
            n225.List = GlobalEnv.BrandData.Nikkei225();

            var a500 = new BrandList();
            a500.Name = "売買代金上位500位";
            List.Insert(2, a500);
            a500.List = GlobalEnv.BrandData.A500();
        }

        /// <summary>
        /// 挿入した銘柄リストを削除してから保存する。
        /// </summary>
        public override void Save()
        {
            var prev = List;
            if (List.Count >= 3 && List[2].Name == "売買代金上位500位")
                List = List.GetRange(3, List.Count - 3);
            base.Save();
            List = prev;
        }

        /// <summary>
        /// 古い設定ファイルを読み込む。
        /// </summary>
        protected override void ReadOldConfig()
        {
            ConfigFile configFile = new ConfigFile("ptenv");
            BrandList favorite = new BrandList();
            favorite.List = new List<string>();
            favorite.Name = "お気に入り";
            try
            {
                configFile.Read();
                for (int i = 0; ; i++)
                {
                    string val = configFile["FavoriteBrandId" + i.ToString()];
                    if (val == null)
                        break;
                    Db.Brand brand = Db.BrandTable.GetRecord(int.Parse(val));
                    favorite.List.Add(brand.Code.ToString());
                }
            }
            catch
            {}
            List.Add(favorite);
            
            Db.Connection.ThrowException = true;
            try
            {
                foreach (Db.BrandList brandList in Db.BrandListTable.GetAllRecords())
                {
                    BrandList bl = new BrandList();
                    bl.Name = brandList.Name;
                    foreach (Db.Brand brand in brandList.Brands)
                        bl.List.Add(brand.Code.ToString());
                    List.Add(bl);
                }
            }
            catch (System.Data.OleDb.OleDbException) // データベースがない場合は無視する。
            { }
        }
    }
}
