// Copyright(C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: PrefixedExpressionNode.cs,v 1.3 2007-12-27 08:34:39 panacoran Exp $

using System;

namespace Protra.Lib.Lang
{
	/// <summary>
	/// Oũm[h\NXłB
	/// </summary>
	public class PrefixedExpressionNode : Node
	{
		/// <summary>
		/// Z̎ނ\LN^
		/// </summary>
		char op;
		/// <summary>
		/// @pf̒lɑΉm[h
		/// </summary>
		Node atNode;
		/// <summary>
		/// LXg̃^Cv
		/// </summary>
		Value.Type type;
		/// <summary>
		/// ̃m[h
		/// </summary>
		Node expNode;

		/// <summary>
		/// ͂܂B
		/// </summary>
		/// <exception cref="Protra.Lib.Lang.ParseException">
		/// ɃG[ꍇthrow܂B
		/// </exception>
		/// <param name="context">ReLXg</param>
		public override void Parse(Context context)
		{
			this.ContextToken = (ContextToken)context.Current;

			ContextToken ct = (ContextToken)context.Current;
			string token = ct.ToString();
			switch(token)
			{
				case "{": // @pf
					op = token[0];
					context.MoveNext();
					atNode = new ExpressionNode();
					atNode.Parse(context);
					ct = (ContextToken)context.Current;
					token = ct.ToString();
					if(token != "}")
						throw new ParseException("'}' expected", this.ContextToken);
					context.MoveNext();
					break;
				case "+":
				case "-":
				case "!":
					op = token[0];
					context.MoveNext();
					break;
				case "(": // LXg̉\
					ContextToken rct = ct;
					context.MoveNext();
					ct = (ContextToken)context.Current;
					if(ct.IsType)
					{
						op = token[0];
						type = (Value.Type)Enum.Parse
							(typeof(Value.Type), ct.ToString(), true);
						context.MoveNext();
						ct = (ContextToken)context.Current;
						token = ct.ToString();
						if(token != ")")
							throw new ParseException("')' expected", this.ContextToken);
						context.MoveNext();
					}
					else
					{
						context.Reset(rct);
						expNode = new SuffixedExpressionNode();
						expNode.Parse(context);
						return;
					}
					break;
				default:
					expNode = new SuffixedExpressionNode();
					expNode.Parse(context);
					return;
			}

			expNode = new PrefixedExpressionNode();
			expNode.Parse(context);
		}

		/// <summary>
		/// \؂ȗ܂B
		/// </summary>
		/// <returns>ȗꂽm[h</returns>
		public override Node Simplify()
		{
			if(op == 0)
				return expNode.Simplify();
			else
			{
				expNode = expNode.Simplify();
				if(atNode != null)
					atNode = atNode.Simplify();
				return this;
			}
		}

		/// <summary>
		/// vOs܂B
		/// </summary>
		/// <exception cref="Protra.Lib.Lang.RuntimeException">
		/// vOsɃG[ꍇthrow܂B
		/// </exception>
		/// <param name="resource">\[X</param>
		/// <param name="at">@pf̒l</param>
		/// <returns>s</returns>
		public override Value Execute(Resource resource, int at)
		{
			switch(op)
			{
				case '{': // @pf
					Value val = atNode.Execute(resource, at);
					if(val.ValueType != Value.Type.Int)
						throw new RuntimeException("assigned non-int value for at operator", this.ContextToken);
					return expNode.Execute(resource, at + (int)val.InnerValue);
				case '+':
					return +expNode.Execute(resource, at);
				case '-':
					return -expNode.Execute(resource, at);
				case '!':
					return !expNode.Execute(resource, at);
				case '(': // LXg
					val = expNode.Execute(resource, at);
					if(val == null)
						throw new RuntimeException("null casted", this.ContextToken);
					return expNode.Execute(resource, at).Cast(type);
			}

			return expNode.Execute(resource, at);
		}
	}
}
