﻿// Copyright (C) 2003 Daisuke Arai <darai@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: PtFileTreeView.cs 306 2010-03-19 13:21:22Z panacoran $

using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.IO;

namespace Protra.Lib.Controls
{
	/// <summary>
	/// 拡張子がptのファイルのTreeViewです。
	/// </summary>
	public class PtFileTreeView : System.Windows.Forms.TreeView
	{
		/// <summary>
		/// ルートディレクトリ
		/// </summary>
		private string dir;

		private System.Windows.Forms.ImageList imageList;
		/// <summary>
		/// 必要なデザイナ変数です。
		/// </summary>
		private System.ComponentModel.IContainer components = null;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public PtFileTreeView()
		{
			// この呼び出しは、Windows.Forms フォーム デザイナで必要です。
			InitializeComponent();

			this.ImageList = this.imageList;
			this.BeforeCollapse += new TreeViewCancelEventHandler(OnBeforeCollapse);
			this.BeforeExpand += new TreeViewCancelEventHandler(OnBeforeExpand);
		}

		/// <summary>
		/// 使用されているリソースに後処理を実行します。
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary>
		/// デザイナ サポートに必要なメソッドです。このメソッドの内容を
		/// コード エディタで変更しないでください。
		/// </summary>
		private void InitializeComponent()
		{
			this.components = new System.ComponentModel.Container();
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(PtFileTreeView));
			this.imageList = new System.Windows.Forms.ImageList(this.components);
			// 
			// imageList
			// 
			this.imageList.ColorDepth = System.Windows.Forms.ColorDepth.Depth8Bit;
			this.imageList.ImageSize = new System.Drawing.Size(16, 16);
			this.imageList.ImageStream = ((System.Windows.Forms.ImageListStreamer)(resources.GetObject("imageList.ImageStream")));
			this.imageList.TransparentColor = System.Drawing.Color.Transparent;

		}
		#endregion

		private void OnBeforeCollapse(object sender, System.Windows.Forms.TreeViewCancelEventArgs e)
		{
			e.Node.ImageIndex = 0;
			e.Node.SelectedImageIndex = 0;
		}

		private void OnBeforeExpand(object sender, System.Windows.Forms.TreeViewCancelEventArgs e)
		{
			e.Node.Nodes.Clear();
			e.Node.ImageIndex = 1;
			e.Node.SelectedImageIndex = 1;
			string path = e.Node.FullPath;
			path = Path.Combine(dir, path);
			if(Directory.Exists(path))
			{
				string[] dirlist = Directory.GetDirectories(path);
				foreach(string s in dirlist)
				{
					TreeNode tn = new TreeNode();
					tn.Text = Path.GetFileName(s);
					tn.ImageIndex = 0;
					tn.SelectedImageIndex = 0;
					tn.Nodes.Add("dummy");
					e.Node.Nodes.Add(tn);
				}
				string[] filelist = Directory.GetFiles(path, "*.pt");
				foreach(string s in filelist)
				{
					TreeNode tn = new TreeNode();
					tn.Text = Path.GetFileNameWithoutExtension(s);
					tn.ImageIndex = 2;
					tn.SelectedImageIndex = 2;
					e.Node.Nodes.Add(tn);
				}
			}
		}

		/// <summary>
		/// ルートディレクトリを取得または設定します。
		/// </summary>
		public string RootDirectory
		{
			get { return dir; }
			set
			{
				dir = value;
				this.Nodes.Clear();
				if(Directory.Exists(dir))
				{
					string[] dirlist = Directory.GetDirectories(dir);
					foreach(string s in dirlist)
					{
						TreeNode tn = new TreeNode();
						tn.Text = Path.GetFileName(s);
						tn.ImageIndex = 0;
						tn.SelectedImageIndex = 0;
						tn.Nodes.Add("dummy");
						this.Nodes.Add(tn);
					}
					string[] filelist = Directory.GetFiles(dir, "*.pt");
					foreach(string s in filelist)
					{
						TreeNode tn = new TreeNode();
						tn.Text = Path.GetFileNameWithoutExtension(s);
						tn.ImageIndex = 2;
						tn.SelectedImageIndex = 2;
						this.Nodes.Add(tn);
					}
					this.Select();
				}
			}
		}

		/// <summary>
		/// 選択されているファイルを取得します。
		/// </summary>
		public string SelectedFile
		{
			get
			{
				if(this.SelectedNode != null)
				{
					string path = this.SelectedNode.FullPath;
					path = Path.Combine(dir, path + ".pt");
					if(File.Exists(path))
						return path;
				}
				return null;
			}
		}

		/// <summary>
		/// 選択されているファイルの相対パスを取得します。
		/// </summary>
		public string SelectedRelativeFile
		{
			get
			{
				if(this.SelectedNode != null)
				{
					string file = this.SelectedNode.FullPath + ".pt";
					string path = Path.Combine(dir, file);
					if(File.Exists(path))
						return file;
				}
				return null;
			}
		}

		/// <summary>
		/// 選択されているディレクトリを取得します。
		/// </summary>
		public string SelectedDirectory
		{
			get
			{
				if(this.SelectedNode != null)
				{
					string path = this.SelectedNode.FullPath;
					path = Path.Combine(dir, path);
					if(Directory.Exists(path))
						return path;
				}
				return null;
			}
		}
	}
}
