﻿// Copyright (C) 2004 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2004, 2010, 2014 panacoran <panacoran@users.sourceforge.jp>
//
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
//
// $Id: EditBrandListDialog.cs 508 2014-02-19 06:05:07Z panacoran $

using System;
using System.Windows.Forms;
using Protra.Lib.Config;

namespace Protra.Lib.Dialogs
{
    /// <summary>
    /// 銘柄リストを編集するためのダイアログ
    /// </summary>
    public partial class EditBrandListDialog : Form
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EditBrandListDialog()
        {
            InitializeComponent();

            foreach (var list in GlobalEnv.BrandListConfig.EditableList)
                listBoxBrandList.Items.Add(list);
            if (listBoxBrandList.Items.Count > 0)
                listBoxBrandList.SelectedIndex = 0;
        }

        private void buttonAdd_Click(object sender, EventArgs e)
        {
            using (var dialog = new EditBrandListContentDialog())
            {
                var result = dialog.ShowDialog(this);
                if (result != DialogResult.OK)
                    return;
                listBoxBrandList.Items.Add(dialog.SelectedList);
                listBoxBrandList.SelectedItem = dialog.SelectedList;
            }
        }

        private void buttonEdit_Click(object sender, EventArgs e)
        {
            var list = (BrandList)listBoxBrandList.SelectedItem;
            if (list == null)
                return;
            using (var dialog = new EditBrandListContentDialog())
            {
                dialog.SelectedList = list;
                var result = dialog.ShowDialog(this);
                if (result != DialogResult.OK)
                    return;
            }
            // Protraで編集されてインスタンスが変わっている可能性がある。
            var index = listBoxBrandList.SelectedIndex;
            listBoxBrandList.Items.RemoveAt(index);
            listBoxBrandList.Items.Insert(index, list);
            listBoxBrandList.SelectedIndex = index;
        }

        private void buttonDelete_Click(object sender, EventArgs e)
        {
            var list = (BrandList)listBoxBrandList.SelectedItem;
            if (list == null)
                return;
            var msg = string.Format("{0}を削除します。よろしいですか？", list.Name);
            using (new CenteredDialogHelper())
                if (MessageBox.Show(this, msg, "確認", MessageBoxButtons.OKCancel,
                                    MessageBoxIcon.Exclamation, MessageBoxDefaultButton.Button1) == DialogResult.Cancel)
                    return;
            var index = listBoxBrandList.SelectedIndex;
            listBoxBrandList.Items.Remove(list);
            GlobalEnv.BrandListConfig.List.Remove(list);
            GlobalEnv.BrandListConfig.Save();
            if (index < listBoxBrandList.Items.Count)
                listBoxBrandList.SelectedIndex = index;
            else if (listBoxBrandList.Items.Count > 0)
                listBoxBrandList.SelectedIndex = index - 1;
            else
                listViewList.Items.Clear();
        }

        private void listBoxBrandList_SelectedIndexChanged(object sender, EventArgs e)
        {
            var list = (BrandList)listBoxBrandList.SelectedItem;
            if (list == null)
                return;
            listViewList.BeginUpdate();
            listViewList.Items.Clear();
            var items = new ListView.ListViewItemCollection(listViewList);
            foreach (var code in list.List)
            {
                var brand = GlobalEnv.BrandData[code];
                items.Add(new ListViewItem(new[] {brand.MarketName, brand.Code, brand.Name}));
            }
            listViewList.EndUpdate();
        }
    }
}