﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008, 2013 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: FunctionDefinitionNode.cs 455 2013-06-08 11:52:33Z panacoran $

using System.Collections.Generic;

namespace Protra.Lib.Lang
{
    /// <summary>
    /// 関数定義のノードを表すクラス。
    /// </summary>
    public class FunctionDefinitionNode : Node
    {
        /// <summary>
        /// 関数名
        /// </summary>
        private string _name;

        /// <summary>
        /// 引数の個数
        /// </summary>
        private int _argnum;

        /// <summary>
        /// ノードのリスト
        /// </summary>
        private readonly List<Node> _nodeList = new List<Node>();

        /// <summary>
        /// ノードのリストを取得する。
        /// </summary>
        public List<Node> Nodes
        {
            get { return _nodeList; }
        }

        /// <summary>
        /// スタックフレームの大きさを取得または設定する。
        /// </summary>
        public int FrameSize { get; private set; }

        /// <summary>
        /// 構文解析する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.ParseException">
        /// 構文解析中にエラーが発生したときにthrowされる。
        /// </exception>
        /// <return>
        /// 解析した結果のノード。
        /// </return>
        public override Node Parse()
        {
            Token = Scanner.Token;

            // 関数名のチェック
            Scanner.Scan();
            if (Scanner.Token.Type != TokenType.FunctionName)
                throw new ParseException("function name expected", Scanner.Token);
            _name = Scanner.Token.Value;

            // 引数とローカル変数用の記号表を用意する。
            LvtStack.Push(new Dictionary<string, int>());
            LvtStack.Peek()["at"] = 0;
            // 引数のチェック
            Scanner.Scan();
            if (Scanner.Token.Value != "(")
            {
                if (Scanner.Token.Value != ";")
                    throw new ParseException("';' expected", Scanner.Token);
            }
            else
            {
                Scanner.Scan();
                if (Scanner.Token.Value == ")")
                {
                    Scanner.Scan();
                    if (Scanner.Token.Value != ";")
                        throw new ParseException("';' expected", Scanner.Token);
                }
                else
                {
                    do
                    {
                        if (Scanner.Token.Type != TokenType.LocalVariable)
                            throw new ParseException("function argument expected", Scanner.Token);
                        var lvt = LvtStack.Peek();
                        if (lvt.ContainsKey(Scanner.Token.Value))
                            throw new ParseException("duplicate argument name", Scanner.Token);
                        _argnum++;
                        lvt[Scanner.Token.Value] = lvt.Count;
                        Scanner.Scan();
                        if (Scanner.Token.Value == ")")
                        {
                            Scanner.Scan();
                            if (Scanner.Token.Value != ";")
                                throw new ParseException("';' expected", Scanner.Token);
                            break;
                        }
                        if (Scanner.Token.Value != ",")
                            throw new ParseException("',' or ')' expected", Scanner.Token);
                    } while (Scanner.Scan());
                }
            }

            // 本体のチェック
            while (Scanner.Scan())
            {
                if (Scanner.Token.Value == "end")
                {
                    Scanner.Scan();
                    if (Scanner.Token.Value != ";")
                        throw new ParseException("';' expected", Scanner.Token);
                    FrameSize = LvtStack.Peek().Count;
                    LvtStack.Pop();
                    return this;
                }
                _nodeList.Add(new StatementNode().Parse());
            }

            throw new ParseException("unexpected EOF", Token);
        }

        /// <summary>
        /// プログラムを実行する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <param name="resource">リソース</param>
        /// <param name="at">int型@作用素の値</param>
        /// <param name="ats">string型@作用素の値</param>
        /// <returns>実行結果</returns>
        public override Value Execute(Resource resource, int at, string ats)
        {
            // 関数テーブルに登録
            var ft = new FunctionType(_name, _argnum);
            if (resource.FunctionTable.ContainsKey(ft))
                throw new RuntimeException("function is already defined --- " + ft, Token);
            resource.FunctionTable[ft] = this;
            return null;
        }
    }
}