﻿// Copyright(C) 2008, 2010 panacorn <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: KabukaJohoUpdator.cs 321 2010-03-29 11:37:51Z panacoran $

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Text;
using Protra.Lib.Data;

namespace Protra.Lib.Update
{
	/// <summary>
	/// 株価情報を利用して株価データの更新を行うクラス。
	/// </summary>
	public class KabukaJohoUpdator: PriceDataUpdator
	{
        /// <summary>
        /// データが存在する最初の日付を取得する。
        /// </summary>
        public override DateTime DataSince
        {
            get { return new DateTime(2000, 1, 4); }
        }

        /// <summary>
        /// 新しいデータが置かれる時刻に達しているか。
        /// </summary>
        /// <param name="time">時刻</param>
        /// <returns></returns>
        protected override bool IsDataAvailable(DateTime time)
        {
            return time.Hour >= 16;
        }

        /// <summary>
        /// データのURLを取得する。
        /// </summary>
        /// <returns>URL</returns>
        protected override string DownloadUrl()
        {
            return "http://www.edatalab.net/kabu/data" +
                    Date.ToString((Date.Year < 2006)
                        ? @"yyyy\/yyyyMMdd"
                        : @"yyyy\/\DyyMMdd") + ".LZH";
        }

        /// <summary>
        /// 文字列を解析して価格データを返す。
        /// </summary>
        /// <param name="line">文字列</param>
        /// <returns>価格データ</returns>
        protected override Price ParseLine(string line)
        {
            string[] tokens = line.Split(',');
            Price r = new Price();
            try
            {
                r.Date = DateTime.Parse(tokens[0]);
                r.Code = tokens[1];
                if (r.Date.Year >= 2006)
                {
                    switch (int.Parse(tokens[2]))
                    {
                        case 1:
                            r.Market = "T1";
                            break;
                        case 2:
                            r.Market = "T2";
                            break;
                        case 3:
                            r.Market = "M";
                            break;
                        case 4:
                            r.Market = "J";
                            break;
                        case 6:
                            r.Market = "O1";
                            break;
                        case 7:
                            r.Market = "O2";
                            break;
                        case 8:
                            r.Market = "H";
                            break;
                        default:
                            return null;
                    }
                }
                // 指数の値に小数が含まれているのでdouble.Parseを利用する。
                r.Open = (int)double.Parse(tokens[3]);
                r.High = (int)double.Parse(tokens[4]);
                r.Low = (int)double.Parse(tokens[5]);
                r.Close = (int)double.Parse(tokens[6]);
                r.Volume = double.Parse(tokens[7]) / 1000;
                if (r.Date.Year < 2006)
                {
                    switch (int.Parse(tokens[8]))
                    {
                        case 1:
                            r.Market = "T1";
                            break;
                        case 2:
                            r.Market = "T2";
                            break;
                        case 3:
                            r.Market = "J";
                            break;
                        default:
                            return null;
                    }
                }
            }
            catch (FormatException)
            {
                // フォーマットエラーは全部無視する。
                return null;
            }
            return r;
        }
	}
}
