﻿// Copyright (C) 2003 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2004, 2005 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: Market.cs 306 2010-03-19 13:21:22Z panacoran $

using System;
using System.Collections;

namespace Protra.Lib.Db
{
	/// <summary>
	/// MarketのIdを表す列挙型。
	/// </summary>
	public enum MarketId
	{
		/// <summary>
		/// 東証1部
		/// </summary>
		Tokyo1 = 1,
		/// <summary>
		/// 東証2部
		/// </summary>
		Tokyo2,
		/// <summary>
		/// マザーズ
		/// </summary>
		Mothers,
		/// <summary>
		/// 東証外国部
		/// </summary>
		TokyoForeign,
		/// <summary>
		/// 大証1部
		/// </summary>
		Osaka1,
		/// <summary>
		/// 大証2部
		/// </summary>
		Osaka2,
		/// <summary>
		/// ヘラクレス
		/// </summary>
		Hercules,
		/// <summary>
		/// 名証1部
		/// </summary>
		Nagoya1,
		/// <summary>
		/// 名証2部
		/// </summary>
		Nagoya2,
		/// <summary>
		/// Jasdaq
		/// </summary>
		Jasdaq,
		/// <summary>
		/// 最大値
		/// </summary>
		Max = Jasdaq
	}

	/// <summary>
	/// Marketテーブルのレコードを表すクラスです。
	/// </summary>
	public class Market: IRecord
	{
		private MarketId id;
		private string name;

		/// <summary>
		/// 既定のコンストラクタ
		/// </summary>
		public Market() {}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public Market(MarketId id, string name)
		{
			this.id = id;
			this.name = name;
		}

		/// <summary>
		/// 市場IDを取得する。
		/// </summary>
		public MarketId Id
		{
			get { return id; }
		}

		/// <summary>
		/// 市場名を取得あるいは変更する。
		/// </summary>
		public string Name
		{
			get { return name; }
			set { 
				MarketTable.ChangeMarketName(id, value);
				name = value;
			}
		}

		/// <summary>
		/// データベースのレコードをオブジェクトに変換する。
		/// </summary>
		/// <param name="values">ネイティブ形式の値の配列</param>
		/// <returns>変換後のMarketオブジェクトを返す。</returns>
		public Object ToObject(Object[] values)
		{
			return new Market((MarketId)values[0], // @Id
							  (string)values[1]); // @Name
		}
	}

	/// <summary>
	/// Marketテーブルを操作する。
	/// このクラスは最初にすべてのレコードを配列に読み込む。一部のメソッドが配列のインデックス+1がレコードの@Idと一致することを仮定している。Marketテーブルを不用意に編集するとこの仮定が崩れるので要注意。
	/// </summary>
	public class MarketTable
	{
		private static Connection conn;
		private static Connection Conn
		{
			get 
			{
				if (conn != null)
					return conn;
				conn = new Connection("protra");
				return conn;
			}
		}

		private static Market[] byId;

		/// <summary>
		/// すべてのレコードを保持する配列を取得する。
		/// </summary>
		private static Market[] ById
		{
			get {
				if (byId != null)
					return byId;
				string sql = "SELECT * FROM Market ORDER BY @Id";
				ArrayList list = Conn.Query(sql, new Market());
				byId = (Market[])list.ToArray(typeof(Market));
				return byId;
			}
		}

		/// <summary>
		/// 指定された市場IDのレコードを取得する。
		/// </summary>
		/// <param name="id">市場ID</param>
		/// <returns>Marketオブジェクトを返す。</returns>
		public static Market GetRecord(MarketId id) 
		{
			return ById[(int)id - 1];
		}

		/// <summary>
		/// すべてのレコードを取得する。
		/// </summary>
		/// <returns>Marketオブジェクトの配列を返す。</returns>
		public static Market[] GetAllRecords()
		{
			return ById;
		}

		/// <summary>
		/// 市場名を変更する。
		/// </summary>
		/// <param name="id">市場リストID</param>
		/// <param name="name">新しい名前</param>
		public static void ChangeMarketName(MarketId id, string name)
		{
			string sql = string.Format(
				"UPDATE Market SET @Name='{0}' WHERE @Id={1}", name, (int)id);
			Conn.Execute(sql);
		}

		/// <summary>
		/// 全データを削除する。
		/// </summary>
		public static void Delete()
		{
			string sql = "DELETE FROM Market";
			Conn.Execute(sql);
		}
	}
}
