//
//  SampleBuffer.m
//

#import "SampleBuffer.h"

@implementation SampleBuffer

- (id)initWithPT1:(PT1 *)pt1;
{
    if ((self = [super init]) != nil)
    {
        _pt1 = [pt1 retain];
        _transferLock = [[NSLock alloc] init];
        _transfer = NO;
        _stop = NO;
        _virtualIndex = 0;
        _imageIndex = 0;
        _blockIndex = 0;
        uint i;
        for (i = 0; i < 4; ++i)
        {
            _stream[i] = nil;
        }

        memset(_buffer, 0x00, kBlockSize);
    }
    return self;
}

- (void)dealloc
{
    [_pt1 release];
    [_transferLock release];
    uint i;
    for (i = 0; i < 4; ++i)
    {
        if (_stream[i] != nil)
        {
            [_stream[i] release];
        }
    }

    [super dealloc];
}

- (uint)offset:(uint)imageIndex blockIndex:(uint)blockIndex additionalOffset:(uint)additionalOffset
{
    blockIndex += kBlockCount * imageIndex;
    uint offset = (kBlockSize * blockIndex + additionalOffset) / sizeof(uint);
    return offset;
}

- (uint)offset:(uint)imageIndex blockIndex:(uint)blockIndex
{
    return [self offset:imageIndex blockIndex:blockIndex additionalOffset:(kBlockSize - sizeof(uint))];
}

- (uint)read:(uint)virtualIndex imageIndex:(uint)imageIndex blockIndex:(uint)blockIndex
{
    void *voidPtr;
    if ([_pt1 getBufferPtr:virtualIndex ptr:&voidPtr] != kStatusSuccess)
    {
        NSLog(@"getBufferPtr(read) ng");
        return 0;
    }

    volatile const uint *ptr = (volatile const uint *)voidPtr;
    if (ptr != NULL)
    {
        return ptr[[self offset:imageIndex blockIndex:blockIndex]];
    }
    else
    {
        NSLog(@"ptr is NULL");
    }
    return 0;
}

- (void)clearBlock:(uint)virtualIndex imageIndex:(uint)imageIndex blockIndex:(uint)blockIndex
{
    void *voidPtr = NULL;

    if ([_pt1 getBufferPtr:virtualIndex ptr:&voidPtr] != kStatusSuccess)
    {
        NSLog(@"getBufferPtr(clearBlock) ng.");
        return;
    }

    uint *ptr = (uint *)voidPtr;
    if (ptr != NULL)
    {
        ptr[[self offset:imageIndex blockIndex:blockIndex]] = 0;
    }
    else
    {
        NSLog(@"ptr is NULL");
    }
}

- (bool)waitBlock
{
    bool result = YES;

    while (YES)
    {
        if (_stop)
        {
            result = false;
            break;
        }

        // ブロックの末尾が 0 でなければ、そのブロックの DMA 転送が完了したことになる
        if ([self read:_virtualIndex imageIndex:_imageIndex blockIndex:_blockIndex] != 0)
        {
            break;
        }

        usleep(1000);
    }

    return result;
}

- (void)copyBlock
{
    void *voidPtr;
	if ([_pt1 getBufferPtr:_virtualIndex ptr:&voidPtr] != kStatusSuccess)
    {
        NSLog(@"getBufferPtr(copyBlock) ng.");
        return;
    }

    uint *srcPtr = (uint *)voidPtr;

    srcPtr += [self offset:_imageIndex blockIndex:_blockIndex additionalOffset:0];

    memcpy(_buffer, srcPtr, kBlockSize);

    // コピーし終わったので、ブロックの末尾を 0 にします。
    srcPtr[kBlockSize / sizeof(*srcPtr) - 1] = 0;

    if (kBlockCount <= ++_blockIndex)
    {
        _blockIndex = 0;

        // 転送カウンタは OS::Memory::PAGE_SIZE * PT::Device::BUFFER_PAGE_COUNT バイトごとにインクリメントします。
		if ([_pt1 incrementTransferCounter] != kStatusSuccess)
        {
            NSLog(@"incrementTransferCounter ng.");
            return;
        }

        if (kVirtualSize <= ++_imageIndex)
        {
            _imageIndex = 0;
            if (kVirtualCount <= ++_virtualIndex)
            {
                _virtualIndex = 0;
            }
        }
    }
}

- (bool)dispatchBlock
{
    const uint *ptr = (uint *)_buffer;

    uint i;
    for (i = 0; i < kBlockSize / sizeof(*ptr); ++i)
    {
        uint packet = ptr[i];

        uint packetId    = BIT_SHIFT_MASK(packet, 29, 3);
        uint packetError = BIT_SHIFT_MASK(packet, 24, 1);

        if (packetError)
        {
            // エラーの原因を調べる
            TransferInfo info;
            if ([_pt1 getTransferInfo:&info] != kStatusSuccess)
            {
                NSLog(@"getTransferInfo ng.");
                return false;
            }

            if (info.TransferCounter1)
            {
                NSLog(@"★転送カウンタが 1 以下になりました。");
            }
            else if (info.BufferOverflow)
            {
                NSLog(@"★PCI バスを長期に渡り確保できなかったため、ボード上の FIFO が溢れました。");
            }
            else
            {
                NSLog(@"★転送エラーが発生しました。");
            }
            NSLog(@"★ファイルへの書き出しを中止します。");
            return false;
        }

        if ((1 <= packetId) && (packetId <= 4))
        {
            [_stream[packetId - 1] addPacket:packet];
        }
    }

    return true;
}

- (bool)transfer:(bool)enable
{
    [_transferLock lock];

    if (_transfer && !enable)
    {
        _stop = YES;
    }

    bool result = NO;
    while (!_transfer && enable)
    {
        // ストリームを生成する
        uint i;
        for (i = 0; i < sizeof(_stream) / sizeof(*_stream); ++i)
        {
            if (_stream[i] == NULL)
            {
//                if (i == 0)
                if (NO)
                {
                    _stream[0] = [[SampleStream alloc] initWithTuner:(i / 2) isdb:(i % 2) fifo:YES];
                }
                else
                {
                    _stream[i] = [[SampleStream alloc] initWithTuner:(i / 2) isdb:(i % 2)];
                }
            }
        }

        // バッファが確保されていない場合は確保する
        BufferInfo bufferInfo;
        if ([_pt1 getBufferInfo:&bufferInfo] != kStatusSuccess)
        {
            NSLog(@"getBufferInfo ng.");
            break;
        }

        if (bufferInfo.VirtualSize == 0)
        {
            bufferInfo.VirtualSize  = kVirtualSize;
            bufferInfo.VirtualCount = kVirtualCount;
            bufferInfo.LockSize     = kLockSize;
            if ([_pt1 setBufferInfo:&bufferInfo] != kStatusSuccess)
            {
                NSLog(@"setBufferInfo ng.");
                break;
            }
        }

        // DMA 転送がどこまで進んだかを調べるため、各ブロックの末尾を 0 でクリアする
        for (i = 0; i < kVirtualCount; ++i)
        {
            uint j;
            for (j = 0; j < kVirtualSize; ++j)
            {
                uint k;
                for (k = 0; k < kBlockCount; ++k)
                {
                    [self clearBlock:i imageIndex:j blockIndex:k];
                }
            }
        }

        // 転送カウンタをリセットする
        if ([_pt1 resetTransferCounter] != kStatusSuccess)
        {
            NSLog(@"Device::ResetTransferCounter()");
            break;
        }

        // 転送カウンタをインクリメントする
        for (i = 0; i < kVirtualSize * kVirtualCount; ++i)
        {
            if ([_pt1 incrementTransferCounter] != kStatusSuccess)
            {
                NSLog(@"Device::IncrementTransferCounter()");
                break;
            }
        }

        // DMA 転送を許可する
        if ([_pt1 setTransferEnable:true] != kStatusSuccess)
        {
            NSLog(@"setTransferEnable ng.");
            break;
        }

        // スレッド開始
        _stop = NO;
        [NSThread detachNewThreadSelector:@selector(transferThread:) toTarget:self withObject:nil];

        result = YES;
        _transfer = YES;

        break;
    }

    if (_stop)
    {
        if ([_pt1 setTransferEnable:false] != kStatusSuccess)
        {
            NSLog(@"setTransferEnable(stop) ng.");
        }
    }

    [_transferLock unlock];

    return result;
}

- (void)transferThread:(id)anObject
{
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];

    _virtualIndex = 0;
    _imageIndex = 0;
    _blockIndex = 0;

    while (true)
    {
        if (![self waitBlock])
        {
            break;
        }

        [self copyBlock];

        if (![self dispatchBlock])
        {
            break;
        }
    }

//    NSLog(@"transferThread done.");

    [_transferLock lock];
    _transfer = NO;

    if ([_pt1 setTransferEnable:false] != kStatusSuccess)
    {
        NSLog(@"setTransferEnable(done) ng.");
    }

    [_transferLock unlock];

    [pool release];
}

@end
