/*
	File:			SimpleDriver.h
	
	Description:	This file shows how to implement a basic I/O Kit driver kernel extension (KEXT).

	Copyright:		Copyright © 2001-2008 Apple Inc. All rights reserved.
	
	Disclaimer:		IMPORTANT:  This Apple software is supplied to you by Apple Computer, Inc.
					("Apple") in consideration of your agreement to the following terms, and your
					use, installation, modification or redistribution of this Apple software
					constitutes acceptance of these terms.  If you do not agree with these terms,
					please do not use, install, modify or redistribute this Apple software.
					
					In consideration of your agreement to abide by the following terms, and subject
					to these terms, Apple grants you a personal, non-exclusive license, under Apple’s
					copyrights in this original Apple software (the "Apple Software"), to use,
					reproduce, modify and redistribute the Apple Software, with or without
					modifications, in source and/or binary forms; provided that if you redistribute
					the Apple Software in its entirety and without modifications, you must retain
					this notice and the following text and disclaimers in all such redistributions of
					the Apple Software.  Neither the name, trademarks, service marks or logos of
					Apple Computer, Inc. may be used to endorse or promote products derived from the
					Apple Software without specific prior written permission from Apple.  Except as
					expressly stated in this notice, no other rights or licenses, express or implied,
					are granted by Apple herein, including but not limited to any patent rights that
					may be infringed by your derivative works or by other works in which the Apple
					Software may be incorporated.
					
					The Apple Software is provided by Apple on an "AS IS" basis.  APPLE MAKES NO
					WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
					WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
					PURPOSE, REGARDING THE APPLE SOFTWARE OR ITS USE AND OPERATION ALONE OR IN
					COMBINATION WITH YOUR PRODUCTS.
					
					IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
					CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
					GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
					ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION AND/OR DISTRIBUTION
					OF THE APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF CONTRACT, TORT
					(INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF APPLE HAS BEEN
					ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
				
	Change History (most recent first):
	
            2.0			08/13/2008			Add Leopard user client API for supporting 64-bit user processes.
											Now requires Xcode 3.0 or later to build.
			
            1.1			05/22/2007			User client performs endian swapping when called from a user process 
											running using Rosetta. Updated to produce a universal binary.
											Now requires Xcode 2.2.1 or later to build.
			
			1.0d3	 	01/14/2003			New sample.

*/

// Set up a minimal set of availability macros.
//
// These macros are useful for conditionally compiling code targeting specific versions
// of Mac OS X. But the standard availability macros aren't available to kernel code
// prior to Mac OS X 10.5. Code targeting 10.5 and later should use AvailabilityMacros.h
// instead of this file. Code targeting 10.6 and later should use Availability.h

#ifndef MAC_OS_X_VERSION_MIN_REQUIRED

#ifndef MAC_OS_X_VERSION_10_4
#define MAC_OS_X_VERSION_10_4 1040
#endif

#ifdef __ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED__
#define MAC_OS_X_VERSION_MIN_REQUIRED __ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED__
#else
#define MAC_OS_X_VERSION_MIN_REQUIRED MAC_OS_X_VERSION_10_4
#endif

#endif

#include <IOKit/IOService.h>
#include <IOKit/IOBufferMemoryDescriptor.h>
#include <IOKit/pci/IOPCIDevice.h>

#if IOMEMORYDESCRIPTOR_SUPPORTS_DMACOMMAND
#include <IOKit/IODMACommand.h>
#endif

enum
{
    kPT1OffState = 0,	//off
    kPT1OnState = 1,	//on
    kNumPT1States = 2	//number of states
};

static const IOPMPowerState ourPowerStates[kNumPT1States] =
{
    {kIOPMPowerStateVersion1,0,0,0,0,0,0,0,0,0,0,0},
    {kIOPMPowerStateVersion1,kIOPMDeviceUsable,IOPMPowerOn,IOPMPowerOn,0,0,0,0,0,0,0,0}
};

#include "PT1.h"

#define kPageSize           4096
#define kBufferPageCount    511
#define kBlockCount         2

class PT1Device : public IOService
{
	// Declare the metaclass information that is used for runtime type checking of I/O Kit objects.
	// Note that the class name is different when targeting Mac OS X 10.4 because support for that
	// version has to be built as a separate kext. This is because the KPIs for 64-bit user processes
	// to access user clients only exist on Mac OS X 10.5 and later.
	
#if MAC_OS_X_VERSION_MIN_REQUIRED <= MAC_OS_X_VERSION_10_4
	OSDeclareDefaultStructors(PT1Device)
#else
	OSDeclareDefaultStructors(PT1Device)
#endif

private:
    IOPCIDevice *_device;
    IOMemoryDescriptor * fLowMemory;
    IOMemoryMap *_map;
    void *_reg;

    uint16_t        _deviceID;
    LnbPower        _lnbPower;
    LnbPower        _lnbPowerWhenClose;
    TunerPowerReset _tunerPowerReset;

    bool            _streamEnable[4];
    uint32_t        _streamGray[4];
    bool            _transfer;

    // DMAバッファ関連
    BufferInfo                  _bufferInfo;
    IOBufferMemoryDescriptor *  _bmdAddress;
    IOBufferMemoryDescriptor *  _bmdData;
    IODMACommand *              _dmaCmdAddress;
    IODMACommand *              _dmaCmdData;

    // I/O用メソッド
    UInt8 memoryRead8(UInt16 offset);
    UInt16 memoryRead16(UInt16 offset);
    UInt32 memoryRead32(UInt16 offset);
    void memoryWrite8(UInt16 offset, UInt8 value);
    void memoryWrite16(UInt16 offset, UInt16 value);
    void memoryWrite32(UInt16 offset, UInt32 value);

    void clearBuffer(void);

protected:

public:
	// IOService methods
	virtual bool init(OSDictionary* dictionary = 0);
	virtual void free(void);
	
    virtual bool open(IOService *forClient, IOOptionBits options = 0, void *arg = 0);
    virtual void close(IOService *forClient, IOOptionBits options = 0);
	virtual IOService* probe(IOService* provider, SInt32* score);
	
	virtual bool start(IOService* provider);
	virtual void stop(IOService* provider);
	
	virtual bool willTerminate(IOService* provider, IOOptionBits options);
	virtual bool didTerminate(IOService* provider, IOOptionBits options, bool* defer);
	
    virtual bool terminate(IOOptionBits options = 0);
    virtual bool finalize(IOOptionBits options);

	virtual IOReturn clientMemoryForType(UInt32 type, IOOptionBits * options, IOMemoryDescriptor ** memory);

    virtual IOReturn setPowerState(unsigned long, IOService *);

    /* Other methods */
    IOMemoryDescriptor * copyGlobalMemory( void );
    IOReturn generateDMAAddresses( IOMemoryDescriptor * memDesc );


    bool init_XC3S(void);

    uint32_t setFrequency_ISDB_S(uint32_t tuner, ISDB isdb, uint32_t channel, int32_t offset);
    uint32_t setFrequency_ISDB_T(uint32_t tuner, ISDB isdb, uint32_t channel, int32_t offset);

	// Driver Interface
	virtual IOReturn getDeviceInfo(DeviceInfo *deviceInfo, uint32_t *structSize);
    virtual IOReturn getPciClockCounter(uint32_t *status, uint32_t *counter);
    virtual IOReturn setPciLatencyTimer(uint32_t *status, uint32_t latencyTimer);
    virtual IOReturn getPciLatencyTimer(uint32_t *status, uint32_t *latencyTimer);
	virtual IOReturn setLnbPower(uint32_t *status, uint32_t lnbPower);
	virtual IOReturn getLnbPower(uint32_t *status, uint32_t *lnbPower);
	virtual IOReturn setLnbPowerWhenClose(uint32_t *status, uint32_t lnbPower);
	virtual IOReturn getLnbPowerWhenClose(uint32_t *status, uint32_t *lnbPower);
    virtual IOReturn setTunerPowerReset(uint32_t *status, uint32_t tunerPowerReset);
    virtual IOReturn getTunerPowerReset(uint32_t *status, uint32_t *tunerPowerReset);
    virtual IOReturn initTuner(uint32_t *status, uint32_t tuner);
    virtual IOReturn setTunerSleep(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t sleep);
    virtual IOReturn getTunerSleep(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *sleep);
    virtual IOReturn setFrequency(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t channel, int32_t offset);
    virtual IOReturn getFrequency(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *channel, int32_t *offset);
    virtual IOReturn getFrequencyOffset(uint32_t *status, uint32_t tuner, ISDB isdb, int32_t *clock, int32_t *carrier);
    virtual IOReturn getCnAgc(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *cn100, uint32_t *currentAgc, uint32_t *maxAgc);
    virtual IOReturn setIdS(uint32_t *status, uint32_t tuner, uint32_t tsid);
    virtual IOReturn getIdS(uint32_t *status, uint32_t tuner, uint32_t *tsid);
    virtual IOReturn getCorrectedErrorRate(uint32_t *status, uint32_t tuner, ISDB isdb, LayerIndex layerIndex, ErrorRate *errorRate);
    virtual IOReturn resetCorrectedErrorCount(uint32_t *status, uint32_t tuner, ISDB isdb);
    virtual IOReturn getErrorCount(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *count);
    virtual IOReturn getTmccS(uint32_t *status, uint32_t tuner, TmccS *tmcc);
    virtual IOReturn getLayerS(uint32_t *status, uint32_t tuner, LayerS *layer);
    virtual IOReturn getTmccT(uint32_t *status, uint32_t tuner, TmccT *tmcc);
    virtual IOReturn getLockedT(uint32_t *status, uint32_t tuner, bool locked[LAYER_COUNT_T]);
    virtual IOReturn setLayerEnable(uint32_t *status, uint32_t tuner, ISDB isdb, LayerMask layerMask);
    virtual IOReturn getLayerEnable(uint32_t *status, uint32_t tuner, ISDB isdb, LayerMask *layerMask);
    virtual IOReturn setBufferInfo(uint32_t *status, const BufferInfo *bufferInfo);
    virtual IOReturn getBufferInfo(uint32_t *status, BufferInfo *bufferInfo);
    virtual IOReturn resetTransferCounter(uint32_t *status);
    virtual IOReturn incrementTransferCounter(uint32_t *status);
    virtual IOReturn setStreamEnable(uint32_t *status, uint32_t tuner, ISDB isdb, bool enable);
    virtual IOReturn getStreamEnable(uint32_t *status, uint32_t tuner, ISDB isdb, bool *enable);
    virtual IOReturn setStreamGray(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t gray);
    virtual IOReturn getStreamGray(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *gray);
    virtual IOReturn setTransferEnable(uint32_t *status, bool enable);
    virtual IOReturn getTransferEnable(uint32_t *status, bool *enable);
    virtual IOReturn getTransferInfo(uint32_t *status, TransferInfo *transferInfo);
};