//
//  PT1Client.cpp
//

#include <IOKit/IOLib.h>
#include <IOKit/IOKitKeys.h>
#include <libkern/OSByteOrder.h>
#include "PT1Client.h"

#define super IOUserClient

// Even though we are defining the convenience macro super for the superclass, you must use the actual class name
// in the OS*MetaClass macros. Note that the class name is different when supporting Mac OS X 10.4.

#if MAC_OS_X_VERSION_MIN_REQUIRED <= MAC_OS_X_VERSION_10_4
OSDefineMetaClassAndStructors(PT1Client, IOUserClient)
#else
OSDefineMetaClassAndStructors(PT1Client, IOUserClient)
#endif

#if MAC_OS_X_VERSION_MIN_REQUIRED <= MAC_OS_X_VERSION_10_4
// Sentinel values for the method dispatch table
enum {
    kMethodObjectThis = 0,
    kMethodObjectProvider
};


// User client method dispatch table.
//
// The user client mechanism is designed to allow calls from a user process to be dispatched to
// any IOService-based object in the kernel. Almost always this mechanism is used to dispatch calls to
// either member functions of the user client itself or of the user client's provider. The provider is
//  the driver which the user client is connecting to the user process.
//
// While this sample shows one case of dispatching calls directly to the driver (ScalarIScalarO),
// it is recommended that calls be dispatched to the user client. This allows the user client to perform
// error checking on the parameters before passing them to the driver. It also allows the user client to
// do any endian-swapping of parameters in the cross-endian case. (See ScalarIStructI below for further
// discussion of this subject.)
//
// The dispatch table makes use of the sentinel values kMethodObjectThis and kMethodObjectProvider to
// represent at compile time the values of the this pointer and fProvider respectively at run time.  
const IOExternalMethod PT1Client::sMethods[kNumberOfMethods] = {
	{   // kMethodOpen
		(IOService *) kMethodObjectThis,									// Target object is this user client.
		(IOMethod) &PT1Client::openUserClient,				// Method pointer.
		kIOUCScalarIScalarO,												// Scalar Input, Scalar Output.
		0,																	// No scalar input values.
		0																	// No scalar output values.
	},
	{   // kMethodClose
		(IOService *) kMethodObjectThis,									// Target object is this user client.
		(IOMethod) &PT1Client::closeUserClient,				// Method pointer.
		kIOUCScalarIScalarO,												// Scalar Input, Scalar Output.
		0,																	// No scalar input values.
		0																	// No scalar output values.
	},
	{   // kMyScalarIStructIMethod
		(IOService *) kMethodObjectThis,									// Target object is this user client.
		(IOMethod) &PT1Client::ScalarIStructI,				// Method pointer.
		kIOUCScalarIStructI,												// Scalar Input, Struct Input.
		1,																	// One scalar input value.
		sizeof(MySampleStruct)												// The size of the input struct.
	},
	{   // kMyScalarIStructOMethod
		(IOService *) kMethodObjectThis,									// Target object is this user client.
		(IOMethod) &PT1Client::ScalarIStructO,				// Method pointer.
		kIOUCScalarIStructO,												// Scalar Input, Struct Output.
		2,																	// Two scalar input values.
		sizeof(MySampleStruct)												// The size of the output struct.
	},
	{   // kMyScalarIScalarOMethod
		(IOService *) kMethodObjectProvider,								// Target object is this user client's provider
																			// (the driver).
		(IOMethod) &PT1Device::ScalarIScalarO,					// Method pointer.
		kIOUCScalarIScalarO,												// Scalar Input, Scalar Output.
		2,																	// Two scalar input values.
		1																	// One scalar output value.
	},
	{   // kMyStructIStructOMethod
		(IOService *) kMethodObjectThis,									// Target object is this user client.
		(IOMethod) &PT1Client::StructIStructO,				// Method pointer.
		kIOUCStructIStructO,												// Struct Input, Struct Output.
		sizeof(MySampleStruct),												// The size of the input struct.
		sizeof(MySampleStruct)												// The size of the output struct.
	}
};
    
// Look up the external methods - supply a description of the parameters 
// available to be called.
//
// This is the legacy approach which only supports 32-bit user processes.
IOExternalMethod* PT1Client::getTargetAndMethodForIndex(IOService** target, UInt32 index)
{
	IOLog("%s[%p]::%s(%p, %ld)\n", getName(), this, __FUNCTION__, target, index);
    
    // Make sure that the index of the function we're calling actually exists in the function table.
    if (index < (UInt32) kNumberOfMethods) {
		if (sMethods[index].object == (IOService *) kMethodObjectThis) {
			*target = this;	   
        }
		else {
			*target = fProvider;	   
		}
		return (IOExternalMethod *) &sMethods[index];
    }
	else {
		*target = NULL;
		return NULL;
	}
}
#else
// This is the technique which supports both 32-bit and 64-bit user processes starting with Mac OS X 10.5.
//
// User client method dispatch table.
//
// The user client mechanism is designed to allow calls from a user process to be dispatched to
// any IOService-based object in the kernel. Almost always this mechanism is used to dispatch calls to
// either member functions of the user client itself or of the user client's provider. The provider is
// the driver which the user client is connecting to the user process.
//
// It is recommended that calls be dispatched to the user client and not directly to the provider driver.
// This allows the user client to perform error checking on the parameters before passing them to the driver.
// It also allows the user client to do any endian-swapping of parameters in the cross-endian case.
// (See ScalarIStructI below for further discussion of this subject.)

const IOExternalMethodDispatch PT1Client::sMethods[kNumberOfMethods] = {
        // Method pointer                                           IN Scalar   IN Struct   OUT Scalar  OUT Struct
    {   (IOExternalMethodAction)&PT1Client::sGetDeviceInfo,             0,          0,          0,          sizeof(DeviceInfo)  },
    {   (IOExternalMethodAction)&PT1Client::sOpen,                      0,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sClose,                     0,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetPciClockCounter,        0,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetPciLatencyTimer,        1,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetPciLatencyTimer,        0,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetLnbPower,               1,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetLnbPower,               0,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetLnbPowerWhenClose,      1,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetLnbPowerWhenClose,      0,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetTunerPowerReset,        1,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetTunerPowerReset,        0,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sInitTuner,                 1,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetTunerSleep,             3,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetTunerSleep,             2,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetFrequency,              4,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetFrequency,              2,          0,          3,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetFrequencyOffset,        2,          0,          3,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetCnAgc,                  2,          0,          4,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetIdS,                    2,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetIdS,                    1,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetCorrectedErrorRate,     3,          0,          1,          sizeof(ErrorRate)   },
    {   (IOExternalMethodAction)&PT1Client::sResetCorrectedErrorCount,  2,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetErrorCount,             2,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetTmccS,                  1,          0,          1,          sizeof(TmccS)       },
    {   (IOExternalMethodAction)&PT1Client::sGetLayerS,                 1,          0,          1,          sizeof(LayerS)      },
    {   (IOExternalMethodAction)&PT1Client::sGetTmccT,                  1,          0,          1,          sizeof(TmccT)       },
    {   (IOExternalMethodAction)&PT1Client::sGetLockedT,                1,          0,          1,          sizeof(bool[LAYER_COUNT_T])},
    {   (IOExternalMethodAction)&PT1Client::sSetLayerEnable,            3,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetLayerEnable,            2,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetBufferInfo,             0,          sizeof(BufferInfo), 1,  0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetBufferInfo,             0,          0,          1,          sizeof(BufferInfo)  },
    {   (IOExternalMethodAction)&PT1Client::sResetTransferCounter,      0,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sIncrementTransferCounter,  0,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetStreamEnable,           3,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetStreamEnable,           2,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetStreamGray,             3,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetStreamGray,             2,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sSetTransferEnable,         1,          0,          1,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetTransferEnable,         0,          0,          2,          0                   },
    {   (IOExternalMethodAction)&PT1Client::sGetTransferInfo,           0,          0,          1,          sizeof(TransferInfo)},
};

IOReturn PT1Client::externalMethod(uint32_t selector, IOExternalMethodArguments* arguments,
												   IOExternalMethodDispatch* dispatch, OSObject* target, void* reference)

{
/*
	IOLog("%s[%p]::%s(%d, %p, %p, %p, %p)\n", getName(), this, __FUNCTION__,
		  selector, arguments, dispatch, target, reference);
*/
    if (selector < (uint32_t) kNumberOfMethods) {
        dispatch = (IOExternalMethodDispatch *) &sMethods[selector];
        
        if (!target) {
/*
            if (selector == kMyScalarIScalarOMethod) {
				target = fProvider;
			}
			else {
*/
				target = this;
/*
			}
*/
		}
    }
        
	return super::externalMethod(selector, arguments, dispatch, target, reference);
}
#endif


// There are two forms of IOUserClient::initWithTask, the second of which accepts an additional OSDictionary* parameter.
// If your user client needs to modify its behavior when it's being used by a process running using Rosetta,
// you need to implement the form of initWithTask with this additional parameter.
//
// initWithTask is called as a result of the user process calling IOServiceOpen.
bool PT1Client::initWithTask(task_t owningTask, void* securityToken, UInt32 type, OSDictionary* properties)
{
    bool	success;
    
	success = super::initWithTask(owningTask, securityToken, type, properties);	    
	
	// This IOLog must follow super::initWithTask because getName relies on the superclass initialization.
    IOLog("%s[%p]::%s(%p, %p, %ld, %p)\n", getName(), this, __FUNCTION__, owningTask, securityToken, (long int)type, properties);

	if (success) {
		// This code will do the right thing on both PowerPC- and Intel-based systems because the cross-endian
		// property will never be set on PowerPC-based Macs. 
		fCrossEndian = false;
	
		if (properties != NULL && properties->getObject(kIOUserClientCrossEndianKey)) {
			// A connection to this user client is being opened by a user process running using Rosetta.
			
			// Indicate that this user client can handle being called from cross-endian user processes by 
			// setting its IOUserClientCrossEndianCompatible property in the I/O Registry.
			if (setProperty(kIOUserClientCrossEndianCompatibleKey, kOSBooleanTrue)) {
				fCrossEndian = true;
				IOLog("%s[%p]::%s(): fCrossEndian = true\n", getName(), this, __FUNCTION__);
			}
		}
	}
	
    fTask = owningTask;
    fProvider = NULL;

    return success;
}


// start is called after initWithTask as a result of the user process calling IOServiceOpen.
bool PT1Client::start(IOService* provider)
{
    bool	success;
	
	IOLog("%s[%p]::%s(%p)\n", getName(), this, __FUNCTION__, provider);
    
    // Verify that this user client is being started with a provider that it knows
	// how to communicate with.
	fProvider = OSDynamicCast(PT1Device, provider);
    success = (fProvider != NULL);
    
    if (success) {
		// It's important not to call super::start if some previous condition
		// (like an invalid provider) would cause this function to return false. 
		// I/O Kit won't call stop on an object if its start function returned false.
		success = super::start(provider);
	}
	
    return success;
}


// We override stop only to log that it has been called to make it easier to follow the user client's lifecycle.
void PT1Client::stop(IOService* provider)
{
	IOLog("%s[%p]::%s(%p)\n", getName(), this, __FUNCTION__, provider);
    
    super::stop(provider);
}


// clientClose is called as a result of the user process calling IOServiceClose.
IOReturn PT1Client::clientClose(void)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);
    
    // Defensive coding in case the user process called IOServiceClose
	// without calling closeUserClient first.
    uint32_t status;
    (void) close(&status);
    
	// Inform the user process that this user client is no longer available. This will also cause the
	// user client instance to be destroyed.
	//
	// terminate would return false if the user process still had this user client open.
	// This should never happen in our case because this code path is only reached if the user process
	// explicitly requests closing the connection to the user client.
	bool success = terminate();
	if (!success) {
		IOLog("%s[%p]::%s(): terminate() failed.\n", getName(), this, __FUNCTION__);
	}

    // DON'T call super::clientClose, which just returns kIOReturnUnsupported.
    
    return kIOReturnSuccess;
}


// clientDied is called if the client user process terminates unexpectedly (crashes).
// We override clientDied only to log that it has been called to make it easier to follow the user client's lifecycle.
// Production user clients need to override clientDied only if they need to take some alternate action if the user process
// crashes instead of exiting normally.
IOReturn PT1Client::clientDied(void)
{
    IOReturn result = kIOReturnSuccess;

	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    // The default implementation of clientDied just calls clientClose.
    result = super::clientDied();

    return result;
}


// willTerminate is called at the beginning of the termination process. It is a notification
// that a provider has been terminated, sent before recursing up the stack, in root-to-leaf order.
//
// This is where any pending I/O should be terminated. At this point the user client has been marked
// inactive and any further requests from the user process should be returned with an error.
bool PT1Client::willTerminate(IOService* provider, IOOptionBits options)
{
	IOLog("%s[%p]::%s(%p, %ld)\n", getName(), this, __FUNCTION__, provider, (long int)options);
	
	return super::willTerminate(provider, options);
}


// didTerminate is called at the end of the termination process. It is a notification
// that a provider has been terminated, sent after recursing up the stack, in leaf-to-root order.
bool PT1Client::didTerminate(IOService* provider, IOOptionBits options, bool* defer)
{
	IOLog("%s[%p]::%s(%p, %ld, %p)\n", getName(), this, __FUNCTION__, provider, (long int)options, defer);
	
	// If all pending I/O has been terminated, close our provider. If I/O is still outstanding, set defer to true
	// and the user client will not have stop called on it.
    uint32_t status;
	close(&status);
	*defer = false;
	
	return super::didTerminate(provider, options, defer);
}


// We override terminate only to log that it has been called to make it easier to follow the user client's lifecycle.
// Production user clients will rarely need to override terminate. Termination processing should be done in
// willTerminate or didTerminate instead.
bool PT1Client::terminate(IOOptionBits options)
{
    bool	success;
    
	IOLog("%s[%p]::%s(%ld)\n", getName(), this, __FUNCTION__, (long int)options);

    success = super::terminate(options);
    
    return success;
}


// We override finalize only to log that it has been called to make it easier to follow the user client's lifecycle.
// Production user clients will rarely need to override finalize.
bool PT1Client::finalize(IOOptionBits options)
{
    bool	success;
    
	IOLog("%s[%p]::%s(%ld)\n", getName(), this, __FUNCTION__, (long int)options);
    
    success = super::finalize(options);
    
    return success;
}


IOReturn PT1Client::clientMemoryForType(UInt32 type, IOOptionBits * options, IOMemoryDescriptor ** memory)
{
    IOReturn result = kIOReturnNoMemory;

	IOLog("%s[%p]::%s(type = %ld)\n", getName(), this, __FUNCTION__, (long int)type);

    if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		result = fProvider->clientMemoryForType(type, options, memory);
	}
    
    return result;
}


#pragma mark
#pragma mark ############### User Interface ###############

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetDeviceInfo(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getDeviceInfo((DeviceInfo *)arguments->structureOutput,
                                 (uint32_t *)&arguments->structureOutputSize);
}
#endif

IOReturn PT1Client::getDeviceInfo(DeviceInfo *deviceInfo, uint32_t *structSize)
{
	IOReturn	result;

//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);
    
	// Endian-swap structure parameters in the user client before passing them to the driver.
	//
	// This may require adding new functions to your user client and modifying the dispatch table in
	// getTargetAndMethodForIndex to point to these new functions.
	//
	// This approach is greatly preferable because it avoids the complexity of a driver which can be opened by multiple clients,
	// each of which may or may not be cross-endian. It also avoids having to change the driver to make it cross-endian-aware.
	//
	// Note that fCrossEndian will always be false if running on a PowerPC-based Mac.
	
	if (fProvider == NULL || isInactive()) {
		// Return an error if we don't have a provider. This could happen if the user process
		// called ScalarIStructI without calling IOServiceOpen first. Or, the user client could be
		// in the process of being terminated and is thus inactive.
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this)) {
		// Return an error if we do not have the driver open. This could happen if the user process
		// did not call openUserClient before calling this function.
		result = kIOReturnNotOpen;
	}
	else {
		result = fProvider->getDeviceInfo(deviceInfo, structSize);

		if (fCrossEndian) {
            if (result == kIOReturnSuccess)
            {
                deviceInfo->Bus = OSSwapInt32(deviceInfo->Bus);
                deviceInfo->Slot = OSSwapInt32(deviceInfo->Slot);
                deviceInfo->Function = OSSwapInt32(deviceInfo->Function);
                deviceInfo->BadBitCount = OSSwapInt32(deviceInfo->BadBitCount);
            }
		}
	}
	
	return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sOpen(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->open((uint32_t *)&arguments->scalarOutput[0]);
}
#endif

IOReturn PT1Client::open(uint32_t *status)
{
    IOReturn	result = kIOReturnSuccess;
	
//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);
    
    if (fProvider == NULL || isInactive())
    {
        result = kIOReturnNotAttached;
	}
    else if (!fProvider->open(this, 0, status))
    {
		result = kIOReturnExclusiveAccess;
	}
        
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sClose(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->close((uint32_t *)&arguments->scalarOutput[0]);
}
#endif

IOReturn PT1Client::close(uint32_t *status)
{
    IOReturn	result = kIOReturnSuccess;
	
//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);
            
    if (fProvider == NULL)
    {
		result = kIOReturnNotAttached;
		IOLog("%s[%p]::%s(): returning kIOReturnNotAttached.\n", getName(), this, __FUNCTION__);
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
		IOLog("%s[%p]::%s(): returning kIOReturnNotOpen.\n", getName(), this, __FUNCTION__);
    }
    else
    {
		fProvider->close(this);
        *status = kStatusSuccess;
	}
	
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetPciClockCounter(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getPciClockCounter((uint32_t *)&arguments->scalarOutput[0],
                                      (uint32_t *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getPciClockCounter(uint32_t *status, uint32_t *counter)
{
    IOReturn	result = kIOReturnSuccess;
	
//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);
    
    if (fProvider == NULL || isInactive())
    {
        result = kIOReturnNotAttached;
	}
    else if (!fProvider->isOpen(this))
    {
        result = kIOReturnNotOpen;
	}
	else
    {
        result = fProvider->getPciClockCounter(status, counter);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *counter = OSSwapInt32(*counter);

			IOLog("%s[%p]::%s(output after swap: *status = %d)\n", getName(), this, __FUNCTION__, *status);
			IOLog("%s[%p]::%s(output after swap: *counter = %d)\n", getName(), this, __FUNCTION__, *counter);
		}
	}

    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetPciLatencyTimer(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setPciLatencyTimer((uint32_t *)&arguments->scalarOutput[0],
                                      (uint32_t)arguments->scalarInput[0]);
}
#endif

IOReturn PT1Client::setPciLatencyTimer(uint32_t *status, uint32_t latencyTimer)
{
    IOReturn	result = kIOReturnSuccess;
	
//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);
    
    if (fProvider == NULL || isInactive())
    {
        result = kIOReturnNotAttached;
	}
    else if (!fProvider->isOpen(this))
    {
        result = kIOReturnNotOpen;
	}
	else
    {
        if (fCrossEndian)
        {
            latencyTimer = OSSwapInt32(latencyTimer);
        }

        result = fProvider->setPciLatencyTimer(status, latencyTimer);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);

			IOLog("%s[%p]::%s(output after swap: *status = %d)\n", getName(), this, __FUNCTION__, *status);
		}
	}

    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetPciLatencyTimer(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getPciLatencyTimer((uint32_t *)&arguments->scalarOutput[0],
                                      (uint32_t *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getPciLatencyTimer(uint32_t *status, uint32_t *latencyTimer)
{
    IOReturn	result = kIOReturnSuccess;
	
//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);
    
    if (fProvider == NULL || isInactive())
    {
        result = kIOReturnNotAttached;
	}
    else if (!fProvider->isOpen(this))
    {
        result = kIOReturnNotOpen;
	}
	else
    {
        result = fProvider->getPciLatencyTimer(status, latencyTimer);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *latencyTimer = OSSwapInt32(*latencyTimer);

			IOLog("%s[%p]::%s(output after swap: *status = %d)\n", getName(), this, __FUNCTION__, *status);
			IOLog("%s[%p]::%s(output after swap: *counter = %d)\n", getName(), this, __FUNCTION__, *latencyTimer);
		}
	}

    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetLnbPower(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setLnbPower((uint32_t *)&arguments->scalarOutput[0],
                               (uint32_t)arguments->scalarInput[0]);
}
#endif

IOReturn PT1Client::setLnbPower(uint32_t *status, uint32_t lnbPower)
{
	IOReturn	result;

//  IOLog("%s[%p]::%s(lnbPower = %d)\n", getName(), this, __FUNCTION__, lnbPower);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			lnbPower = OSSwapInt32(lnbPower);
			
			IOLog("%s[%p]::%s(input after swap: lnbPower = %d)\n", getName(), this, __FUNCTION__, lnbPower);
		}
		
		result = fProvider->setLnbPower(status, lnbPower);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetLnbPower(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getLnbPower((uint32_t *)&arguments->scalarOutput[0],
                               (uint32_t *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getLnbPower(uint32_t *status, uint32_t *lnbPower)
{
	IOReturn	result;

//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
        result = fProvider->getLnbPower(status, lnbPower);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *lnbPower = OSSwapInt32(*lnbPower);

			IOLog("%s[%p]::%s(output after swap: *lnbPower = %d)\n", getName(), this, __FUNCTION__, *lnbPower);
		}
	}

	return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetLnbPowerWhenClose(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setLnbPowerWhenClose((uint32_t *)&arguments->scalarOutput[0],
                                        (uint32_t)arguments->scalarInput[0]);
}
#endif

IOReturn PT1Client::setLnbPowerWhenClose(uint32_t *status, uint32_t lnbPower)
{
	IOReturn	result;

//  IOLog("%s[%p]::%s(lnbPower = %d)\n", getName(), this, __FUNCTION__, lnbPower);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			lnbPower = OSSwapInt32(lnbPower);
			
			IOLog("%s[%p]::%s(input after swap: lnbPower = %d)\n", getName(), this, __FUNCTION__, lnbPower);
		}
		
		result = fProvider->setLnbPowerWhenClose(status, lnbPower);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetLnbPowerWhenClose(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getLnbPowerWhenClose((uint32_t *)&arguments->scalarOutput[0],
                                        (uint32_t *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getLnbPowerWhenClose(uint32_t *status, uint32_t *lnbPower)
{
	IOReturn	result;

//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
        result = fProvider->getLnbPowerWhenClose(status, lnbPower);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *lnbPower = OSSwapInt32(*lnbPower);

			IOLog("%s[%p]::%s(output after swap: *lnbPower = %d)\n", getName(), this, __FUNCTION__, *lnbPower);
		}
	}

	return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetTunerPowerReset(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setTunerPowerReset((uint32_t *)&arguments->scalarOutput[0],
                                      (uint32_t)arguments->scalarInput[0]);
}
#endif
IOReturn PT1Client::setTunerPowerReset(uint32_t *status, uint32_t tunerPowerReset)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tunerPowerReset = %d)\n", getName(), this, __FUNCTION__, tunerPowerReset);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tunerPowerReset = OSSwapInt32(tunerPowerReset);
			
			IOLog("%s[%p]::%s(input after swap: tunerPowerReset = %d)\n", getName(), this, __FUNCTION__, tunerPowerReset);
		}
		
		result = fProvider->setTunerPowerReset(status, tunerPowerReset);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetTunerPowerReset(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getTunerPowerReset((uint32_t *)&arguments->scalarOutput[0],
                                        (uint32_t *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getTunerPowerReset(uint32_t *status, uint32_t *tunerPowerReset)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
        result = fProvider->getTunerPowerReset(status, tunerPowerReset);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *tunerPowerReset = OSSwapInt32(*tunerPowerReset);

			IOLog("%s[%p]::%s(output after swap: *tunerPowerReset = %d)\n", getName(), this, __FUNCTION__, *tunerPowerReset);
		}
	}

	return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sInitTuner(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->initTuner((uint32_t *)&arguments->scalarOutput[0],
                             (uint32_t)arguments->scalarInput[0]);
}
#endif

IOReturn PT1Client::initTuner(uint32_t *status, uint32_t tuner)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d)\n", getName(), this, __FUNCTION__, tuner);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d)\n", getName(), this, __FUNCTION__, tuner);
		}
		
		result = fProvider->initTuner(status, tuner);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetTunerSleep(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setTunerSleep((uint32_t *)&arguments->scalarOutput[0],
                                 (uint32_t)arguments->scalarInput[0],
                                 (ISDB)arguments->scalarInput[1],
                                 (uint32_t)arguments->scalarInput[2]);
}
#endif

IOReturn PT1Client::setTunerSleep(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t sleep)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d, sleep = %d)\n", getName(), this, __FUNCTION__, tuner, isdb, sleep);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
            sleep = OSSwapInt32(sleep);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d, sleep = %d)\n",
                    getName(), this, __FUNCTION__, tuner, isdb, sleep);
		}
		
		result = fProvider->setTunerSleep(status, tuner, isdb, sleep);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetTunerSleep(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getTunerSleep((uint32_t *)&arguments->scalarOutput[0],
                                 (uint32_t)arguments->scalarInput[0],
                                 (ISDB)arguments->scalarInput[1],
                                 (uint32_t *)&arguments->scalarOutput[1]);
}
#endif
IOReturn PT1Client::getTunerSleep(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *sleep)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);
		}
		
		result = fProvider->getTunerSleep(status, tuner, isdb, sleep);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *sleep = OSSwapInt32(*sleep);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetFrequency(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setFrequency((uint32_t *)&arguments->scalarOutput[0],
                                (uint32_t)arguments->scalarInput[0],
                                (ISDB)arguments->scalarInput[1],
                                (uint32_t)arguments->scalarInput[2],
                                (int32_t)arguments->scalarInput[3]);
}
#endif
IOReturn PT1Client::setFrequency(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t channel, int32_t offset)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d, channel = %d, offset = %d)\n", getName(), this, __FUNCTION__, tuner, isdb, channel, offset);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
            channel = OSSwapInt32(channel);
            offset = OSSwapInt32(offset);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d, channel = %d, offset = %d)\n",
                    getName(), this, __FUNCTION__, tuner, isdb, channel, offset);
		}
		
		result = fProvider->setFrequency(status, tuner, isdb, channel, offset);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetFrequency(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getFrequency((uint32_t *)&arguments->scalarOutput[0],
                                (uint32_t)arguments->scalarInput[0],
                                (ISDB)arguments->scalarInput[1],
                                (uint32_t *)&arguments->scalarOutput[1],
                                (int32_t *)&arguments->scalarOutput[2]);
}
#endif
IOReturn PT1Client::getFrequency(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *channel, int32_t *offset)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);
		}
		
		result = fProvider->getFrequency(status, tuner, isdb, channel, offset);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *channel = OSSwapInt32(*channel);
            *offset = OSSwapInt32(*offset);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetFrequencyOffset(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getFrequencyOffset((uint32_t *)&arguments->scalarOutput[0],
                                      (uint32_t)arguments->scalarInput[0],
                                      (ISDB)arguments->scalarInput[1],
                                      (int32_t *)&arguments->scalarOutput[1],
                                      (int32_t *)&arguments->scalarOutput[2]);
}
#endif

IOReturn PT1Client::getFrequencyOffset(uint32_t *status, uint32_t tuner, ISDB isdb, int32_t *clock, int32_t *carrier)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);

			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);
		}
		
		result = fProvider->getFrequencyOffset(status, tuner, isdb, clock, carrier);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *clock = OSSwapInt32(*clock);
            *carrier = OSSwapInt32(*carrier);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetCnAgc(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getCnAgc((uint32_t *)&arguments->scalarOutput[0],
                            (uint32_t)arguments->scalarInput[0],
                            (ISDB)arguments->scalarInput[1],
                            (uint32_t *)&arguments->scalarOutput[1],
                            (uint32_t *)&arguments->scalarOutput[2],
                            (uint32_t *)&arguments->scalarOutput[3]);
}
#endif

IOReturn PT1Client::getCnAgc(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *cn100, uint32_t *currentAgc, uint32_t *maxAgc)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);

			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);
		}
		
		result = fProvider->getCnAgc(status, tuner, isdb, cn100, currentAgc, maxAgc);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetIdS(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setIdS((uint32_t *)&arguments->scalarOutput[0],
                          (uint32_t)arguments->scalarInput[0],
                          (uint32_t)arguments->scalarInput[1]);
}
#endif

IOReturn PT1Client::setIdS(uint32_t *status, uint32_t tuner, uint32_t tsid)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, tsid = %d)\n", getName(), this, __FUNCTION__, tuner, tsid);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            tsid = OSSwapInt32(tsid);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, tsid = %d)\n", getName(), this, __FUNCTION__, tuner, tsid);
		}
		
		result = fProvider->setIdS(status, tuner, tsid);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetIdS(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getIdS((uint32_t *)&arguments->scalarOutput[0],
                          (uint32_t)arguments->scalarInput[0],
                          (uint32_t *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getIdS(uint32_t *status, uint32_t tuner, uint32_t *tsid)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d)\n", getName(), this, __FUNCTION__, tuner);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d)\n", getName(), this, __FUNCTION__, tuner);
		}
		
		result = fProvider->getIdS(status, tuner, tsid);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetCorrectedErrorRate(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getCorrectedErrorRate((uint32_t *)&arguments->scalarOutput[0],
                                         (uint32_t)arguments->scalarInput[0],
                                         (ISDB)arguments->scalarInput[1],
                                         (LayerIndex)arguments->scalarInput[2],
                                         (ErrorRate *)arguments->structureOutput);
}
#endif

IOReturn PT1Client::getCorrectedErrorRate(uint32_t *status, uint32_t tuner, ISDB isdb, LayerIndex layerIndex, ErrorRate *errorRate)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d, layerIndex = %d)\n", getName(), this, __FUNCTION__, tuner, isdb, layerIndex);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
            layerIndex = (LayerIndex)OSSwapInt32((uint32_t)layerIndex);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d, layerIndex = %d)\n",
                    getName(), this, __FUNCTION__, tuner, isdb, layerIndex);
		}
		
		result = fProvider->getCorrectedErrorRate(status, tuner, isdb, layerIndex, errorRate);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            errorRate->Numerator = OSSwapInt32(errorRate->Numerator);
            errorRate->Denominator = OSSwapInt32(errorRate->Denominator);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sResetCorrectedErrorCount(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->resetCorrectedErrorCount((uint32_t *)&arguments->scalarOutput[0],
                                            (uint32_t)arguments->scalarInput[0],
                                            (ISDB)arguments->scalarInput[1]);
}
#endif

IOReturn PT1Client::resetCorrectedErrorCount(uint32_t *status, uint32_t tuner, ISDB isdb)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);
		}
		
		result = fProvider->resetCorrectedErrorCount(status, tuner, isdb);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetErrorCount(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getErrorCount((uint32_t *)&arguments->scalarOutput[0],
                                 (uint32_t)arguments->scalarInput[0],
                                 (ISDB)arguments->scalarInput[1],
                                 (uint32_t *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getErrorCount(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *count)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);
		}
		
		result = fProvider->getErrorCount(status, tuner, isdb, count);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *count = OSSwapInt32(*count);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetTmccS(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getTmccS((uint32_t *)&arguments->scalarOutput[0],
                            (uint32_t)arguments->scalarInput[0],
                            (TmccS *)arguments->structureOutput);
}

#endif
IOReturn PT1Client::getTmccS(uint32_t *status, uint32_t tuner, TmccS *tmcc)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d)\n", getName(), this, __FUNCTION__, tuner);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d)\n", getName(), this, __FUNCTION__, tuner);
		}
		
		result = fProvider->getTmccS(status, tuner, tmcc);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);

            tmcc->Indicator = OSSwapInt32(tmcc->Indicator);
            for (int i = 0; i < 4; ++i)
            {
                tmcc->Mode[i] = OSSwapInt32(tmcc->Mode[i]);
                tmcc->Slot[i] = OSSwapInt32(tmcc->Slot[i]);
            }
            for (int i = 0; i < 8; ++i)
            {
                tmcc->Id[i] = OSSwapInt32(tmcc->Id[i]);
            }
            tmcc->Emergency = OSSwapInt32(tmcc->Emergency);
            tmcc->UpLink = OSSwapInt32(tmcc->UpLink);
            tmcc->ExtFlag = OSSwapInt32(tmcc->ExtFlag);
            tmcc->ExtData[0] = OSSwapInt32(tmcc->ExtData[0]);
            tmcc->ExtData[1] = OSSwapInt32(tmcc->ExtData[1]);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetLayerS(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getLayerS((uint32_t *)&arguments->scalarOutput[0],
                             (uint32_t)arguments->scalarInput[0],
                             (LayerS *)arguments->structureOutput);
}

#endif

IOReturn PT1Client::getLayerS(uint32_t *status, uint32_t tuner, LayerS *layer)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d)\n", getName(), this, __FUNCTION__, tuner);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d)\n", getName(), this, __FUNCTION__, tuner);
		}
		
		result = fProvider->getLayerS(status, tuner, layer);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            for (int i = 0; i < LAYER_COUNT_S; ++i)
            {
                layer->Mode[i] = OSSwapInt32(layer->Mode[i]);
                layer->Count[i] = OSSwapInt32(layer->Count[i]);
            }
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetTmccT(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getTmccT((uint32_t *)&arguments->scalarOutput[0],
                            (uint32_t)arguments->scalarInput[0],
                            (TmccT *)arguments->structureOutput);
}

#endif
IOReturn PT1Client::getTmccT(uint32_t *status, uint32_t tuner, TmccT *tmcc)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d)\n", getName(), this, __FUNCTION__, tuner);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d)\n", getName(), this, __FUNCTION__, tuner);
		}
		
		result = fProvider->getTmccT(status, tuner, tmcc);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);

            tmcc->System = OSSwapInt32(tmcc->System);
            tmcc->Indicator = OSSwapInt32(tmcc->Indicator);
            tmcc->Emergency = OSSwapInt32(tmcc->Emergency);
            tmcc->Partial = OSSwapInt32(tmcc->Partial);
            for (int i = 0; i < LAYER_COUNT_T; ++i)
            {
                tmcc->Mode[i] = OSSwapInt32(tmcc->Mode[i]);
                tmcc->Rate[i] = OSSwapInt32(tmcc->Rate[i]);
                tmcc->Interleave[i] = OSSwapInt32(tmcc->Interleave[i]);
                tmcc->Segment[i] = OSSwapInt32(tmcc->Segment[i]);
            }
            tmcc->Phase = OSSwapInt32(tmcc->Phase);
            tmcc->Reserved = OSSwapInt32(tmcc->Reserved);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetLockedT(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getLockedT((uint32_t *)&arguments->scalarOutput[0],
                              (uint32_t)arguments->scalarInput[0],
                             (bool *)arguments->structureOutput);
}
#endif

IOReturn PT1Client::getLockedT(uint32_t *status, uint32_t tuner, bool locked[LAYER_COUNT_T])
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d)\n", getName(), this, __FUNCTION__, tuner);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);

			IOLog("%s[%p]::%s(input after swap: tuner = %d)\n", getName(), this, __FUNCTION__, tuner);
		}
		
		result = fProvider->getLockedT(status, tuner, locked);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);

            for (int i = 0; i < LAYER_COUNT_T; ++i)
            {
                locked[i] = (bool)OSSwapInt32((uint32_t)locked[i]);
            }
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetLayerEnable(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setLayerEnable((uint32_t *)&arguments->scalarOutput[0],
                                  (uint32_t)arguments->scalarInput[0],
                                  (ISDB)arguments->scalarInput[1],
                                  (LayerMask)arguments->scalarInput[2]);
}
#endif

IOReturn PT1Client::setLayerEnable(uint32_t *status, uint32_t tuner, ISDB isdb, LayerMask layerMask)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d, layerMask = %d)\n", getName(), this, __FUNCTION__, tuner, isdb, layerMask);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
            layerMask = (LayerMask)OSSwapInt32((uint32_t)layerMask);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d, layerMask = %d)\n",
                    getName(), this, __FUNCTION__, tuner, isdb, layerMask);
		}
		
		result = fProvider->setLayerEnable(status, tuner, isdb, layerMask);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetLayerEnable(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getLayerEnable((uint32_t *)&arguments->scalarOutput[0],
                                  (uint32_t)arguments->scalarInput[0],
                                  (ISDB)arguments->scalarInput[1],
                                  (LayerMask *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getLayerEnable(uint32_t *status, uint32_t tuner, ISDB isdb, LayerMask *layerMask)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
			tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
			
			IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);
		}
		
		result = fProvider->getLayerEnable(status, tuner, isdb, layerMask);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *layerMask = (LayerMask)OSSwapInt32((uint32_t)*layerMask);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetBufferInfo(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setBufferInfo((uint32_t *)&arguments->scalarOutput[0],
                                 (const BufferInfo *)arguments->structureInput);
}
#endif

IOReturn PT1Client::setBufferInfo(uint32_t *status, const BufferInfo *bufferInfo)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
        const BufferInfo *bufferInfo_;
        BufferInfo tmpInfo;
        if (fCrossEndian)
        {
            tmpInfo = *bufferInfo;
            tmpInfo.LockSize     = OSSwapInt32(bufferInfo->LockSize);
            tmpInfo.VirtualSize  = OSSwapInt32(bufferInfo->VirtualSize);
            tmpInfo.VirtualCount = OSSwapInt32(bufferInfo->VirtualCount);
            bufferInfo_ = &tmpInfo;
        }
        else
        {
            bufferInfo_ = bufferInfo;
        }

		result = fProvider->setBufferInfo(status, bufferInfo_);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetBufferInfo(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getBufferInfo((uint32_t *)&arguments->scalarOutput[0],
                                 (BufferInfo *)arguments->structureOutput);
}
#endif

IOReturn PT1Client::getBufferInfo(uint32_t *status, BufferInfo *bufferInfo)
{
	IOReturn	result;

//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if (fProvider == NULL || isInactive())
    {
        result = kIOReturnNotAttached;
    }
    else if (!fProvider->isOpen(this))
    {
        result = kIOReturnNotOpen;
    }
    else
    {
		result = fProvider->getBufferInfo(status, bufferInfo);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            bufferInfo->VirtualSize  = OSSwapInt32(bufferInfo->VirtualSize);
            bufferInfo->VirtualCount = OSSwapInt32(bufferInfo->VirtualCount);
            bufferInfo->LockSize     = OSSwapInt32(bufferInfo->LockSize);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sResetTransferCounter(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->resetTransferCounter((uint32_t *)&arguments->scalarOutput[0]);
}
#endif

IOReturn PT1Client::resetTransferCounter(uint32_t *status)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		result = fProvider->resetTransferCounter(status);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sIncrementTransferCounter(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->incrementTransferCounter((uint32_t *)&arguments->scalarOutput[0]);
}
#endif

IOReturn PT1Client::incrementTransferCounter(uint32_t *status)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		result = fProvider->incrementTransferCounter(status);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetStreamEnable(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setStreamEnable((uint32_t *)&arguments->scalarOutput[0],
                                   (uint32_t)arguments->scalarInput[0],
                                   (ISDB)arguments->scalarInput[1],
                                   (bool)arguments->scalarInput[2]);
}
#endif

IOReturn PT1Client::setStreamEnable(uint32_t *status, uint32_t tuner, ISDB isdb, bool enable)
{
    IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d, enable = %d)\n", getName(), this, __FUNCTION__, tuner, isdb, enable);

    if (fProvider == NULL || isInactive())
    {
        result = kIOReturnNotAttached;
    }
    else if (!fProvider->isOpen(this))
    {
        result = kIOReturnNotOpen;
    }
    else
    {
        if (fCrossEndian)
        {
            tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
            enable = (bool)OSSwapInt32((uint32_t)enable);

            IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d, enable = %d)\n",
                    getName(), this, __FUNCTION__, tuner, isdb, enable);
        }

        result = fProvider->setStreamEnable(status, tuner, isdb, enable);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
    }

    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetStreamEnable(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getStreamEnable((uint32_t *)&arguments->scalarOutput[0],
                                   (uint32_t)arguments->scalarInput[0],
                                   (ISDB)arguments->scalarInput[1],
                                   (bool *)&arguments->scalarOutput[1]);
}
#endif
IOReturn PT1Client::getStreamEnable(uint32_t *status, uint32_t tuner, ISDB isdb, bool *enable)
{
    IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);

    if (fProvider == NULL || isInactive())
    {
        result = kIOReturnNotAttached;
    }
    else if (!fProvider->isOpen(this))
    {
        result = kIOReturnNotOpen;
    }
    else
    {
        if (fCrossEndian)
        {
            tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);

            IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);
        }

        result = fProvider->getStreamEnable(status, tuner, isdb, enable);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *enable = (bool)OSSwapInt32((uint32_t)*enable);
        }
    }

    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetStreamGray(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setStreamGray((uint32_t *)&arguments->scalarOutput[0],
                                 (uint32_t)arguments->scalarInput[0],
                                 (ISDB)arguments->scalarInput[1],
                                 (uint32_t)arguments->scalarInput[2]);
}
#endif

IOReturn PT1Client::setStreamGray(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t gray)
{
	IOReturn	result;

//	IOLog("%s[%p]::%s(tuner = %d, isdb = %d, gray = %d)\n", getName(), this, __FUNCTION__, tuner, isdb, gray);

    if (fProvider == NULL || isInactive())
    {
        result = kIOReturnNotAttached;
    }
    else if (!fProvider->isOpen(this))
    {
        result = kIOReturnNotOpen;
    }
    else
    {
        if (fCrossEndian)
        {
            tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);
            gray = OSSwapInt32(gray);
			
            IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d, gray = %d)\n",
                    getName(), this, __FUNCTION__, tuner, isdb, gray);
        }

        result = fProvider->setStreamGray(status, tuner, isdb, gray);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
    }
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetStreamGray(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getStreamGray((uint32_t *)&arguments->scalarOutput[0],
                                 (uint32_t)arguments->scalarInput[0],
                                 (ISDB)arguments->scalarInput[1],
                                 (uint32_t *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getStreamGray(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *gray)
{
    IOReturn	result;

//    IOLog("%s[%p]::%s(tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);

    if (fProvider == NULL || isInactive())
    {
        result = kIOReturnNotAttached;
    }
    else if (!fProvider->isOpen(this))
    {
        result = kIOReturnNotOpen;
    }
    else
    {
        if (fCrossEndian)
        {
            tuner = OSSwapInt32(tuner);
            isdb = (ISDB)OSSwapInt32((uint32_t)isdb);

            IOLog("%s[%p]::%s(input after swap: tuner = %d, isdb = %d)\n", getName(), this, __FUNCTION__, tuner, isdb);
        }

        result = fProvider->getStreamGray(status, tuner, isdb, gray);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *gray = OSSwapInt32(*gray);
        }
    }

    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sSetTransferEnable(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->setTransferEnable((uint32_t *)&arguments->scalarOutput[0],
                                     (bool)arguments->scalarInput[0]);
}
#endif

IOReturn PT1Client::setTransferEnable(uint32_t *status, bool enable)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s(enable = %d)\n", getName(), this, __FUNCTION__, enable);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		if (fCrossEndian)
        {
            enable = (bool)OSSwapInt32((uint32_t)enable);
			
			IOLog("%s[%p]::%s(input after swap: enable = %d)\n", getName(), this, __FUNCTION__, enable);
		}
		
		result = fProvider->setTransferEnable(status, enable);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetTransferEnable(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getTransferEnable((uint32_t *)&arguments->scalarOutput[0],
                                     (bool *)&arguments->scalarOutput[1]);
}
#endif

IOReturn PT1Client::getTransferEnable(uint32_t *status, bool *enable)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		result = fProvider->getTransferEnable(status, enable);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);
            *enable = (bool)OSSwapInt32((uint32_t)*enable);
        }
	}
    
    return result;
}

#if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_X_VERSION_10_4
IOReturn PT1Client::sGetTransferInfo(PT1Client* target, void* reference, IOExternalMethodArguments* arguments)
{
    return target->getTransferInfo((uint32_t *)&arguments->scalarOutput[0],
                                   (TransferInfo *)arguments->structureOutput);
}
#endif

IOReturn PT1Client::getTransferInfo(uint32_t *status, TransferInfo *transferInfo)
{
	IOReturn	result;

//    IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

	if (fProvider == NULL || isInactive())
    {
		result = kIOReturnNotAttached;
	}
	else if (!fProvider->isOpen(this))
    {
		result = kIOReturnNotOpen;
	}
	else
    {
		result = fProvider->getTransferInfo(status, transferInfo);

        if (fCrossEndian)
        {
            *status = OSSwapInt32(*status);

            transferInfo->TransferCounter0 = OSSwapInt32(transferInfo->TransferCounter0);
            transferInfo->TransferCounter1 = OSSwapInt32(transferInfo->TransferCounter1);
            transferInfo->BufferOverflow = OSSwapInt32(transferInfo->BufferOverflow);
        }
	}
    
    return result;
}

