//----------------------------------------------------------------------
//
//			File:			"mainwindow.cpp"
//			Created:		24-Feb-2011
//			Author:			Nobuhide Tsuda
//			Description:	MainWindow NX
//
//----------------------------------------------------------------------

/*

	Copyright (C) 2011 by Nobuhide Tsuda

	{\[XR[h͊{I MIT CZXɏ]B
	http://www.opensource.org/licenses/mit-license.php
	http://sourceforge.jp/projects/opensource/wiki/licenses%2FMIT_license

	A͕sRŎg̈ GPL 匙Ȃ̂ŁA
	GPL CZXvWFNg{\[X𗬗p邱Ƃւ

*/

#ifdef	WIN32
#include <windows.h>
#endif	//	WIN32
#include <QtGui>
#include "vi.h"
#include "mainwindow.h"
#include "ViEditView.h"
#include "ViEngine.h"
#include "TestViCommands.h"
#include "charEncoding.h"

#define	VERSION_STR			"0.014 Dev"

#ifdef	WIN32
#pragma comment(lib, "Imm32.lib")
#endif	//WIN32

MainWindow::MainWindow(QWidget *parent, Qt::WFlags flags)
	: QMainWindow(parent, flags)
	, m_output(0)
{
	init();

	///doOutput("test\nHello, world.\n");
}
MainWindow::MainWindow(const QString &fileName, QWidget *parent, Qt::WFlags flags)
	: QMainWindow(parent, flags)
{
	init();
	if( !fileName.isEmpty() )
		loadFile(fileName);
}

void MainWindow::init()
{
	m_isUntitled = true;
	m_isModified = false;
	m_viEngine = new ViEngine;
	m_editor = new ViEditView;
	setCentralWidget(m_editor);
	m_viEngine->setEditor(m_editor);
#if !USE_EVENT_FILTER
	m_editor->setViEngine(m_viEngine);
#endif
	connect(m_viEngine, SIGNAL(modeChanged(Mode, ushort)), this, SLOT(onModeChanged(Mode, ushort)));
	connect(m_viEngine, SIGNAL(modeChanged(Mode)), m_editor, SLOT(setMode(Mode)));
	connect(m_viEngine, SIGNAL(closeView(ViEditView *, bool)), this, SLOT(closeView(ViEditView *, bool)));
	connect(m_viEngine, SIGNAL(closeAllViews(bool)), this, SLOT(closeAllViews(bool)));
	connect(m_viEngine, SIGNAL(open(const QString &)), this, SLOT(open(const QString &)));
	connect(m_viEngine, SIGNAL(save(const QString &)), this, SLOT(save(const QString &)));
	connect(m_viEngine, SIGNAL(testViCommands(const QString &)), this, SLOT(testViCommands(const QString &)));
	connect(m_viEngine, SIGNAL(showMessage(const QString &)), this, SLOT(showMessage(const QString &)));
	connect(m_viEngine, SIGNAL(doOutput(const QString &)), this, SLOT(doOutput(const QString &)));
	connect(m_viEngine, SIGNAL(clearOutput()), this, SLOT(clearOutput()));
	connect(m_viEngine, SIGNAL(regexpSearched(const QString &)), this, SLOT(onRegexpSearced(const QString &)));
	statusBar()->addWidget(m_cmdLineEdit = new QLineEdit(), 1);
	m_cmdLineEdit->installEventFilter(this);
	connect(m_cmdLineEdit, SIGNAL(textChanged(const QString &)), this, SLOT(cmdLineTextChanged(const QString &)));
	connect(m_cmdLineEdit, SIGNAL(returnPressed()), this, SLOT(cmdLineReturnPressed()));
	connect(m_cmdLineEdit, SIGNAL(cursorPositionChanged(int, int)), this, SLOT(cmdLineCursorPositionChanged(int, int)));
	connect(m_editor, SIGNAL(setFocusToCmdLine()), this, SLOT(setFocusToCmdLine()));
	connect(m_editor, SIGNAL(showMessage(const QString &)), this, SLOT(showMessage(const QString &)));
    connect(m_editor->document(), SIGNAL(contentsChanged()), this, SLOT(documentWasModified()));

	createActions();
	createMenus();
	createToolBars();
	createDockWindows();
    readSettings();			//	createActions() ̌ɃR[邱

	onModeChanged(m_viEngine->mode());

	update();		//	Ô߂ɉʖ
}

MainWindow::~MainWindow()
{

}
void MainWindow::createActions()
{
	newAct = new QAction(QIcon(":qvi/Resources/images/new.png"), tr("&New"), this);
	newAct->setIconText(tr("new doc"));
	//newAct->setToolTip(tr("new doc (tool tip)"));
	newAct->setShortcuts(QKeySequence::New);
	newAct->setStatusTip(tr("Create a new document"));
	connect(newAct, SIGNAL(triggered()), this, SLOT(newFile()));

	openAct = new QAction(QIcon(":qvi/Resources/images/open.png"), tr("&Open..."), this);
	openAct->setShortcuts(QKeySequence::Open);
	openAct->setStatusTip(tr("Open an existing file"));
	connect(openAct, SIGNAL(triggered()), this, SLOT(open()));

	saveAct = new QAction(QIcon(":qvi/Resources/images/save.png"), tr("&Save"), this);
	saveAct->setShortcuts(QKeySequence::Save);
	saveAct->setStatusTip(tr("Save the document to disk"));
	connect(saveAct, SIGNAL(triggered()), this, SLOT(save()));

	saveAsAct = new QAction(/*QIcon(":qvi/Resources/images/save.png"),*/ tr("Save&As..."), this);
	saveAsAct->setShortcuts(QKeySequence::SaveAs);
	saveAsAct->setStatusTip(tr("Save the document under a new fileName"));
	connect(saveAsAct, SIGNAL(triggered()), this, SLOT(saveAs()));

	closeAllAct = new QAction(tr("C&loseAll"), this);
	//closeAllAct->setShortcuts(QKeySequence::SaveAs);
	closeAllAct->setStatusTip(tr("Close all documents"));
	connect(closeAllAct, SIGNAL(triggered()), this, SLOT(closeAllViews()));

    //	RecentFilesMenu ̂߂̏
    for (int i = 0; i < MaxRecentFiles; ++i) {
        recentFileActs[i] = new QAction(this);
        recentFileActs[i]->setVisible(false);
        connect(recentFileActs[i], SIGNAL(triggered()),
                this, SLOT(openRecentFile()));
    }

    cutAct = new QAction(QIcon(":qvi/Resources/images/cut.png"), tr("Cu&t"), this);
    cutAct->setShortcuts(QKeySequence::Cut);
    cutAct->setStatusTip(tr("Cut the current selection's contents to the clipboard"));
    connect(cutAct, SIGNAL(triggered()), m_editor, SLOT(cut()));

    copyAct = new QAction(QIcon(":qvi/Resources/images/copy.png"), tr("&Copy"), this);
    copyAct->setShortcuts(QKeySequence::Copy);
    copyAct->setStatusTip(tr("Copy the current selection's contents to the clipboard"));
    connect(copyAct, SIGNAL(triggered()), m_editor, SLOT(copy()));

    pasteAct = new QAction(QIcon(":qvi/Resources/images/paste.png"), tr("&Paste"), this);
    pasteAct->setShortcuts(QKeySequence::Paste);
    pasteAct->setStatusTip(tr("Paste the clipboard's contents into the current selection"));
    connect(pasteAct, SIGNAL(triggered()), m_editor, SLOT(paste()));

	undoAct = new QAction(QIcon(":qvi/Resources/images/editundo.png"), tr("&Undo"), this);
    undoAct->setShortcuts(QKeySequence::Undo);
    undoAct->setStatusTip(tr("undo edit commande"));
	undoAct->setEnabled(false);
    connect(undoAct, SIGNAL(triggered()), m_editor, SLOT(undo()));
    connect(m_editor, SIGNAL(undoAvailable(bool)), undoAct, SLOT(setEnabled(bool)));

	redoAct = new QAction(QIcon(":qvi/Resources/images/editredo.png"), tr("&Redo"), this);
    redoAct->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_R));
    //redoAct->setShortcuts(QKeySequence::Redo);
    redoAct->setStatusTip(tr("redo edit commande"));
	redoAct->setEnabled(false);
    connect(redoAct, SIGNAL(triggered()), m_editor, SLOT(redo()));
    connect(m_editor, SIGNAL(redoAvailable(bool)), redoAct, SLOT(setEnabled(bool)));

    aboutAct = new QAction(QIcon(":qvi/Resources/images/Info.png"), tr("&About qvi"), this);
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(showAboutDlg()));

#if 0
	m_drawTextAct = new QAction(tr("darwText"), this);
	m_drawTextAct->setCheckable(true);
	m_drawTextAct->setChecked(true);
	connect(m_drawTextAct, SIGNAL(toggled(bool)), m_editor, SLOT(setDrawText(bool)));
#endif
}
void MainWindow::createMenus()
{
	QMenu *fileMenu = menuBar()->addMenu(tr("&File"));
	fileMenu->addAction(newAct);
	fileMenu->addAction(openAct);
	fileMenu->addAction(saveAct);
    separatorMRUAct = fileMenu->addSeparator();
    //	RecentFilesMenu ACeǉ
    for (int i = 0; i < MaxRecentFiles; ++i)
        fileMenu->addAction(recentFileActs[i]);
    updateRecentFileActions();
    fileMenu->addSeparator();
	fileMenu->addAction(closeAllAct);

	QMenu *editMenu = menuBar()->addMenu(tr("&Edit"));
    editMenu->addAction(undoAct);
    editMenu->addAction(redoAct);
	editMenu->addAction(cutAct);
	editMenu->addAction(copyAct);
	editMenu->addAction(pasteAct);

	QMenu *otherMenu = menuBar()->addMenu(tr("&Other"));
	otherMenu->addAction(aboutAct);
}
void MainWindow::createToolBars()
{
	QToolBar *fileToolBar = addToolBar(tr("File"));
	fileToolBar->addAction(newAct);
	fileToolBar->addAction(openAct);
	fileToolBar->addAction(saveAct);

	QToolBar *editToolBar = addToolBar(tr("Edit"));
	editToolBar->addAction(undoAct);
	editToolBar->addAction(redoAct);
	editToolBar->addAction(cutAct);
	editToolBar->addAction(copyAct);
	editToolBar->addAction(pasteAct);

	QToolBar *otherToolBar = addToolBar(tr("Other"));
	otherToolBar->addAction(aboutAct);
#if 0
	QToolBar *options = addToolBar(tr("Options"));
	options->addAction(m_drawTextAct);
#endif
}
void MainWindow::createDockWindows()
{
    QDockWidget *dock = new QDockWidget(tr("Output"), this);
    dock->setAllowedAreas(Qt::AllDockWidgetAreas);
    m_output = new QPlainTextEdit(dock);
    m_output->setReadOnly(true);
	QFont ft = m_output->font();
	ft.setPointSize(11);
	m_output->setFont(ft);
    dock->setWidget(m_output);
    addDockWidget(Qt::BottomDockWidgetArea, dock);
    //viewMenu->addAction(dock->toggleViewAction());
	m_output->viewport()->installEventFilter(this);
	//dock->update();
}
//----------------------------------------------------------------------
bool MainWindow::maybeSave()
{
	if( isWindowModified()) {
		QMessageBox::StandardButton ret;
		ret = QMessageBox::warning(this, tr("qvi"),
					 tr("The document has been modified.\n"
						"Do you want to save your changes?"),
					 QMessageBox::Save | QMessageBox::Discard
			 | QMessageBox::Cancel);
		if( ret == QMessageBox::Save )
			return save();
		else if( ret == QMessageBox::Cancel )
			return false;
	}
	return true;
}
void MainWindow::closeEvent(QCloseEvent *event)
{
	if( maybeSave()) {
		writeSettings();
		//event->accept();
		QMainWindow::closeEvent(event);
	} else {
		event->ignore();
	}
}
void MainWindow::writeSettings()
{
	QSettings settings;
	settings.setValue("geometry", saveGeometry());
	settings.setValue("windowState", saveState());
}
void MainWindow::readSettings()
{
	QSettings settings;
	restoreGeometry(settings.value("geometry").toByteArray());
	restoreState(settings.value("windowState").toByteArray());
}
void MainWindow::closeView(ViEditView *view, bool noSaveDlg)
{
	if( view == m_editor ) {
		if( noSaveDlg ) 
		    setWindowModified(false);
		close();		//	MainWindow N[Y
	}
}
void MainWindow::closeAllViews(bool noSaveDlg)
{
    foreach( QWidget *widget, qApp->topLevelWidgets() ) {
        MainWindow *mainWin = qobject_cast<MainWindow *>(widget);
        if( mainWin ) {
        	if( noSaveDlg )
        		mainWin->setWindowModified(false);
            mainWin->close();
        }
    }
}
void MainWindow::setFocusToCmdLine()
{
	//qDebug() << "MainWindow::setFocusToCmdLine()";
	m_cmdLineEdit->setFocus(Qt::OtherFocusReason);
}
void MainWindow::onImeOpenStatusChanged()
{
	//qDebug() << "MainWindow::onImeOpenStatusChanged()";
	m_viEngine->onImeOpenStatusChanged();
}
void MainWindow::onModeChanged(Mode mode, ushort subMode)
{
	//qDebug() << "mode = " << mode;
	QString text;
	switch( mode ) {
	case CMD: {
#ifdef	WIN32
		HWND hwnd = window()->winId();
		HIMC hC = ImmGetContext(hwnd);
		ImmSetOpenStatus(hC, FALSE);		//	IME OFF
#endif	//WIN32
		text = "CMD";
		break;
	}
	case INSERT:
		text = "INSERT";
		break;
	case REPLACE:
		text = "REPLACE";
		break;
	case CMDLINE:
		m_exCmdsIx = -1;
		m_cmdLineEdit->setText(QChar(subMode));
		m_cmdLineEdit->show();
		m_cmdLineEdit->setFocus(Qt::OtherFocusReason);
		statusBar()->repaint();
		return;
	}
	m_cmdLineEdit->hide();
	statusBar()->repaint();
	if( !m_viEngine->message().isEmpty() )
		text = m_viEngine->message();
	statusBar()->showMessage(text);
}
void MainWindow::cmdLineReturnPressed()
{
	QString text = m_cmdLineEdit->text();
	if( !text.isEmpty() ) {
		if( text[0] == ':' )
			m_viEngine->doExCommand(text.mid(1));
		else
			m_viEngine->doFind(text.mid(1), text[0] == '/');
	}
	m_viEngine->setMode(CMD);
}
void MainWindow::cmdLineCursorPositionChanged(int oldPos, int newPos)
{
	//qDebug() << "MainWindow::cmdLineCursorPositionChanged()";
	if( newPos == 0 )
		m_cmdLineEdit->setCursorPosition(1);
}
void MainWindow::cmdLineTextChanged(const QString & text)
{
	//qDebug() << "MainWindow::cmdLineTextChanged()";
	m_cmdText = text;
	if( text.isEmpty() || (text[0] != ':' && text[0] != '/' && text[0] != '?') )
		m_viEngine->setMode(CMD);
}
QString MainWindow::findCommand(const QStringList cmds, const QString text, bool up)
{
	int ix = m_exCmdsIx;
	QChar firstChar = text[0];
	const QString cmd = text.mid(1);
	for(;;) {
		if( up ) {
			if( --m_exCmdsIx < 0 ) m_exCmdsIx = cmds.count() - 1;
		} else {
			if( ++m_exCmdsIx >= cmds.count() ) m_exCmdsIx = 0;
		}
		if( cmd.isEmpty() || cmds[m_exCmdsIx].startsWith(cmd) || m_exCmdsIx == ix )
			return firstChar + cmds[m_exCmdsIx];
	}
}
bool MainWindow::eventFilter(QObject *obj, QEvent *event)
{
	if( obj == m_cmdLineEdit && event->type() == QEvent::KeyPress &&
		m_viEngine->mode() == CMDLINE )
	{
		QKeyEvent *keyEvent = static_cast<QKeyEvent *>(event);
		if( keyEvent->key() == Qt::Key_Escape ) {
			m_viEngine->setMode(CMD);
			return true;
		}
		const QString text = m_cmdText;
		QStringList cmds;
		if( text[0] == ':' )
			cmds = m_viEngine->exCommands();
		else if( text[0] == '/' || text[0] == '?' )
			cmds = findStrings();
		if( !cmds.isEmpty() ) {
			if( keyEvent->key() == Qt::Key_Up ) {
				//if( --m_exCmdsIx < 0 ) m_exCmdsIx = cmds.count() - 1;
				//m_cmdLineEdit->setText(":" + cmds[m_exCmdsIx]);
				m_cmdLineEdit->setText(findCommand(cmds, text, true));
				m_cmdText = text;		//	̓R}h
				return true;
			}
			if( keyEvent->key() == Qt::Key_Down ) {
				//if( ++m_exCmdsIx >= cmds.count() ) m_exCmdsIx = 0;
				//m_cmdLineEdit->setText(":" + cmds[m_exCmdsIx]);
				m_cmdLineEdit->setText(findCommand(cmds, text, false));
				m_cmdText = text;		//	̓R}h
				return true;
			}
		}
	}
	if( obj == m_output->viewport() ) {
		//qDebug() << event->type();
		if( event->type() == QEvent::MouseButtonDblClick ) {
			const QTextCursor cur = m_output->textCursor();
			const QTextBlock block = cur.block();
			//qDebug() << block.text();
			QString text = block.text();
#if 1
			int ix, ix2;
			if( !text.isEmpty() && text[0] == '"' &&
				(ix = text.indexOf("\"(", 1)) > 1 &&
				(ix2 = text.indexOf(")", ix)) > 1 )
			{
				QString fileName = text.mid(1, ix - 1);
				int lineNum = text.mid(ix+2, ix2-(ix+2)).toInt();
				loadFile(fileName, lineNum);
			}
#else
			//QRegExp tagexp("\"(.+)\"\\((\\d+)\\)");
			QRegExp tagexp("\"(.+)\"\\(");
			int pos = tagexp.indexIn(text);
			if( pos >= 0 ) {
				QString fileName = tagexp.cap(0);
				int lineNum = tagexp.cap(1).toInt();
			}
#endif
		}
	}
	
	return false;
}
void MainWindow::onRegexpSearced(const QString &text)
{
	if( text.isEmpty() ) return;
	m_findStrings.removeOne(text);		//	d폜
	m_findStrings.push_back(text);
	while( m_findStrings.count() > 100 )
		m_findStrings.pop_front();
}
void MainWindow::showMessage(const QString & text)
{
	//qDebug() << "MainWindow::showMessage()";
	if( m_viEngine->mode() != CMDLINE ) {
		//m_cmdLineEdit->hide();
		//qDebug() << "text = " << text;
		statusBar()->showMessage(text);
	}
}
//----------------------------------------------------------------------
MainWindow *MainWindow::findMainWindow(const QString &fileName)
{
    QString canonicalFilePath = QFileInfo(fileName).canonicalFilePath();

    foreach( QWidget *widget, qApp->topLevelWidgets() ) {
        MainWindow *mainWin = qobject_cast<MainWindow *>(widget);
        if( mainWin && mainWin->m_curFile == canonicalFilePath )
            return mainWin;
    }
    return 0;
}
void MainWindow::newFile()
{
	MainWindow *other = new MainWindow;
	other->move(x() + 40, y() + 40);
	other->show();
}
void MainWindow::open(const QString &fileName)
{
	if( fileName.isEmpty() )
		open();
	else {
		if( m_isUntitled && m_editor->document()->isEmpty()
				&& !isWindowModified() )
		{
			loadFile(fileName);
		} else {
			MainWindow *other = new MainWindow(fileName);
			if( other->m_isUntitled ) {
				delete other;
				return;
			}
			other->move(x() + 40, y() + 40);
			other->show();
		}
	}
}
void MainWindow::open()
{
	QString fileName = QFileDialog::getOpenFileName(this,
										tr("Select one or more files to open"),
										"",
										"*.*");
	//setDefaultSuffix("svg");
	if( !fileName.isEmpty()) {
		open(fileName);
		//loadFile(fileName);
	}
}
void MainWindow::loadFile(const QString &fileName, int lineNum)
{
    MainWindow *existing = findMainWindow(fileName);
    if( existing ) {
    	if( lineNum ) existing->doJump(lineNum);
        existing->show();
        existing->raise();
        existing->activateWindow();
        return;
    }

#if 1
	QApplication::setOverrideCursor(Qt::WaitCursor);
    QString buffer;
    QString errorString;
    uchar ce;
    bool withBOM;
	if( !::loadFile(fileName, buffer, errorString, &ce, &withBOM) ) {
		QMessageBox::warning(this, tr("qvi"),
							 tr("Cannot read file %1:\n%2.")
							 .arg(fileName)
							 .arg(errorString));
		QApplication::restoreOverrideCursor();
		return;
    }
	m_editor->setPlainText(buffer);
	m_editor->setCharEncodeing(ce);
	m_editor->setWithBOM(withBOM);
	QApplication::restoreOverrideCursor();
#elif 1
	QFile file(fileName);
	if( !file.open(QFile::ReadOnly /*| QFile::Text*/) ) {
		QMessageBox::warning(this, tr("qvi"),
							 tr("Cannot read file %1:\n%2.")
							 .arg(fileName)
							 .arg(file.errorString()));
		return;
	}
	QApplication::setOverrideCursor(Qt::WaitCursor);
	QByteArray ba = file.readAll();
	cuchar *ptr = (uchar *)(ba.data());
	//cuchar *ptr = static_cast<uchar *>(ba.data());
	cuchar *endptr = ptr + ba.size();
	int BOMLength;
	uchar ce = checkCharEncoding(ptr, endptr, BOMLength);
	QTextCodec *codec = 0;
	switch( ce ) {
	case CharEncoding::UTF8:
		codec = QTextCodec::codecForName("UTF-8");
		break;
	case CharEncoding::UTF16_LE:
		codec = QTextCodec::codecForName("UTF-16LE");
		break;
	case CharEncoding::UTF16_BE:
		codec = QTextCodec::codecForName("UTF-16BE");
		break;
	case CharEncoding::UNKNOWN:
	default:
		codec = QTextCodec::codecForName("Shift-JIS");
	}
	m_editor->setPlainText(codec->toUnicode(ba));
	m_editor->setCharEncodeing(ce);
	m_editor->setWithBOM(BOMLength != 0);
	//m_editor->setBOMLength(BOMLength);
	QApplication::restoreOverrideCursor();
#else
	QFile file(fileName);
	if( !file.open(QFile::ReadOnly | QFile::Text) ) {
		QMessageBox::warning(this, tr("qvi"),
							 tr("Cannot read file %1:\n%2.")
							 .arg(fileName)
							 .arg(file.errorString()));
		return;
	}

	QTextStream in(&file);
	QApplication::setOverrideCursor(Qt::WaitCursor);
	m_editor->setPlainText(in.readAll());
	QApplication::restoreOverrideCursor();
#endif
	m_editor->doJump(lineNum);

	setCurrentFile(fileName);
	statusBar()->showMessage(tr("File loaded"), 2000);
}
//	settings  RecentFile oArecentFileActs ɐݒ
void MainWindow::updateRecentFileActions()
{
    QSettings settings;
    QStringList files = settings.value("recentFileList").toStringList();
    int numRecentFiles = qMin(files.size(), (int)MaxRecentFiles);
    for (int i = 0; i < numRecentFiles; ++i) {
        QString text = tr("&%1 %2").arg((i + 1) % 10).arg(strippedName(files[i]));
        recentFileActs[i]->setText(text);
        recentFileActs[i]->setStatusTip(files[i]);
        recentFileActs[i]->setData(files[i]);
        recentFileActs[i]->setVisible(true);
    }
    for (int j = numRecentFiles; j < MaxRecentFiles; ++j)
        recentFileActs[j]->setVisible(false);

    separatorMRUAct->setVisible(numRecentFiles > 0);
}
QString MainWindow::strippedName(const QString &fullFileName)
{
    return QFileInfo(fullFileName).fileName();
}
void MainWindow::openRecentFile()
{
    QAction *action = qobject_cast<QAction *>(sender());
    if (action)
        loadFile(action->data().toString());
}
void MainWindow::doJump(int lineNum)
{
	if( lineNum && m_editor != 0 )
		m_editor->doJump(lineNum);
}
bool MainWindow::save()
{
	if( m_isUntitled ) {
		return saveAs();
	} else {
		return saveFile(m_curFile);
	}
}

bool MainWindow::saveAs()
{
	QString fileName = QFileDialog::getSaveFileName(this, tr("Save As"), m_curFile, "*.*");

	if( fileName.isEmpty() )
		return false;

	return saveFile(fileName);
}
void MainWindow::save(const QString &fileName)
{
	if( fileName.isEmpty() )
		save();
	else {
		saveFile(fileName, false);
	}
}
bool MainWindow::saveFile(const QString &fileName, bool replace)
{
	QFile file(fileName);
#if 1
	if( !file.open(QFile::WriteOnly /*| QFile::Text*/) ) {
		QMessageBox::warning(this, tr("qvi"),
							 tr("Cannot write file %1:\n%2.")
							 .arg(fileName)
							 .arg(file.errorString()));
		return false;
	}
	QApplication::setOverrideCursor(Qt::WaitCursor);
	QTextCodec *codec = 0;
	switch( m_editor->charEncoding() ) {
	case CharEncoding::UTF8:
		if( m_editor->withBOM() )
			file.write((cchar*)UTF8_BOM, UTF8_BOM_LENGTH);
		codec = QTextCodec::codecForName("UTF-8");
		break;
	case CharEncoding::UTF16_LE:
		if( m_editor->withBOM() )
			file.write((cchar*)UTF16LE_BOM, UTF16_BOM_LENGTH);
		codec = QTextCodec::codecForName("UTF-16LE");
		break;
	case CharEncoding::UTF16_BE:
		if( m_editor->withBOM() )
			file.write((cchar*)UTF16BE_BOM, UTF16_BOM_LENGTH);
		codec = QTextCodec::codecForName("UTF-16BE");
		break;
	case CharEncoding::EUC:
		codec = QTextCodec::codecForName("EUC-JP");
		break;
	case CharEncoding::UNKNOWN:
	default:
		codec = QTextCodec::codecForName("Shift-JIS");
	}
	QByteArray ba = codec->fromUnicode(m_editor->toPlainText());
	file.write(ba);
#else
	if( !file.open(QFile::WriteOnly | QFile::Text) ) {
		QMessageBox::warning(this, tr("qvi"),
							 tr("Cannot write file %1:\n%2.")
							 .arg(fileName)
							 .arg(file.errorString()));
		return false;
	}

	QTextStream out(&file);
	QApplication::setOverrideCursor(Qt::WaitCursor);
	out << m_editor->toPlainText();
#endif
	QApplication::restoreOverrideCursor();

	if( replace )
		setCurrentFile(fileName);
	statusBar()->showMessage(tr("File saved"), 2000);
	return true;
}
void MainWindow::updateCurFile()
{
    static int sequenceNumber = 0;
    if( m_curFile.isEmpty() )
        m_curFile = tr("document%1.txt").arg(++sequenceNumber);
}
void MainWindow::updateWindowTitle()
{
	updateCurFile();
	QString title = m_curFile;
	if( isWindowModified() )
		title += "*";
	title += " - qvi ";
	title += VERSION_STR;
    setWindowTitle(title);
}
void MainWindow::setCurrentFile(const QString &fileName)
{

    m_isUntitled = fileName.isEmpty();
    if( m_isUntitled ) {
        m_curFile.clear();
        updateCurFile();
    } else {
        m_curFile = QFileInfo(fileName).canonicalFilePath();
    }
    setWindowModified(false);
    updateWindowTitle();

#if 1
    QSettings settings;
    QStringList files = settings.value("recentFileList").toStringList();
    files.removeAll(fileName);
    files.prepend(fileName);
    while (files.size() > MaxRecentFiles)
        files.removeLast();
    settings.setValue("recentFileList", files);
    updateRecentFileActions();
#endif
}
void MainWindow::doOutput(const QString &text)
{
	if( m_output == 0 ) return;
	QTextCursor cur = m_output->textCursor();
	cur.movePosition(QTextCursor::End);
	cur.insertText(text);
	//m_output->setTextCursor(cur);
	//m_output->viewport()->update();
	m_output->viewport()->repaint();
	//update();
	//qDebug() << text;
}
void MainWindow::clearOutput()
{
	m_output->document()->clear();
	m_output->viewport()->repaint();	//	Ô
}

void MainWindow::testViCommands(const QString &fileName)
{
	if( m_isUntitled )
		::testViCommands(this, m_viEngine, fileName);
	else {
		//	t@CI[v͐VJ
		MainWindow *other = new MainWindow;
		other->move(x() + 40, y() + 40);
		other->show();
		::testViCommands(other, other->m_viEngine, fileName);
	}
}
void MainWindow::documentWasModified(bool mod)
{
    setWindowModified(mod);
    updateWindowTitle();
}
void MainWindow::showAboutDlg()
{
	QMessageBox msgBox;
	msgBox.setIconPixmap(QPixmap(":qvi/Resources/images/ayabu-004.png"));
	msgBox.setText(tr("<div align=center><b>qvi</b> version %1</div>").arg(VERSION_STR));
	msgBox.setInformativeText(QString("<div align=center>Copyright (C) 2011 by N.Tsuda<br>"
								"mailto:ntsuda@master.email.ne.jp<br>"
								"<a href=\"http://vivi.dyndns.org/?from=qvi%1\">http://vivi.dyndns.org/</a><br><br>"
								"Powered by <a href=\"http://qt.nokia.com/\">Qt</a>"
								"<p>illustrated by <a href=\"http://www.pixiv.net/member.php?id=220294\"><img src=\":qvi/Resources/images/momoshiki.png\"></a>"
								"</div>").arg(VERSION_STR));
	msgBox.exec();
}
