#!ruby
#vim: set fileencoding:utf-8

# chdisp.rb -
#   Windows向け解像度変更ユーティリティ
#   display mode changer for windows


if true
    $stdout.set_encoding("locale", undef: :replace, invalid: :replace)
else
    Encoding.default_internal = "utf-8"
    $stdout.binmode
    $stdout.set_encoding("locale", undef: :replace, invalid: :replace) if $stdout.tty?
end


class DisplayMode < Struct.new(:width,
                               :height,
                               :colordepth,
                               :vsync)
end

require "Win32API"

module WINDOWS
    class DEVMODE < Struct.new(                     # typedef struct _devicemodeA {
                               :dmDeviceName,       #   BYTE   dmDeviceName[CCHDEVICENAME]; # 0 :  0 (32)    
                               :dmSpecVersion,      #   WORD   dmSpecVersion;               # 1 : 32 ( 2)
                               :dmDriverVersion,    #   WORD   dmDriverVersion;             # 2 : 34 ( 2)
                               :dmSize,             #   WORD   dmSize;                      # 3 : 36 ( 2)
                               :dmDriverExtra,      #   WORD   dmDriverExtra;               # 4 : 38 ( 2)
                               :dmFields,           #   DWORD  dmFields;                    # 5 : 40 ( 4)
                               :"@1",               #   _ANONYMOUS_UNION union {            # 6 : 44 (16)
                                                    #     _ANONYMOUS_STRUCT struct {            : (16)
                                                    #       short dmOrientation;                    : ( 2)
                                                    #       short dmPaperSize;                      : ( 2)
                                                    #       short dmPaperLength;                    : ( 2)
                                                    #       short dmPaperWidth;                     : ( 2)
                                                    #       short dmScale;                          : ( 2)
                                                    #       short dmCopies;                         : ( 2)
                                                    #       short dmDefaultSource;                  : ( 2)
                                                    #       short dmPrintQuality;                   : ( 2)
                                                    #     } DUMMYSTRUCTNAME;                   
                                                    #     POINTL dmPosition;                    : ( 8)
                                                    #     DWORD  dmDisplayOrientation;          : ( 4)
                                                    #     DWORD  dmDisplayFixedOutput;          : ( 4)
                                                    #   } DUMMYUNIONNAME;
                                                    #
                               :dmColor,            #   short  dmColor;                     # 7 : 60 ( 2)
                               :dmDuplex,           #   short  dmDuplex;                    # 8 : 62 ( 2)
                               :dmYResolution,      #   short  dmYResolution;               # 9 : 64 ( 2)
                               :dmTTOption,         #   short  dmTTOption;                  #10 : 66 ( 2)
                               :dmCollate,          #   short  dmCollate;                   #11 : 68 ( 2)
                               :dmFormName,         #   BYTE   dmFormName[CCHFORMNAME];     #12 :100 (32)
                               :dmLogPixels,        #   WORD   dmLogPixels;                 #13 :102 ( 2)
                               :dmBitsPerPel,       #   DWORD  dmBitsPerPel;                #14 :104 ( 4)
                               :dmPelsWidth,        #   DWORD  dmPelsWidth;                 #15 :108 ( 4)
                               :dmPelsHeight,       #   DWORD  dmPelsHeight;                #16 :112 ( 4)
                               :"@2",               #   _ANONYMOUS_UNION union {            #17 :116 ( 4)
                                                    #     DWORD  dmDisplayFlags;                : ( 4)
                                                    #     DWORD  dmNup;                         : ( 4)
                                                    #   } DUMMYUNIONNAME2;
                               :dmDisplayFrequency, #   DWORD  dmDisplayFrequency;          #18 :120 ( 4)
                                                    # #if(WINVER >= 0x0400)
                               :dmICMMethod,        #   DWORD  dmICMMethod;                 #19 :124 ( 4)
                               :dmICMIntent,        #   DWORD  dmICMIntent;                 #20 :128 ( 4)
                               :dmMediaType,        #   DWORD  dmMediaType;                 #21 :132 ( 4)
                               :dmDitherType,       #   DWORD  dmDitherType;                #22 :136 ( 4)
                               :dmReserved1,        #   DWORD  dmReserved1;                 #23 :140 ( 4)
                               :dmReserved2,        #   DWORD  dmReserved2;                 #24 :144 ( 4)
                                                    # #if (WINVER >= 0x0500) || (_WIN32_WINNT >= 0x0400)
                               :dmPanningWidth,     #   DWORD  dmPanningWidth;              #25 :148 ( 4)
                               :dmPanningHeight)    #   DWORD  dmPanningHeight;             #26 :152 ( 4)
                                                    # #endif
                                                    # #endif /* WINVER >= 0x0400 */
                                                    # } DEVMODEA,*LPDEVMODEA,*PDEVMODEA;    :156

        def self.unpack(image)
            return new *image.unpack("Z32 vvvvV a16 vvvvv Z32 v VVVa4V VVVVVV VV")
        end
    end

    API_EnumDisplaySettings = Win32API.new("user32.dll", "EnumDisplaySettingsA", "plp", "l")
    API_GetDC = Win32API.new("user32.dll", "GetDC", "l", "l")
    API_GetDeviceCaps = Win32API.new("gdi32.dll", "GetDeviceCaps", "ll", "l")
    API_ReleaseDC = Win32API.new("user32.dll", "ReleaseDC", "l", "l")

    CDS_UPDATEREGISTRY  =  1
    CDS_TEST            =  2
    CDS_FULLSCREEN      =  4
    CDS_GLOBAL          =  8
    CDS_SET_PRIMARY     = 16

    ENUM_CURRENT_SETTINGS = (-1) & 0xffffffff
    ENUM_REGISTRY_SETTINGS = (-2) & 0xffffffff
    EDS_RAWMODE = 0x00000002

    def EnumDisplaySettings(devicename = nil)
        return EnumDisplaySettings!(devicename) do |buf|
            yield DEVMODE.unpack(buf)
        end
    end

    def EnumDisplaySettings!(devicename = nil)
        buf0 = ["", 156, ""].pack("a36 v a218")

        i = -1
        while API_EnumDisplaySettings.call(devicename, i += 1, buf = buf0.dup) != 0
            yield buf
        end

        return nil
    end


    class DC < Struct.new(:dc)
        HORZRES = 8
        VERTRES = 10
        BITSPIXEL = 12
        VREFRESH = 116

        def initialize(dc = nil)
            super(API_GetDC.call(dc.to_i))
        end

        def getcaps(q)
            raise("") unless dc
            return API_GetDeviceCaps.call(dc, q)
        end

        def release
            API_ReleaseDC.call(dc)
            self.dc = nil
            return nil
        end

        def self.open(dc = nil)
            dc = new(dc)
            if block_given?
                begin
                    return yield(dc)
                ensure
                    dc.release
                end
            else
                return dc
            end
        end
    end


    ##: BOOL EnumDisplaySettingsEx(
    ##:   LPCTSTR lpszDeviceName,  // 表示デバイス
    ##:   DWORD iModeNum,          // グラフィックスモード
    ##:   LPDEVMODE lpDevMode      // グラフィックスモードの設定
    ##:   DWORD dwFlags            // オプション
    ##: );
    API_EnumDisplaySettingsEx = Win32API.new "user32.dll", "EnumDisplaySettingsExA", "plpl", "l" rescue nil

    module_function
    def EnumDisplaySettingsEx(devicename = nil, index = nil, *flags)
        EnumDisplaySettingsEx!(devicename, index, *flags) do |buf|
            yield DEVMODE.unpack(buf)
        end

        return nil
    end

    module_function
    def EnumDisplaySettingsEx!(devicename = nil, index = nil, *flags)
        case flags.length
        when 0
            flags = 0
        when 1
            flags = flags.first
            if flags.kind_of?(Hash)
                flags = flags[:rawmode] ? EDS_RAWMODE : 0
            end
        else
            flags = flags.reduce(0) { |a, b| a | b }
        end

        buf0 = ["", 156, ""].pack("a36 v a218")
        case index
        when ENUM_CURRENT_SETTINGS, ENUM_REGISTRY_SETTINGS
            raise "error" if API_EnumDisplaySettingsEx.(devicename, index, buf0, flags) == 0
            yield buf0
        when nil
            index = -1
            while API_EnumDisplaySettingsEx.(devicename, index += 1, buf = buf0.dup, flags) != 0
                yield buf
            end
        else
            raise ArgumentError, "index must give nil, ENUM_CURRENT_SETTINGS or ENUM_REGISTRY_SETTINGS"
        end

        return nil
    end
end

include WINDOWS


def getdisplaymode
    DC.open do |dc|
        return DisplayMode.new(dc.getcaps(DC::HORZRES),
                               dc.getcaps(DC::VERTRES),
                               dc.getcaps(DC::BITSPIXEL),
                               dc.getcaps(DC::VREFRESH))
    end
end

def setdisplay(mode, &block)
    dms = finddisplay(mode)
    chdisp = Win32API.new("user32.dll", "ChangeDisplaySettingsExA", "ppllp", "l")

    if block
        begin
            raise("") unless chdisp.call(nil, dms, 0, CDS_FULLSCREEN, 0) == 0
            return yield
        ensure
            chdisp = Win32API.new("user32.dll", "ChangeDisplaySettingsA", "ll", "l")
            chdisp.call(0, 0)
        end
    else
        chdisp.call(nil, dms, 0, CDS_UPDATEREGISTRY, 0)
        return nil
    end
end

def finddisplay(mode)
    EnumDisplaySettings! do |devmode|
        dm = DEVMODE.unpack(devmode)

        if mode.vsync == dm.dmDisplayFrequency &&
            mode.width == dm.dmPelsWidth &&
            mode.height == dm.dmPelsHeight &&
            mode.colordepth == dm.dmBitsPerPel

            return devmode
        end
    end

    raise "display not matched"
end

def enumdisplay
    i = 0
    EnumDisplaySettings do |dm|
        i += 1
        yield DisplayMode.new(dm.dmPelsWidth, dm.dmPelsHeight, dm.dmBitsPerPel, dm.dmDisplayFrequency)
    end
    return i
end

DISPLAYMODE_PATTERN = %r(^
    (?:(?<WIDTH>\d+)x(?<HEIGHT>\d+))?
    (?:
        (?:\@(?<VSYNC>\d+))?
        (?:\#(?<DEPTH>\d+))?
        |
        (?:\#(?<DEPTH>\d+))?
        (?:\@(?<VSYNC>\d+))?
    )
$)x

def parsemode(mode, text)
    unless text.gsub(/\s+/m, "") =~ DISPLAYMODE_PATTERN
        raise("wrong display mode format (given \"#{text}\", valid [width x height] [ @ vsync ] [ # colordepth ]).")
    end

    width = $~[:WIDTH]
    height = $~[:HEIGHT]
    vsync = $~[:VSYNC]
    depth = $~[:DEPTH]

    mode.width = width.to_i if width
    mode.height = height.to_i if height
    mode.vsync = vsync.to_i if vsync
    mode.colordepth = depth.to_i if depth

    return mode
end


require "optparse"

display = getdisplaymode

opt = OptionParser.new(<<-__END__, 12)
使い方:\t#{File.basename $0}
\t#{File.basename $0} -l
\t#{File.basename $0} displaymode
\t#{File.basename $0} displaymode コマンドライン コマンドライン引数

\tdisplaymodeは、<width>x<height>@<vsync>#<colorbits>の形をとり、
\t<width>x<height> @<vsync> #<colorbits> のいずれかの値が省略できます。
\t@<vsync> と #<colorbits> の位置は入れ替えることができます
\t(例) 2560x1600#64@300
__END__

opt.separator ""

command = nil
opt.on("-l", "利用可能なディスプレイモードの一覧を表示します") { command = :listdisp }

opt.separator ""
opt.separator <<-__END__
#{File.basename $0}でディスプレイモードを指示した場合、解像度を変更します。
ディスプレイモードの後にコマンドライン(任意で引数も取ります)を指示した場合は、そのプログラムが終了したら元のディスプレイモードに戻します。
__END__

opt.order!(ARGV)

case command
when :listdisp
    modelist = {} # [width, height, colordepth] => [vsync, ...]
    enumdisplay do |display|
        t = [display.width, display.height, display.colordepth]
        modelist[t] ||= []
        modelist[t] << display.vsync
    end

    modelist.each do |a, b|
        b = b.join(" ")
        printf("%dx%d#%d @ %sHz\n", *a, b)
    end
when nil
    if ARGV.empty?
        printf("%dx%d@%d#%d\n",
               display.width, display.height, display.vsync, display.colordepth)
    else
        parsemode(display, ARGV.shift)
        if ARGV.empty?
            setdisplay(display)
        else
            setdisplay(display) { exit system(*ARGV) }
        end
    end
else
    raise "** BUG **"
end
