#!/usr/local/bin/ruby
#vim: set fileencoding:utf-8

# kls: Kernel object list (ls に kernel の k を付けた名前)
#
# WindowsNT内のカーネルオブジェクト名を一覧表示させることが出来る。
# winobjが利用できるなら、そっちのほうが高機能。
#
# 出来ること:
# * カーネル内の名前付きオブジェクト名を表示 (一階層ずつ)
# * オブジェクトの形態表示
# * シンボリックリンクの場合は、参照先を表示
# このくらい
#
# ruby-1.9.3p0 (mingw32) で動作確認


require "dl/import"

module Win32
    Win32 = self

    module Constants
        MAX_PATH = 260

        DELETE                           = 0x00010000
        READ_CONTROL                     = 0x00020000
        WRITE_DAC                        = 0x00040000
        WRITE_OWNER                      = 0x00080000
        SYNCHRONIZE                      = 0x00100000
        STANDARD_RIGHTS_REQUIRED         = 0x000F0000
        STANDARD_RIGHTS_READ             = READ_CONTROL
        STANDARD_RIGHTS_WRITE            = READ_CONTROL
        STANDARD_RIGHTS_EXECUTE          = READ_CONTROL
        STANDARD_RIGHTS_ALL              = 0x001F0000
        SPECIFIC_RIGHTS_ALL              = 0x0000FFFF

        DIRECTORY_QUERY                 = 0x0001
        DIRECTORY_TRAVERSE              = 0x0002
        DIRECTORY_CREATE_OBJECT         = 0x0004
        DIRECTORY_CREATE_SUBDIRECTORY   = 0x0008
        DIRECTORY_ALL_ACCESS            = STANDARD_RIGHTS_REQUIRED | 0x0F

        GENERIC_READ = 0x80000000 | (-1 << 32) # dl が unsigned を理解しないための仕掛け
        GENERIC_WRITE = 0x40000000
        FILE_SHARE_DELETE = 0x00000004
        FILE_SHARE_READ = 0x00000001
        FILE_SHARE_WRITE = 0x00000002
        CREATE_ALWAYS = 2
        CREATE_NEW = 1
        OPEN_ALWAYS = 4
        OPEN_EXISTING = 3
        TRUNCATE_EXISTING = 5
        FILE_ATTRIBUTE_ARCHIVE = 0x20
        FILE_ATTRIBUTE_ENCRYPTED = 0x4000
        FILE_ATTRIBUTE_HIDDEN = 0x2
        FILE_ATTRIBUTE_NORMAL = 0x80
        FILE_ATTRIBUTE_OFFLINE = 0x1000
        FILE_ATTRIBUTE_READONLY = 0x1
        FILE_ATTRIBUTE_SYSTEM = 0x4
        FILE_ATTRIBUTE_TEMPORARY = 0x100
        FILE_FLAG_BACKUP_SEMANTICS = 0x02000000
        FILE_FLAG_DELETE_ON_CLOSE = 0x04000000
        FILE_FLAG_NO_BUFFERING = 0x20000000
        FILE_FLAG_OPEN_NO_RECALL = 0x00100000
        FILE_FLAG_OPEN_REPARSE_POINT = 0x00200000
        FILE_FLAG_OVERLAPPED = 0x40000000
        FILE_FLAG_POSIX_SEMANTICS = 0x0100000
        FILE_FLAG_RANDOM_ACCESS = 0x10000000
        FILE_FLAG_SEQUENTIAL_SCAN = 0x08000000
        FILE_FLAG_WRITE_THROUGH = 0x80000000
    end
    include Constants

    require_relative "ntstatus"
    include NTSTATUS

    module Utils
        module_function
        def FIX_UTF16!(str)
            str.force_encoding(Encoding::BINARY)
            str << ?\0
            str.slice!(-1)
            str.force_encoding(Encoding::UTF_16LE)
        end

        module_function
        def strtowcs(str)
            FIX_UTF16!(str.encode(Encoding::UTF_16LE))
        end

        module_function
        def strtowcs!(str)
            FIX_UTF16!(str.encode!(Encoding::UTF_16LE))
        end

        module_function
        def WCSTOSTR!(wcs)
            wcs.force_encoding(Encoding::BINARY)
            wcs.slice!(/(?:\0{2})(?:.{2})*\z/m)
            wcs.encode!(Encoding::UTF_8, Encoding::UTF_16LE)
        end

        module_function
        def wcstostr(wcs)
            WCSTOSTR!(wcs.encode(Encoding::BINARY, Encoding::BINARY))
        end

        module_function
        def wcstostr!(wcs)
            WCSTOSTR!(wcs)
        end

        module_function
        def filetimetotime(filetime)
            # FILETIMEの起算時は 1601年1月1日 0時0分0秒 (UTC) で、精度は 100ns
            filetime = filetime - 116444736000000000
            sec = filetime / 10000000
            usec = (filetime % 10000000) / 10.0
            Time.at(sec, usec)
        end

        module_function
        def memalloc(size)
            buf = "\0".force_encoding(Encoding::BINARY) * size
            class << buf
                def resize(size)
                    self[size .. -1] = ""
                    self
                end
            end
            buf
        end

        # 指定された番地からデータをぶっこ抜いてStringインスタンスを構築しちゃうよ!!
        # 番地はまったく確認しないから地獄を見るよ!!
        module_function
        def buftostr(addr, size)
            dest = Utils.memalloc(size)
            MSVCRT.memcpy(Utils.getaddr(dest), addr, size)
            dest
        end

        module_function
        def getaddr(str)
            [str].pack("P").unpack("V")[0]
        end
    end
    include Utils

    # typedef struct _WIN32_FIND_DATA {
    #   DWORD    dwFileAttributes;
    #   FILETIME ftCreationTime;
    #   FILETIME ftLastAccessTime;
    #   FILETIME ftLastWriteTime;
    #   DWORD    nFileSizeHigh;
    #   DWORD    nFileSizeLow;
    #   DWORD    dwReserved0;
    #   DWORD    dwReserved1;
    #   TCHAR    cFileName[MAX_PATH];
    #   TCHAR    cAlternateFileName[14];
    # } WIN32_FIND_DATA, *PWIN32_FIND_DATA, *LPWIN32_FIND_DATA;
    class WIN32_FIND_DATA < Struct.new(:dwFileAttributes,
                                       :ftCreationTime,
                                       :ftLastAccessTime,
                                       :ftLastWriteTime,
                                       :nFileSize,
                                       :dwReserved0,
                                       :dwReserved1,
                                       :cFileName,
                                       :cAlternateFileName)
        def to_s
            sprintf("%08x  %s  %12d  %s",
                    dwFileAttributes,
                    ftLastWriteTime,
                    nFileSize,
                    cFileName)
        end

        def self.unpackW(seq)
            d = seq.unpack("IQ<3I2I2a#{Constants::MAX_PATH * 2}a#{14 * 2}")
            d[1] = Utils.filetimetotime(d[1])
            d[2] = Utils.filetimetotime(d[2])
            d[3] = Utils.filetimetotime(d[3])
            d[4] <<= 32
            d[4] |= d[5]
            d.slice!(5)
            Utils.wcstostr!(d[7])
            Utils.wcstostr!(d[8])
            new(*d)
        end

        def self.mallocW
            size = 4 + 8 + 8 + 8 + 8 + 4 + 4 + 260 * 2 + 14 * 2
            pool = Utils.memalloc(size)
            class << pool
                def unpack!
                    WIN32_FIND_DATA.unpackW(self)
                end
            end
            pool
        end
    end

    # typedef struct _UNICODE_STRING {
    #   USHORT Length;
    #   USHORT MaximumLength;
    #   PWSTR  Buffer;
    # } UNICODE_STRING, *PUNICODE_STRING;
    class UNICODE_STRING < Struct.new(:length, :capacity, :string)
        def pack
            [length, capacity, string].pack("vvP")
        end

        def self.newbystr(str)
            str = Utils.strtowcs(str)
            new(str.bytesize, str.bytesize, str)
        end
    end

    # typedef struct _OBJECT_ATTRIBUTES {
    #   ULONG Length;
    #   HANDLE RootDirectory;
    #   PUNICODE_STRING ObjectName;
    #   ULONG Attributes;
    #   PVOID SecurityDescriptor;
    #   PVOID SecurityQualityOfService;
    # } OBJECT_ATTRIBUTES, *POBJECT_ATTRIBUTES;
    class OBJECT_ATTRIBUTES < Struct.new(:root, :name, :attr, :secdesc, :secqos)
        def pack
            name = UNICODE_STRING.newbystr(self.name).pack
            [24, root, name, attr, nil, nil].pack("IIPIPP")
        end
    end

    # typedef struct _OBJECT_DIRECTORY_INFORMATION {
    #     UNICODE_STRING Name;
    #     UNICODE_STRING TypeName;
    # } OBJECT_DIRECTORY_INFORMATION, *POBJECT_DIRECTORY_INFORMATION;
    class OBJECT_DIRECTORY_INFORMATION < Struct.new(:name, :typename)
        def self.alloc(size = 1024)
            return Utils.memalloc(size)
        end
    end

    module NTDLL
        module API
            extend DL::Importer
            dlload "ntdll.dll"

            typealias "NTSTATUS", "unsigned long"
            typealias "DWORD", "unsigned long"

            # NTSTATUS WINAPI NtOpenDirectoryObject(
            #   __out  PHANDLE DirectoryHandle,
            #   __in   ACCESS_MASK DesiredAccess,
            #   __in   POBJECT_ATTRIBUTES ObjectAttributes
            # );
            extern "NTSTATUS NtOpenDirectoryObject(void*, unsigned long, void*)"

            # NTSTATUS WINAPI NtQueryDirectoryObject(
            #   __in       HANDLE DirectoryHandle,
            #   __out_opt  PVOID Buffer,
            #   __in       ULONG Length,
            #   __in       BOOLEAN ReturnSingleEntry,
            #   __in       BOOLEAN RestartScan,
            #   __inout    PULONG Context,
            #   __out_opt  PULONG ReturnLength
            # );
            extern "NTSTATUS NtQueryDirectoryObject(int, void*, unsigned long, int, int, void*, void*)"

            # NTSTATUS WINAPI NtOpenSymbolicLinkObject(
            #   __out  PHANDLE LinkHandle,
            #   __in   ACCESS_MASK DesiredAccess,
            #   __in   POBJECT_ATTRIBUTES ObjectAttributes
            # );
            extern "NTSTATUS NtOpenSymbolicLinkObject(void*, unsigned long, void*)"

            # NTSTATUS WINAPI NtQuerySymbolicLinkObject(
            #   __in       HANDLE LinkHandle,
            #   __inout    PUNICODE_STRING LinkTarget,
            #   __out_opt  PULONG ReturnedLength
            # );
            extern "NTSTATUS NtQuerySymbolicLinkObject(int, void*, void*)"
        end

        def self.NtOpenDirectoryObject(*argv)
            status = API.NtOpenDirectoryObject(*argv)
            return nil if status == NTSTATUS::STATUS_SUCCESS
            NTSTATUS.raise(status, caller)
        end

        def self.NtQueryDirectoryObject *argv
            status = API.NtQueryDirectoryObject(*argv)
            return true if status == NTSTATUS::STATUS_SUCCESS
            return false if status == NTSTATUS::STATUS_NO_MORE_ENTRIES
            NTSTATUS.raise(status, caller)
        end

        def self.NtOpenSymbolicLinkObject *argv
            status = API.NtOpenSymbolicLinkObject(*argv)
            return nil if status == NTSTATUS::STATUS_SUCCESS
            NTSTATUS.raise(status, caller)
        end

        def self.NtQuerySymbolicLinkObject *argv
            status = API.NtQuerySymbolicLinkObject(*argv)
            return nil if status == NTSTATUS::STATUS_SUCCESS
            NTSTATUS.raise(status, caller)
        end
    end

    module Kernel32
        extend DL::Importer
        dlload "Kernel32.dll"

        FORMAT_MESSAGE_FROM_SYSTEM = 0x1000
        FORMAT_MESSAGE_IGNORE_INSERTS = 0x200

        typealias "DWORD", "uint"

        # DWORD WINAPI GetLastError(void);
        extern "unsigned int GetLastError()"

        # DWORD WINAPI FormatMessage(
        #   __in      DWORD dwFlags,
        #   __in_opt  LPCVOID lpSource,
        #   __in      DWORD dwMessageId,
        #   __in      DWORD dwLanguageId,
        #   __out     LPTSTR lpBuffer,
        #   __in      DWORD nSize,
        #   __in_opt  va_list *Arguments
        # );
        extern "unsigned int FormatMessageW(DWORD, void *, DWORD, DWORD, void *, DWORD, void *)"

        # HANDLE WINAPI CreateFile(
        #   __in      LPCTSTR lpFileName,
        #   __in      DWORD dwDesiredAccess,
        #   __in      DWORD dwShareMode,
        #   __in_opt  LPSECURITY_ATTRIBUTES lpSecurityAttributes,
        #   __in      DWORD dwCreationDisposition,
        #   __in      DWORD dwFlagsAndAttributes,
        #   __in_opt  HANDLE hTemplateFile
        # );
        extern "int CreateFileW(void *, DWORD, DWORD, void *, DWORD, DWORD, int)"

        # BOOL WINAPI CloseHandle(
        #   __in  HANDLE hObject
        # );
        extern "int CloseHandle(int)"

        # HANDLE WINAPI FindFirstFile(
        #   __in   LPCTSTR lpFileName,
        #   __out  LPWIN32_FIND_DATA lpFindFileData
        # );
        extern "int FindFirstFileW(void *, void *)"

        # BOOL WINAPI FindNextFile(
        #   __in   HANDLE hFindFile,
        #   __out  LPWIN32_FIND_DATA lpFindFileData
        # );
        extern "int FindNextFileW(int, void *)"

        # BOOL WINAPI FindClose(
        #   __inout  HANDLE hFindFile
        # );
        extern "int FindClose(int)"
    end

    module MSVCRT
        extend DL::Importer
        dlload "msvcrt.dll"

        extern "void memcpy(unsigned long, unsigned long, unsigned long)"
    end

    module_function
    def findfile(path)
        data = WIN32_FIND_DATA.mallocW
        find = Kernel32.FindFirstFileW(Utils.strtowcs(path), data)
        if find <= 0
            err = Kernel32.GetLastError
            raise RuntimeError,
                  "#{geterrmesg(err).encode "locale"} (#{err})"
        end
        yield data.unpack!
        while Kernel32.FindNextFileW(find, data) != 0
            yield data.unpack!
        end
        nil
    ensure
        Kernel32.FindClose(find) if find && find > 0
    end

    module_function
    def geterrmesg(errcode)
        buf = Utils.memalloc(4096)
        size = Kernel32.FormatMessageW(Kernel32::FORMAT_MESSAGE_FROM_SYSTEM | Kernel32::FORMAT_MESSAGE_IGNORE_INSERTS,
                                       nil,
                                       errcode.to_i,
                                       0,
                                       buf,
                                       buf.bytesize / 2,
                                       nil)
        buf.resize(size * 2)
        buf.encode!("utf-8", "utf-16le")
        buf.strip! # いらない改行を付けてくるので削除。chomp!じゃないのはなんとなく。
        return buf
    end
end


if $0 == __FILE__
    module Win32
        base = ARGV[0] || "\\"
        base = "\\" << base unless base[0] == "\\"
        dir = [0].pack("V")
        objattr = OBJECT_ATTRIBUTES.new(0, base, 0, nil, nil)
        objattr = objattr.pack
        begin
            NTDLL.NtOpenDirectoryObject(dir, DIRECTORY_QUERY | DIRECTORY_TRAVERSE, objattr)
            dir = dir.unpack("V")[0]
            begin
                base += "\\" unless base[-1] == "\\"

                entries = []
                1000000.times do |i| # 1000000は、実質的な無限と見ることが出来るものとして選択
                    info = OBJECT_DIRECTORY_INFORMATION.alloc
                    context = [i].pack("V")
                    size = [0].pack("V")
                    break unless NTDLL.NtQueryDirectoryObject(dir, info, info.bytesize, 1, 0, context, size)
                    context = context.unpack("V")[0]
                    size = size.unpack("V")[0]
                    info1 = info.unpack("vvVvvV")
                    name = Utils.wcstostr(Utils.buftostr(info1[2], info1[0]))
                    type = Utils.wcstostr(Utils.buftostr(info1[5], info1[3]))
                    entries << [base + name, type.intern]
                end

                entries.sort_by! { |e| [e[0].downcase, e[1]] }
                entries.each do |e|
                    if e[1] == :SymbolicLink
                        linkh = [0].pack("V")
                        attr = OBJECT_ATTRIBUTES.new(0, e[0], 0, nil, nil).pack
                        begin
                            NTDLL.NtOpenSymbolicLinkObject(linkh, GENERIC_READ, attr)
                            linkh = linkh.unpack("V")[0]
                            begin
                                name = Utils.memalloc(1024)
                                namep = [0, name.length / 2, name].pack "vvP"
                                NTDLL.NtQuerySymbolicLinkObject(linkh, namep, nil)
                                Utils.wcstostr!(name)
                                link = "  <#{name}>"
                            ensure
                                Kernel32.CloseHandle(linkh)
                            end
                        rescue Errno::EPERM
                            link = "  <<#$!>>"
                        end
                    else
                        link = ""
                    end

                    puts "%-16s  %s%s\n" % [e[1], e[0], link]
                end
            ensure
                Kernel32.CloseHandle(dir)
            end
        rescue SystemCallError
            $stderr.puts "#{File.basename __FILE__}:#{__LINE__}: #$!\n"
            exit 1
        end
    end
end
