#!/usr/local/bin/ruby
#vim: set fileencoding:utf-8

# listvolumes: Windows上でストレージボリュームを一覧表示する
#
# できること:
# * ボリューム名、ファイルシステム名、空き容量、全容量、ボリュームラベル それぞれを表示
# これだけ
#
# オプションも引数も与えても無視されます


require "dl/import"

module Win32
    Win32 = self

    module_function
    def newbuf(size)
        buf = ("\0" * size).force_encoding(Encoding::BINARY)
        class << buf
            def resize(size)
                buf[size .. -1] = ""
                self
            end
        end
        buf
    end

    module_function
    def strtowcs(str)
        str = str.encode(Encoding::UTF_16LE)

        # 以下の4行は encode が付加しない、UTF-16としての終端記号を追加している
        str.force_encoding(Encoding::BINARY)
        str << "\0"
        str[-1] = ""
        str.force_encoding(Encoding::UTF_16LE)
    end

    module Kernel32
        module API
            extend DL::Importer
            dlload "kernel32.dll"

            # HANDLE WINAPI FindFirstVolume(
            #   __out  LPTSTR lpszVolumeName,
            #   __in   DWORD cchBufferLength
            # );
            extern "int FindFirstVolumeA(void *, unsigned int)"

            # BOOL WINAPI FindNextVolume(
            #   __in   HANDLE hFindVolume,
            #   __out  LPTSTR lpszVolumeName,
            #   __in   DWORD cchBufferLength
            # );
            extern "int FindNextVolumeA(int, void *, int)"

            # BOOL WINAPI FindVolumeClose(
            #   __in  HANDLE hFindVolume
            # );
            extern "int FindVolumeClose(int)"

            # BOOL WINAPI GetVolumeInformation(
            #   __in_opt   LPCTSTR lpRootPathName,
            #   __out      LPTSTR lpVolumeNameBuffer,
            #   __in       DWORD nVolumeNameSize,
            #   __out_opt  LPDWORD lpVolumeSerialNumber,
            #   __out_opt  LPDWORD lpMaximumComponentLength,
            #   __out_opt  LPDWORD lpFileSystemFlags,
            #   __out      LPTSTR lpFileSystemNameBuffer,
            #   __in       DWORD nFileSystemNameSize
            # );
            extern "int GetVolumeInformationW(void *, void *, unsigned int, void *, void *, void *, void *, unsigned int)"

            # BOOL GetDiskFreeSpaceEx(
            #   LPCTSTR lpDirectoryName,                 // ディレクトリ名
            #   PULARGE_INTEGER lpFreeBytesAvailable,    // 呼び出し側が利用できるバイト数
            #   PULARGE_INTEGER lpTotalNumberOfBytes,    // ディスク全体のバイト数
            #   PULARGE_INTEGER lpTotalNumberOfFreeBytes // ディスク全体の空きバイト数
            # );
            extern "int GetDiskFreeSpaceExW(void *, void *, void *, void *)"
        end

        module_function
        def FindVolume
            buf = Win32.newbuf(1024)
            findvol = API.FindFirstVolumeA(buf, buf.bytesize)
            if findvol > 0
                yield buf.unpack("Z*")[0]
                while API.FindNextVolumeA(findvol, buf, buf.bytesize) != 0
                    yield buf.unpack("Z*")[0]
                end
            end
            nil
        ensure
            if findvol && findvol > 0
                API.FindVolumeClose(findvol)
            end
        end

        DiskFreeSpace = Struct.new(:avail, :total, :free)

        module_function
        def GetDiskFreeSpaceEx(path)
            path = path.gsub("/", "\\\\")
            path << "\\" unless path[-1] == "\\"
            buf1 = Win32.newbuf(8)
            buf2 = Win32.newbuf(8)
            buf3 = Win32.newbuf(8)
            return false if API.GetDiskFreeSpaceExW(Win32.strtowcs(path), buf1, buf2, buf3) == 0

            buf1 = buf1.unpack("Q<")[0]
            buf2 = buf2.unpack("Q<")[0]
            buf3 = buf3.unpack("Q<")[0]

            DiskFreeSpace.new(buf1, buf2, buf3)
        end
    end

    module_function
    def each_volume
        unless block_given?
            list = []
            each_volume { |vol| list << vol }
            list
        end

        Kernel32.FindVolume do |volname|
            yield volname
        end
    end

    class VolumeInfo < Struct.new(:volumename, :serialnumber, :maxcomponents, :volumeflags, :filesystem)
    end

    module_function
    def getvolumeinfo(rootpath = nil)
        rootpath = Win32.strtowcs(rootpath) if rootpath
        volname = Win32.newbuf(1024)
        serialnum = Win32.newbuf(4)
        maxcomplen = Win32.newbuf(4)
        fsflags = Win32.newbuf(4)
        fsname = Win32.newbuf(1024)
        if Kernel32::API.GetVolumeInformationW(rootpath, volname, volname.bytesize,
                                               serialnum, maxcomplen, fsflags,
                                               fsname, fsname.bytesize) == 0
            raise "GetVolumeInformationW で失敗 (しょうさいはGetLastErrorでみてね)"
        end
        volname.sub!(/(?:\0{2})*\Z/m, "")
        volname.encode!(Encoding::UTF_8, Encoding::UTF_16LE)
        serialnum = serialnum.unpack("V")[0]
        maxcomplen = maxcomplen.unpack("V")[0]
        fsflags = fsflags.unpack("V")[0]
        fsname.sub!(/(?:\0{2})*\Z/m, "")
        fsname.encode!(Encoding::UTF_8, Encoding::UTF_16LE)
        VolumeInfo.new(volname, serialnum, maxcomplen, fsflags, fsname)
    end

    module_function
    def getfreespace(path)
        Kernel32.GetDiskFreeSpaceEx(path)
    end
end

Win32.each_volume do |vol|
    info = Win32.getvolumeinfo(vol) rescue nil
    if info
        space = Win32.getfreespace(vol)
        printf("%s  %-10s  %8d Mo  %8d Mo  %s\n",
               vol, info.filesystem,
               space.avail / 1000 / 1000, space.total / 1000 / 1000,
               info.volumename)
    else
        puts "%s  <情報の取得に失敗>\n" % [vol]
    end
end
