#!/usr/local/bin/ruby
#vim: set fileencoding:utf-8

=begin

matroska.rb - mkv/mka/webm 解析ライブラリ

AUTHOR:: dearblue <dearblue@gmx.com>
LICENSE:: 2-clause BSD license
PROJECT SITE:: http://sourceforge.jp/projects/rutsubo/

=end

class EBML < Struct.new(:io, :doctype)
    BasicStruct = superclass

    module Exceptions
        BasicException = Class.new(::RuntimeError)
        NotEBML = Class.new BasicException
    end
    include Exceptions

    def initialize(io)
        doctype = self.class.read(io)
        raise(NotEBML, nil, caller) unless doctype.id == Marker::EBML
        super(io, doctype)
    end

    def root
        io.pos = doctype.offset + doctype.size
        Element.new(self, nil, self.class.read(io))
    end

    def path(*path, &block)
        n = root
        if path.first == n.id
            if path.size > 1
                n.path(*path.slice(1 .. -1), &block)
            else
                yield(n)
            end
        end
        self
    end

    def [](tag)
        r = root
        return r if r.id == tag
        nil
    end

    class Element < Struct.new(:ebml, :parent, :entity)
        BasicStruct = superclass

        class Entity < Struct.new(:id, :offset, :size)
            BasicStruct = superclass

            def to_s
                "%8x [%8x + %5d]" % to_a
            end
        end

        def id
            entity.id
        end

        def size
            entity.size
        end

        def offset
            entity.offset
        end

        def eachchild
            io = ebml.io
            io.pos = offset
            border = offset + size
            while io.pos < border
                e = self.class.new(ebml, self, ebml.class.read(io))
                yield(e.dup)
                io.pos = e.offset + e.size
            end
        end

        def [](tag)
            eachchild do |e|
                return e if e.id == tag
            end
            return nil
        end

        def path(*path, &block)
            eachchild do |e|
                if e.id == path.first
                    if path.size > 1
                        e.path(*path.slice(1 .. -1), &block)
                    else
                        yield(e) 
                    end
                end
            end
            return self
        end

        def readint
            ebml.io.pos = offset
            buf = ebml.io.read(entity.size)
            num = 0
            buf.each_byte { |e| num = (num << 8) | e }
            num
        end

        def readfloat
            ebml.io.pos = offset
            buf = ebml.io.read(4)
            buf.unpack("g").first
        end

        # でかい要素は安全のため弾く
        def read(buf = "".force_encoding(Encoding::BINARY))
            raise(DataTooHuge, nil, caller) if size > 16 * 1024 * 1024
            read!(buf)
        end

        def read!(buf = "".force_encoding(Encoding::BINARY))
            io = ebml.io
            io.pos = offset
            io.read(size, buf)
        end
    end

    def self.read(io)
        elemid = readeid(io)
        elemsize = readvint(io)
        Element::Entity.new(elemid, io.pos, elemsize)
    end

    VINT_DESCRIPT = [
        # leading threashould / leading mask / rest octets
        [0x80, 0x7f,  0],
        [0x40, 0x3f,  1],
        [0x20, 0x1f,  2],
        [0x10, 0x0f,  3],
        [0x08, 0x07,  4],
        [0x04, 0x03,  5],
        [0x02, 0x01,  6],
        [0x01, 0x00,  7],
    ]
    class << VINT_DESCRIPT
        def lookup(leading)
            each do |desc|
                return desc if leading >= desc[0]
            end
            raise(WrongVariableInteger)
        end
    end

    def self.readvint(io, clearleadbit = true)
        lead = io.readbyte
        desc = VINT_DESCRIPT.lookup(lead)
        num = lead
        num &= desc[1] if clearleadbit
        rest = desc[2]
        rest.times do
            d = io.readbyte
            raise(WrongStreamEnd, nil, caller) if d < 0
            num <<= 8
            num |= d
        end
        num
    rescue Object
        STDERR.puts("offset: #{io.pos rescue -1}")
        raise
    end

    def self.readeid(io)
        readvint(io, false)
    end

    module Marker
        #define elements {
            EBML                    = 0x1a45dfa3 # container [ card:+; ] {
                EBMLVersion         = 0x4286 # uint [ def:1; ]
                EBMLReadVersion     = 0x42f7 # uint [ def:1; ]
                EBMLMaxIDLength     = 0x42f2 # uint [ def:4; ]
                EBMLMaxSizeLength   = 0x42f3 # uint [ def:8; ]
                DocType             = 0x4282 # string [ range:32..126; ]
                DocTypeVersion      = 0x4287 # uint [ def:1; ]
                DocTypeReadVersion  = 0x4285 # uint [ def:1; ]
        #    }
            CRC32                   = 0xc3 # container [ level:1..; card:*; ] {
        #       %children;
                CRC32Value          = 0x42fe # binary [ size:4; ]
        #   }
            Void                    = 0xec # binary [ level:1..; card:*; ]
        #}
    end

    include Marker
end


module Matroska
    module Marker
        # Appendix D.  Matroska DTD
        #  
        #   declare header {
        #     DocType := "matroska";
        #     EBMLVersion := 1;
        #   }
        #   define types {
        #     bool := uint [ range:0..1; ]
        #     ascii := string [ range:32..126; ]
        #   }
        #   define elements {
        #     Segment := 18538067 container [ card:*; ] {
              Segment = 0x18538067

        #       // Meta Seek Information
        #       SeekHead := 114d9b74 container [ card:*; ] {
        #         Seek := 4dbb container [ card:*; ] {
        #           SeekID := 53ab binary;
        #           SeekPosition := 53ac uint;
        #         }
        #       }
                SeekHead = 0x114d9b74
                  Seek = 0x4dbb
                    SeekID = 0x53ab
                    SeekPosition = 0x53ac

        #       // Segment Information
        #       Info := 1549a966 container [ card:*; ] {
        #         SegmentUID := 73a4 binary;
        #         SegmentFilename := 7384 string;
        #         PrevUID := 3cb923 binary;
        #         PrevFilename := 3c83ab string;
        #         NextUID := 3eb923 binary;
        #         NextFilename := 3e83bb string;
        #         TimecodeScale := 2ad7b1 uint [ def:1000000; ]
        #         Duration := 4489 float [ range:>0.0; ]
        #         DateUTC := 4461 date;
        #         Title := 7ba9 string;
        #         MuxingApp := 4d80 string;
        #         WritingApp := 5741 string;
        #       }
                Info = 0x1549a966
                  SegmentUID = 0x73a4
                  SegmentFilename = 0x7384
                  PrevUID = 0x3cb923
                  PrevFilename = 0x3c83ab
                  NextUID = 0x3eb923
                  NextFilename = 0x3e83bb
                  TimecodeScale = 0x2ad7b1
                  Duration = 0x4489
                  DateUTC = 0x4461
                  Title = 0x7ba9
                  MuxingApp = 0x4d80
                  WritingApp = 0x5741

        #       // Cluster
        #       Cluster := 1f43b675 container [ card:*; ] {
        #         Timecode := e7 uint;
        #         Position := a7 uint;
        #         PrevSize := ab uint;
        #         BlockGroup := a0 container [ card:*; ] {
        #           Block := a1 binary;
        #           BlockVirtual := a2 binary;
        #           BlockAdditions := 75a1 container {
        #             BlockMore := a6 container [ card:*; ] {
        #               BlockAddID := ee uint [ range:1..; ]
        #               BlockAdditional := a5 binary;
        #             }
        #           }
        #           BlockDuration := 9b uint [ def:TrackDuration; ];
        #           ReferencePriority := fa uint;
        #           ReferenceBlock := fb int [ card:*; ]
        #           ReferenceVirtual := fd int;
        #           CodecState := a4 binary;
        #           Slices := 8e container [ card:*; ] {
        #             TimeSlice := e8 container [ card:*; ] {
        #               LaceNumber := cc uint [ def:0; ]
        #               FrameNumber := cd uint [ def:0; ]
        #               BlockAdditionID := cb uint [ def:0; ]
        #               Delay := ce uint [ def:0; ]
        #               Duration := cf uint [ def:TrackDuration; ];
        #             }
        #           }
        #         }
        #       }
                Cluster = 0x1f43b675
                  Timecode = 0xe7
                  Position = 0xa7
                  PrevSize = 0xab
                  BlockGroup = 0xa0
                    Block = 0xa1
                    BlockVirtual = 0xa2
                    BlockAdditions = 0x75a1
                      BlockMore = 0xa6
                        BlockAddID = 0xee
                        BlockAdditional = 0xa5
                    BlockDuration = 0x9b
                    ReferencePriority = 0xfa
                    ReferenceBlock = 0xfb
                    ReferenceVirtual = 0xfd
                    CodecState = 0xa4
                    Slices = 0x8e
                      TimeSlice = 0xe8
                        LaceNumber = 0xcc
                        FrameNumber = 0xcd
                        BlockAdditionID = 0xcb
                        Delay = 0xce
                        SliceDuration = 0xcf

        #       // Track
        #       Tracks := 1654ae6b container [ card:*; ] {
        #         TrackEntry := ae container [ card:*; ] {
        #           TrackNumber := d7 uint [ range:1..; ]
        #           TrackUID := 73c5 uint [ range:1..; ]
        #           TrackType := 83 uint [ range:1..254; ]
        #           FlagEnabled := b9 uint [ range:0..1; def:1; ]
        #           FlagDefault := 88 uint [ range:0..1; def:1; ]
        #           FlagLacing  := 9c uint [ range:0..1; def:1; ]
        #           MinCache := 6de7 uint [ def:0; ]
        #           MaxCache := 6df8 uint;
        #           DefaultDuration := 23e383 uint [ range:1..; ]
        #           TrackTimecodeScale := 23314f float [ range:>0.0; def:1.0; ]
        #           Name := 536e string;
        #           Language := 22b59c string [ def:"eng"; range:32..126; ]
        #           CodecID := 86 string [ range:32..126; ];
        #           CodecPrivate := 63a2 binary;
        #           CodecName := 258688 string;
        #           CodecSettings := 3a9697 string;
        #           CodecInfoURL := 3b4040 string [ card:*; range:32..126; ]
        #           CodecDownloadURL := 26b240 string [ card:*; range:32..126; ]
        #           CodecDecodeAll := aa uint [ range:0..1; def:1; ]
        #           TrackOverlay := 6fab uint;
                Tracks = 0x1654ae6b
                  TrackEntry = 0xae
                    TrackNumber = 0xd7
                    TrackUID = 0x73c5
                    TrackType = 0x83
                    FlagEnabled = 0xb9
                    FlagDefault = 0x88
                    FlagLacing  = 0x9c
                    MinCache = 0x6de7
                    MaxCache = 0x6df8
                    DefaultDuration = 0x23e383
                    TrackTimecodeScale = 0x23314f
                    Name = 0x536e
                    Language = 0x22b59c
                    CodecID = 0x86
                    CodecPrivate = 0x63a2
                    CodecName = 0x258688
                    CodecSettings = 0x3a9697
                    CodecInfoURL = 0x3b4040
                    CodecDownloadURL = 0x26b240
                    CodecDecodeAll = 0xaa
                    TrackOverlay = 0x6fab

        #           // Video
        #           Video := e0 container {
        #             FlagInterlaced := 9a uint [ range:0..1; def:0; ]
        #             StereoMode := 53b8 uint [ range:0..3; def:0; ]
        #             PixelWidth := b0 uint [ range:1..; ]
        #             PixelHeight := ba uint [ range:1..; ]
        #             DisplayWidth := 54b0 uint [ def:PixelWidth; ]
        #             DisplayHeight := 54ba uint [ def:PixelHeight; ]
        #             DisplayUnit := 54b2 uint [ def:0; ]
        #             AspectRatioType := 54b3 uint [ def:0; ]
        #             ColourSpace := 2eb524 binary;
        #             GammaValue := 2fb523 float [ range:>0.0; ]
        #           }
                    Video = 0xe0
                      FlagInterlaced = 0x9a
                      StereoMode = 0x53b8
                      PixelWidth = 0xb0
                      PixelHeight = 0xba
                      DisplayWidth = 0x54b0
                      DisplayHeight = 0x54ba
                      DisplayUnit = 0x54b2
                      AspectRatioType = 0x54b3
                      ColourSpace = 0x2eb524
                      GammaValue = 0x2fb523

        #           // Audio
        #           Audio := e1 container {
        #             SamplingFrequency := b5 float [ range:>0.0; def:8000.0; ]
        #             OutputSamplingFrequency := 78b5 float [ range:>0.0;
        #                                                     def:8000.0; ]
        #             Channels := 94 uint [ range:1..; def:1; ]
        #             ChannelPositions := 7d7b binary;
        #             BitDepth := 6264 uint [ range:1..; ]
        #           }
                    Audio = 0xe1
                      SamplingFrequency = 0xb5
                      OutputSamplingFrequency = 0x78b5
                      Channels = 0x94
                      ChannelPositions = 0x7d7b
                      BitDepth = 0x6264

        #           // Content Encoding
        #           ContentEncodings := 6d80 container {
        #             ContentEncoding := 6240 container [ card:*; ] {
        #               ContentEncodingOrder := 5031 uint [ def:0; ]
        #               ContentEncodingScope := 5032 uint [ range:1..; def:1; ]
        #               ContentEncodingType := 5033 uint;
        #               ContentCompression := 5034 container {
        #                 ContentCompAlgo := 4254 uint [ def:0; ]
        #                 ContentCompSettings := 4255 binary;
        #               }
        #               ContentEncryption := 5035 container {
        #                 ContentEncAlgo := 47e1 uint [ def:0; ]
        #                 ContentEncKeyID := 47e2 binary;
        #                 ContentSignature := 47e3 binary;
        #                 ContentSigKeyID := 47e4 binary;
        #                 ContentSigAlgo := 47e5 uint;
        #                 ContentSigHashAlgo := 47e6 uint;
        #               }
        #             }
        #           }
                    ContentEncodings = 0x6d80
                      ContentEncoding = 0x6240
                        ContentEncodingOrder = 0x5031
                        ContentEncodingScope = 0x5032
                        ContentEncodingType = 0x5033
                        ContentCompression = 0x5034
                          ContentCompAlgo = 0x4254
                          ContentCompSettings = 0x4255
                        ContentEncryption = 0x5035
                          ContentEncAlgo = 0x47e1
                          ContentEncKeyID = 0x47e2
                          ContentSignature = 0x47e3
                          ContentSigKeyID = 0x47e4
                          ContentSigAlgo = 0x47e5
                          ContentSigHashAlgo = 0x47e6
        #         }
        #       }

        #       // Cueing Data
        #       Cues := 1c53bb6b container {
        #         CuePoint := bb container [ card:*; ] {
        #           CueTime := b3 uint;
        #           CueTrackPositions := b7 container [ card:*; ] {
        #             CueTrack := f7 uint [ range:1..; ]
        #             CueClusterPosition := f1 uint;
        #             CueBlockNumber := 5378 uint [ range:1..; def:1; ]
        #             CueCodecState := ea uint [ def:0; ]
        #             CueReference := db container [ card:*; ] {
        #               CueRefTime := 96 uint;
        #               CueRefCluster := 97 uint;
        #               CueRefNumber := 535f uint [ range:1..; def:1; ]
        #               CueRefCodecState := eb uint [ def:0; ]
        #             }
        #           }
        #         }
        #       }
                Cues = 0x1c53bb6b
                  CuePoint = 0xbb
                    CueTime = 0xb3
                    CueTrackPositions = 0xb7
                      CueTrack = 0xf7
                      CueClusterPosition = 0xf1
                      CueBlockNumber = 0x5378
                      CueCodecState = 0xea
                      CueReference = 0xdb
                        CueRefTime = 0x96
                        CueRefCluster = 0x97
                        CueRefNumber = 0x535f
                        CueRefCodecState = 0xeb

        #       // Attachment
        #       Attachments := 1941a469 container {
        #         AttachedFile := 61a7 container [ card:*; ] {
        #           FileDescription := 467e string;
        #           FileName := 466e string;
        #           FileMimeType := 4660 string [ range:32..126; ]
        #           FileData := 465c binary;
        #           FileUID := 46ae uint;
        #         }
        #       }
                Attachments = 0x1941a469
                  AttachedFile = 0x61a7
                    FileDescription = 0x467e
                    FileName = 0x466e
                    FileMimeType = 0x4660
                    FileData = 0x465c
                    FileUID = 0x46ae

        #       // Chapters
        #       Chapters := 1043a770 container {
        #         EditionEntry := 45b9 container [ card:*; ] {
        #           ChapterAtom := b6 container [ card:*; ] {
        #             ChapterUID := 73c4 uint [ range:1..; ]
        #             ChapterTimeStart := 91 uint;
        #             ChapterTimeEnd := 92 uint;
        #             ChapterFlagHidden := 98 uint [ range:0..1; def:0; ]
        #             ChapterFlagEnabled := 4598 uint [ range:0..1; def:0; ]
        #             ChapterTrack := 8f container {
        #               ChapterTrackNumber := 89 uint [ card:*; range:0..1; ]
        #               ChapterDisplay := 80 container [ card:*; ] {
        #                 ChapString := 85 string;
        #                 ChapLanguage := 437c string [ card:*; def:"eng";
        #                                               range:32..126; ]
        #                 ChapCountry := 437e string [ card:*; range:32..126; ]
        #               }
        #             }
        #           }
        #         }
        #       }
                Chapters = 0x1043a770
                  EditionEntry = 0x45b9
                    ChapterAtom = 0xb6
                      ChapterUID = 0x73c4
                      ChapterTimeStart = 0x91
                      ChapterTimeEnd = 0x92
                      ChapterFlagHidden = 0x98
                      ChapterFlagEnabled = 0x4598
                      ChapterTrack = 0x8f
                        ChapterTrackNumber = 0x89
                        ChapterDisplay = 0x80
                          ChapString = 0x85
                          ChapLanguage = 0x437c
                          ChapCountry = 0x437e

        #       // Tagging
        #       Tags := 1254c367 container [ card:*; ] {
        #         Tag := 7373 container [ card:*; ] {
        #           Targets := 63c0 container {
        #             TrackUID := 63c5 uint [ card:*; def:0; ]
        #             ChapterUID := 63c4 uint [ card:*; def:0; ]
        #             AttachmentUID := 63c6 uint [ card:*; def:0; ]
        #           }
        #           SimpleTag := 67c8 container [ card:*; ] {
        #             TagName := 45a3 string;
        #             TagString := 4487 string;
        #             TagBinary := 4485 binary;
        #           }
        #         }
        #       }
                Tags = 0x1254c367
                  Tag = 0x7373
                    Targets = 0x63c0
                      TargetTrackUID = 0x63c5
                      TargetChapterUID = 0x63c4
                      AttachmentUID = 0x63c6
                    SimpleTag = 0x67c8
                      TagName = 0x45a3
                      TagString = 0x4487
                      TagBinary = 0x4485
        #     }
        #   }

        # Copyright R 2005 - 2009 CoreCodec, Inc. All Rights Reserved.
        # Matroska, MKV, and the Matroska logos are either trademarks or registered trademarks of CoreCodec, Inc.
        # XHTML & CSS Valid.

    end

    include Marker
end
