#!/usr/local/bin/ruby
#vim: set fileencoding:utf-8

# stream: Windows上でNTFSの代替データストリーム(ADS)情報を表示する
#
# できること:
# * 引数に指定したファイルやディレクトリにくっついている代替データストリームの情報を表示する
# * 項目名のみではなく、ADSの容量やボリューム上に確保された割当量も表示する
# このくらい
#
# ファイル名の後ろにコロン『:』と名前がくっついているのがADS名になる。
# 『::$DATA』はそのファイルの実データなので気をつけるべし。
# 『rm test.txt:adsname』とするとそのADSが削除可能 (rubyのFile.unlinkでも可能)。
#
# 詳細は『代替データストリーム』『ADS』などでwwwを検索するといいでしょう。


require "dl/import"

module Win32
    module KERNEL32
        module_function
        def CreateFile(path, access, share, security, creation, flags, template)
            path1 = path.encode("utf-16le")
            path1.force_encoding("binary")
            path1 << "\0"

            file = Routine.CreateFileW(path1, access, share, security, creation, flags, template)
            raise(SystemCallError, "open error - #{path}") unless file > 0

            file
        end

        module_function
        def CloseHandle(handle)
            Routine.CloseHandle(handle)
            nil
        end

        module Routine
            extend DL::Importer

            dlload "kernel32.dll"

            # BOOL WINAPI CloseHandle(
            #   __in  HANDLE hObject
            # );
            extern "int CloseHandle(int)"

            # HANDLE WINAPI CreateFile(
            #   __in      LPCTSTR lpFileName,
            #   __in      DWORD dwDesiredAccess,
            #   __in      DWORD dwShareMode,
            #   __in_opt  LPSECURITY_ATTRIBUTES lpSecurityAttributes,
            #   __in      DWORD dwCreationDisposition,
            #   __in      DWORD dwFlagsAndAttributes,
            #   __in_opt  HANDLE hTemplateFile
            # );
            extern "int CreateFileW(void *, uint, uint, void *, uint, uint, int)"
        end
    end

    module NTDLL
        # typedef enum FILE_INFORMATION_CLASS
        # {
        #     FileStreamInformation = 22,
        # }
        FILE_STREAM_INFORMATION = 22

        module_function
        def test_ntstatus(status, frame = caller)
            status &= 0xffffffff
            raise(SystemCallError, "error (NTSTATUS: #{status} [#{status.to_s(16)}h])", frame) unless status == 0
        end

        module_function
        def NtQueryInformationFile(filehandle, iostatusblock, fileinformation, length, fileinformationclass)
            status = Routine.NtQueryInformationFile(filehandle, iostatusblock,
                                                    fileinformation, length,
                                                    fileinformationclass)
            test_ntstatus(status)
            nil
        end

        module Routine
            extend DL::Importer

            dlload "ntdll.dll"

            class FileStreamInformation < Struct.new(:next_entry_offset, # ULONG NextEntryOffset;
                                                     :name_length, # ULONG StreamNameLength;
                                                     :size, # LARGE_INTEGER StreamSize;
                                                     :allocated, # LARGE_INTEGER StreamAllocationSize;
                                                     :name) # WCHAR StreamName[1]; // 可変長 
            end

            # NTSTATUS NtQueryInformationFile(
            #   __in   HANDLE FileHandle,
            #   __out  PIO_STATUS_BLOCK IoStatusBlock,
            #   __out  PVOID FileInformation,
            #   __in   ULONG Length,
            #   __in   FILE_INFORMATION_CLASS FileInformationClass
            # );
            extern "int NtQueryInformationFile(int, void *, void *, uint, int)"
        end
    end

    module MSVCRT
        module_function
        def get_osfhandle(fd)
            file = Routine._get_osfhandle(fd)
            raise(EBADF, "wrong file descriptor") unless file > 0
            file
        end

        module Routine
            extend DL::Importer

            dlload "msvcrt.dll"

            extern "int _get_osfhandle(int)"
        end
    end

    class FileStreamEntry < Struct.new(:size, :allocated, :name)
    end

    module_function
    def FindStream(file)
        iostatusblock = "\0" * 4096
        infobuf = "\0" * 65536
        Win32::NTDLL.NtQueryInformationFile(file, iostatusblock,
                                            infobuf, infobuf.bytesize,
                                            Win32::NTDLL::FILE_STREAM_INFORMATION)
        nextentry = 1
        offset = 0
        while nextentry > 0
            entry = infobuf.unpack("@#{offset}VVQ<Q<")
            break unless entry[1] > 0
            entry << infobuf.unpack("@#{offset + 24}a#{entry[1]}")[0].encode("utf-8", "utf-16le")
            nextentry = entry[0]
            offset += nextentry
            yield(FileStreamEntry.new(entry[2], entry[3], entry[4]))
        end

        nil
    end
end

class File
    # NTFSのAlternative Data Stream(代替ストリーム)名を列挙する
    def each_stream
        Win32.FindStream(Win32::MSVCRT.get_osfhandle(fileno), &proc)
        self
    end

    def self.each_stream(path)
        file = Win32::KERNEL32.CreateFile(path, 0, 0x07, nil, 3, 0x02000000 | 0x00200000, 0)
        Win32.FindStream(file, &proc)
        self
    ensure
        Win32::KERNEL32.CloseHandle(file) if file && file > 0
    end
end



if $0 == __FILE__
    $stdout.binmode
    $stderr.binmode
    Encoding.default_internal = "utf-8"

    require "optparse"

    opt = OptionParser.new(<<-__END__, 12, "  ")
Usage: #{File.basename($0)} [options] files ...
Enumerate name(s) for alternative data stream on NTFS (and etc.).
For more information, search "NTFS alternative data stream" or "NTFS ADS" on web.
    __END__

    opt.separator("")

    verbosery = 0
    opt.on("-v", "inclement verbosery information level") { verbosery += 1 }

    opt.order!

    if ARGV.empty?
        puts opt.help
        exit 1
    end

    notshowmainstream = verbosery < 1

    printf("%-12s  %-12s  %s\n", "stream size", "allocated", "name")
    ARGV.each do |path|
        begin
            File.each_stream(path) do |e|
                name = e.name
                next if notshowmainstream && name == "::$DATA" # メインデータストリームは表示から取り除く
                name = name.sub(/:\$DATA$/, "") unless name == "::$DATA"
                printf("%12d  %12d  %s\n", e.size, e.allocated, path + name)
            end
        rescue SystemCallError
            $stderr.puts "#$0: #$! (#{path})\n"
        end
    end
end
