// encoding:utf-8

/*
 * nazuna.c - Filesystem in userspace for ruby/dokan
 *
 * - AUTHOR: dearblue <dearblue@users.sourceforge.jp>
 * - PROJECTPAGE: http://sourceforge.jp/projects/rutsubo/
 * - LICENSE: 2-clause BSD License
 */


#ifdef HAVE_DOKAN_H

#define _UNICODE
#define __MSVCRT_VERSION__ 0x0601 // for struct __stat64

#include <windows.h>
#include <winbase.h>
#include <accctrl.h>
#include <aclapi.h>
#include <dokan.h>

//#define NAZUNA_DEBUG_OUT 1
#include "nazuna.h"


// mingw32に存在しないfcntl.hの定数定義
#define O_SHLOCK        0x0010
#define O_EXLOCK        0x0020
#define O_ASYNC         0x0040
#define O_FSYNC         0x0080
#define O_SYNC          0x0080
#define O_NOFOLLOW      0x0100
#define O_NOCTTY        0x8000
#define O_DIRECT        0x00010000
#define O_EXEC          0x00040000


// ビルド時のパス漏れを防ぐ
#define __FILE__ "nazuna-dokan.c"


struct nazuna;
struct request;


enum {
    NAZUNA_STATUS_INCOMPLETE = -1,
};


typedef VALUE nazuna_dispatch_f(struct nazuna *nazuna, VALUE dokan, VALUE refs[], va_list args);

struct nazuna
{
    VALUE mounter;  // DokanMainを実行するスレッド
    VALUE dispatch; // Nazuna::*::Context.mountのdispatch引数と等価
    VALUE threads;  // I/O要求の処理スレッド (配列にスレッドを詰める)
    VALUE logger;   // デバッグ用途の記録子

    struct request *request; // I/O Request queue (片方向 list)
    struct request *request_tail; // I/O Request queue の末端

    // I/O要求queueの同期機構
    pthread_mutex_t mutex;
    pthread_cond_t cond;


    wchar_t mountpoint[PATH_MAX + 1];
    int nthread;

    DOKAN_OPTIONS opt;
};


// operation information
struct opinfo
{
    DOKAN_FILE_INFO *dokan;
    const wchar_t *path;
};



static VALUE mNazuna;       // module Nazuna

static ID IDflags, IDctime, IDatime, IDmtime, IDdev, IDsize, IDnlink, IDino;
static ID IDname, IDflags, IDctime, IDatime, IDmtime, IDsize;


// SECTION: utilities


static ID IDtv_sec;
static ID IDtv_nsec;


static const int64_t FILETIME_BASE = 116444736000000000LL; // 精度100ナノ秒のAD1601年とAD1970年との差
static const int64_t FILETIME_PRECISION = 10000000LL; // 1秒は何100ナノ秒?

static FILETIME
time2ftime(VALUE time)
{
    FILETIME ftime;
    if (NIL_P(time)) {
        ftime.dwHighDateTime = 0;
        ftime.dwLowDateTime = 0;
    } else if (rb_obj_is_kind_of(time, rb_cInteger)) {
        int64_t n = NUM2ULL(time);
        ftime.dwHighDateTime = n >> 32;
        ftime.dwLowDateTime = n;
    } else if (rb_obj_is_kind_of(time, rb_cFloat)) {
        int64_t n = NUM2DBL(time) * FILETIME_PRECISION;
        ftime.dwHighDateTime = n >> 32;
        ftime.dwLowDateTime = n;
    } else if (rb_obj_is_kind_of(time, rb_cTime)) {
        int64_t n = NUM2LL(rb_funcall(time, IDtv_sec, 0)) * FILETIME_PRECISION + FILETIME_BASE;
        n += NUM2LL(rb_funcall(time, IDtv_nsec, 0)) / 100;
        ftime.dwHighDateTime = n >> 32;
        ftime.dwLowDateTime = n;
    } else {
        rb_raise(rb_eTypeError, "wrong value for FILETIME");
    }
    return ftime;
}

static inline int64_t
filetime2int64(const FILETIME *time)
{
    return (((uint64_t)time->dwHighDateTime) << 32) | time->dwLowDateTime;
}

static VALUE
ftime2time(const FILETIME *time)
{
    int64_t n = filetime2int64(time) - FILETIME_BASE;
    int64_t seconds = n / FILETIME_PRECISION;
    double microseconds = (n % FILETIME_PRECISION) / 10.0;
    static ID IDat;
    if (!IDat) { IDat = rb_intern("at"); }
    return rb_funcall(rb_cTime, IDat, 2,
                      LL2NUM(seconds), DBL2NUM(microseconds));
}


static rb_encoding *ENCutf8p;
static VALUE ENCutf8;
static rb_encoding *ENCutf16lep;
static VALUE ENCutf16le;


static wchar_t *
str2wcs(VALUE str, wchar_t ptr[], const wchar_t *end)
{
    str = rb_str_encode(str, ENCutf16le, 0, Qnil);
    const ssize_t len = RSTRING_LEN(str) / sizeof(wchar_t);
    if (len >= (end - ptr)) { return NULL; }
    memcpy(ptr, RSTRING_PTR(str), len * sizeof(wchar_t));
    ptr[len] = L'\0';
    return ptr;
}

static wchar_t *
str2wpath(VALUE str, wchar_t ptr[], const wchar_t *end)
{
    if (!str2wcs(str, ptr, end)) { return NULL; }
    wchar_t *p;
    for (p = ptr; p < end && *p != L'\0'; p ++) {
        if (*p == L'/') { *p = L'\\'; }
    }
    return ptr;
}


static VALUE
wcs2str(const wchar_t str[])
{
    if (!str) { return Qnil; }
    VALUE v = rb_str_new((const char *)str, wcslen(str) * sizeof(str[0]));
    rb_enc_associate(v, ENCutf16lep);

    return rb_str_encode(v, ENCutf8, ECONV_INVALID_REPLACE | ECONV_UNDEF_REPLACE, Qnil);
}

static VALUE
wpath2str(const wchar_t path[])
{
    VALUE v = wcs2str(path);
    if (NIL_P(v)) { return Qnil; }
    char *p = RSTRING_PTR(v);
    const char *end = p + RSTRING_LEN(v);
    for (; p < end; p ++) {
        if (*p == '\\') { *p = '/'; }
    }
    return v;
}


// SECTION: Nazuna::Exceptions

VALUE mExceptions;          // Nazuna::Exceptions

VALUE eMountError;          // Nazuna::Exceptions::MountError
    VALUE eMountPointError; // Nazuna::Exceptions::MountPointError
    VALUE eBusyMountPoint;  // Nazuna::Exceptions::BusyMountPoint
    VALUE eDriverError;     // Nazuna::Exceptions::DriverError
VALUE eStatusError;         // Nazuna::Exceptions::StatusError
VALUE eBadReference;        // Nazuna::Exceptions::BadReference


// SECTION:

struct processinfo
{
    size_t pid;
    PTOKEN_USER tokenuser;
};

static VALUE cProcessInfo;

static void
processinfo_free(struct processinfo *pinfo)
{
    if (pinfo) {
        free(pinfo);
    }
}

static size_t
processinfo_memsize(struct processinfo *pinfo)
{
    return sizeof(struct processinfo);
}

static const rb_data_type_t processinfo_datatype = {
    .wrap_struct_name = "nazuna.processinfo",
    .function.dmark = NULL,
    .function.dfree = (void (*)(void *))processinfo_free,
    .function.dsize = (size_t (*)(const void *))processinfo_memsize,
    .parent = NULL,
    .data = NULL,
};

static inline VALUE
processinfo_new(size_t pid, PTOKEN_USER tokenuser, size_t length)
{
    struct processinfo *pinfo = (struct processinfo *)ALLOC_N(char, sizeof(struct processinfo) + length);
    pinfo->pid = pid;
    pinfo->tokenuser = (PTOKEN_USER)(pinfo + 1);
    memcpy(pinfo->tokenuser, tokenuser, length);
    return TypedData_Wrap_Struct(cProcessInfo, &processinfo_datatype, pinfo);
}

static struct processinfo *
processinfo_refp(VALUE ref)
{
    struct processinfo *pinfo = (struct processinfo *)RTYPEDDATA_DATA(ref);
    if (!pinfo) {
        rb_raise(eBadReference, "expired reference to Nazuna::ProcessInfo");
    }
    return pinfo;
}

static inline struct processinfo *
processinfo_ref(VALUE ref)
{
    rb_check_typeddata(ref, &processinfo_datatype);
    return processinfo_refp(ref);
}

static VALUE
processinfo_pid(VALUE self)
{
    struct processinfo *pinfo = processinfo_ref(self);
    return SIZET2NUM(pinfo->pid);
}

static VALUE
processinfo_userinfo(VALUE self)
{
    struct processinfo *pinfo = processinfo_ref(self);

    wchar_t accountname[256];
    wchar_t domainname[256];
    DWORD accountlen = ELEMENTOF(accountname);
    DWORD domainlen = ELEMENTOF(domainname);
    SID_NAME_USE snu;

    if (!LookupAccountSidW(NULL, pinfo->tokenuser->User.Sid,
                           accountname, &accountlen,
                           domainname, &domainlen, &snu)) {
        return Qnil;
    }

    VALUE values[2] = { wcs2str(accountname), wcs2str(domainname), };
    return rb_ary_new4(2, values);
}


// SECTION: Nazuna::~::Context::OperationInfo

static VALUE iOpInfo;

static size_t
opinfo_memsize(const void *p)
{
    return 0;
}

static const rb_data_type_t opinfo_datatype = {
    .wrap_struct_name = "nazuna.opinfo",
    .function.dmark = NULL,
    .function.dfree = NULL,
    .function.dsize = opinfo_memsize,
    .parent = NULL,
    .data = NULL,
};


static VALUE
opinfo_new(struct opinfo *info)
{
    return TypedData_Wrap_Struct(iOpInfo, &opinfo_datatype, info);
}

static struct opinfo *
opinfo_refp(VALUE accessor)
{
    struct opinfo *info = (struct opinfo *)RTYPEDDATA_DATA(accessor);
    if (info == NULL) {
        rb_raise(eBadReference, "expired reference to Nazuna::OperationInformation");
    }
    return info;
}

static struct opinfo *
opinfo_ref(VALUE accessor)
{
    rb_check_typeddata(accessor, &opinfo_datatype);
    return opinfo_refp(accessor);
}


static VALUE
opinfo_get_path(VALUE self)
{
    return wpath2str(opinfo_ref(self)->path);
}

static VALUE
opinfo_get_context(VALUE self)
{
    uint64_t context = opinfo_ref(self)->dokan->Context;
    if (context) {
        return ULL2NUM(context);
    } else {
        return Qnil;
    }
}

static VALUE
opinfo_get_process(VALUE self)
{
    PDOKAN_FILE_INFO info = opinfo_ref(self)->dokan;

    HANDLE token = DokanOpenRequestorToken(info);
    if (token == INVALID_HANDLE_VALUE) {
        return Qnil;
    }

    char buffer[1024];
    DWORD length;
    if (!GetTokenInformation(token, TokenUser, buffer, sizeof(buffer), &length)) {
        CloseHandle(token);
        return Qnil;
    }

    CloseHandle(token);

    return processinfo_new(info->ProcessId, (PTOKEN_USER)buffer, length);
}

static VALUE
opinfo_is_directory(VALUE self)
{
    return (opinfo_ref(self)->dokan->IsDirectory ? Qtrue : Qfalse);
}

static VALUE
opinfo_is_delete_on_close(VALUE self)
{
    return (opinfo_ref(self)->dokan->DeleteOnClose ? Qtrue : Qfalse);
}

static VALUE
opinfo_is_paging_io(VALUE self)
{
    return (opinfo_ref(self)->dokan->PagingIo ? Qtrue : Qfalse);
}

static VALUE
opinfo_is_synchronous_io(VALUE self)
{
    return (opinfo_ref(self)->dokan->SynchronousIo ? Qtrue : Qfalse);
}

static VALUE
opinfo_is_no_cache(VALUE self)
{
    return (opinfo_ref(self)->dokan->Nocache ? Qtrue : Qfalse);
}

static VALUE
opinfo_is_write_to_end_of_file(VALUE self)
{
    return (opinfo_ref(self)->dokan->WriteToEndOfFile ? Qtrue : Qfalse);
}


// SECTION: request

struct request
{
    pthread_mutex_t mutex;
    pthread_cond_t cond;
    struct opinfo *info;
    struct nazuna *nazuna;
    nazuna_dispatch_f *dispatch;
    const char *funcname;
    int status;
    struct request *next;
    int done;

    // dokan managed thread で受けた関数引数を ruby managed thread に渡すための変数。
    // clang にも関数内関数 (nested function) があれば幸せになれるんだけどなあ。
    va_list args;

    // ruby managed thread で使う変数
    VALUE accessor;
    VALUE *refs;
};

static inline int
setup_request(struct request *r, struct opinfo *info, nazuna_dispatch_f *dispatch, const char funcname[], va_list args)
{
    memset(r, 0, sizeof(*r));
    pthread_mutex_init(&r->mutex, NULL);
    pthread_cond_init(&r->cond, NULL);
    r->info = info;
    r->nazuna = (struct nazuna *)(uintptr_t)info->dokan->DokanOptions->GlobalContext;
    r->dispatch = dispatch;
    r->funcname = funcname;
    r->status = -1;
    r->next = NULL;
    r->done = -1;
    r->args = args;

    r->accessor = Qnil;
    r->refs = NULL;

    return 0;
}

static inline struct request *
last_request(struct request *p)
{
    if (!p) { return NULL; }
    for (; p->next; p = p->next)
        ;
    return p;
}

static void
push_request(struct request *r)
{
    if (pthread_mutex_lock(&r->mutex) == 0) {
        struct nazuna *nazuna = r->nazuna;
        if (pthread_mutex_lock(&nazuna->mutex) == 0) {
            if (nazuna->request) {
                struct request *p = last_request(nazuna->request);
                p->next = r;
            } else {
                nazuna->request = r;
            }
            pthread_cond_signal(&nazuna->cond);
            pthread_mutex_unlock(&nazuna->mutex);
        }
        pthread_cond_wait(&r->cond, &r->mutex);
        pthread_mutex_unlock(&r->mutex);
    }
}

static struct request *
pop_request_pop(struct nazuna *nazuna)
{
    if (pthread_mutex_lock(&nazuna->mutex) != 0) {
        rb_raise(rb_eRuntimeError, "%s:%d(%s): pthread_mutex_lock error", __FILE__, __LINE__, __func__);
        return NULL;
    }
    struct request *request = nazuna->request;
    if (!request) {
        pthread_cond_wait(&nazuna->cond, &nazuna->mutex);
        request = nazuna->request;
    }
    // Dokanマウントポイントが破棄される場合、pthread_cond_broadcastされるためrequestがないかもしれない
    if (request) {
        nazuna->request = request->next;
        request->next = NULL;
    }
    pthread_mutex_unlock(&nazuna->mutex);
    return request;
}

typedef VALUE thread_blocking_region_f(void *);

static inline thread_blocking_region_f *
pop_request_pop_f(struct request *(*func)(struct nazuna *))
{
    return (thread_blocking_region_f *)(func);
}

static inline struct request *
pop_request(struct nazuna *nazuna)
{
    return (struct request *)rb_thread_blocking_region(pop_request_pop_f(pop_request_pop), (void *)nazuna, RUBY_UBF_IO, NULL);
}

// I/Oりくえすとを待ち状態から復帰させる
static void
finish_request(struct request *r)
{
    if (pthread_mutex_lock(&r->mutex) == 0) {
        pthread_cond_signal(&r->cond);
        pthread_mutex_unlock(&r->mutex);
    }
}

static void
cleanup_request(struct request *r)
{
    pthread_cond_destroy(&r->cond);
    pthread_mutex_destroy(&r->mutex);
}

// rubyスレッドに処理を渡し、完了するまで待機する
// 引数のfuncnameは例外発生時に役立てる
static int
send_request(const char funcname[], nazuna_dispatch_f *dispatch, struct opinfo *info, ...)
{
    struct request r;
    va_list args;
    va_start(args, info);
    setup_request(&r, info, dispatch, funcname, args);
    push_request(&r);
    cleanup_request(&r);
    va_end(args);
    return r.status;
}

// I/Oリクエストキューをすべて破棄する
static void
clearqueue(struct nazuna *nazuna)
{
    if (pthread_mutex_lock(&nazuna->mutex) == 0) {
        struct request *r = nazuna->request;
        nazuna->request = NULL;
        while (r) {
            pthread_mutex_lock(&r->mutex);
            r->status = NAZUNA_STATUS_INCOMPLETE;
            struct request *rr = r->next;
            r->next = NULL;
            pthread_cond_signal(&r->cond);
            pthread_mutex_unlock(&r->mutex);
            r = rr;
        }
        pthread_mutex_unlock(&nazuna->mutex);
    }
}

#define SEND_REQUEST(NAME, DOKAN, PATH, ...)                        \
    ({                                                              \
        struct opinfo info = { .dokan = (DOKAN), .path = (PATH), }; \
        send_request(#NAME, wrap ## NAME, &info, ## __VA_ARGS__);   \
    })                                                              \

//#define REFPATH(REFS, PATH) (*(REFS) ++ = refpath((PATH)))


// SECTION: Nazuna::~::Context::SecurityDescriptor

static VALUE cSecDesc;
static VALUE mSecConsts;

static void
setup_securitydescriptor(VALUE iNazuna)
{
    cSecDesc = rb_define_class_under(iNazuna, "SecurityDescriptor", rb_cObject);

    mSecConsts = rb_define_module_under(cSecDesc, "Constants");
    rb_include_module(cSecDesc, mSecConsts);

    rb_define_const(mSecConsts, "OWNER", UINT2NUM(OWNER_SECURITY_INFORMATION));
    rb_define_const(mSecConsts, "GROUP", UINT2NUM(GROUP_SECURITY_INFORMATION));
    rb_define_const(mSecConsts, "DACL",  UINT2NUM(DACL_SECURITY_INFORMATION));
    rb_define_const(mSecConsts, "SACL",  UINT2NUM(SACL_SECURITY_INFORMATION));
    rb_define_const(mSecConsts, "LABEL", UINT2NUM(LABEL_SECURITY_INFORMATION));

    rb_define_const(mSecConsts, "OWNER_SECURITY_INFORMATION", UINT2NUM(OWNER_SECURITY_INFORMATION));
    rb_define_const(mSecConsts, "GROUP_SECURITY_INFORMATION", UINT2NUM(GROUP_SECURITY_INFORMATION));
    rb_define_const(mSecConsts, "DACL_SECURITY_INFORMATION",  UINT2NUM(DACL_SECURITY_INFORMATION));
    rb_define_const(mSecConsts, "SACL_SECURITY_INFORMATION",  UINT2NUM(SACL_SECURITY_INFORMATION));
    rb_define_const(mSecConsts, "LABEL_SECURITY_INFORMATION", UINT2NUM(LABEL_SECURITY_INFORMATION));
}


// SECTION: Nazuna::~::Context.mount dispatch

static ID IDCreateFile;
static ID IDOpenDirectory;
static ID IDCreateDirectory;
static ID IDCleanup;
static ID IDCloseFile;
static ID IDReadFile;
static ID IDWriteFile;
static ID IDFlushFileBuffers;
static ID IDGetFileInformation;
static ID IDFindFiles;
static ID IDFindFilesWithPattern;
static ID IDSetFileAttributes;
static ID IDSetFileTime;
static ID IDDeleteFile;
static ID IDDeleteDirectory;
static ID IDMoveFile;
static ID IDSetEndOfFile;
static ID IDSetAllocationSize;
static ID IDLockFile;
static ID IDUnlockFile;
static ID IDGetDiskFreeSpace;
static ID IDGetVolumeInformation;
static ID IDUnmount;
static ID IDGetFileSecurity;
static ID IDSetFileSecurity;


static inline int32_t
status_getint(VALUE status)
{
    if (! rb_obj_is_kind_of(status, rb_cInteger)) {
        rb_raise(rb_eTypeError, "%s", "returned status is not an Integer");
    }

    return NUM2INT(status);
}


// TODO: IMPLEMENT ME!
static VALUE
transportsecurity(VALUE security, PSECURITY_INFORMATION info, PSECURITY_DESCRIPTOR desc, ULONG ndesc, PULONG nneed)
{
    SID_IDENTIFIER_AUTHORITY worldauth = { SECURITY_WORLD_SID_AUTHORITY };
    SID_IDENTIFIER_AUTHORITY localauth = { SECURITY_LOCAL_SID_AUTHORITY };
    PSECURITY_DESCRIPTOR wdesc = NULL;
    PSID self = NULL;
    PSID everyone = NULL;
    PSID local = NULL;
    PACL acl = NULL;
    int err = 0;

    wdesc = (PSECURITY_DESCRIPTOR)LocalAlloc(LPTR, SECURITY_DESCRIPTOR_MIN_LENGTH);
    if (!wdesc) {
__LOG0();
        goto cleanup_and_error;
    }
    if (!InitializeSecurityDescriptor(wdesc, SECURITY_DESCRIPTOR_REVISION)) {
__LOG0();
        goto cleanup_and_error;
    }

    if (!AllocateAndInitializeSid(&worldauth, 1, SECURITY_WORLD_RID, 0, 0, 0, 0, 0, 0, 0, &everyone)) {
__LOG0();
        goto cleanup_and_error;
    }

    if (!AllocateAndInitializeSid(&localauth, 1, SECURITY_LOCAL_RID, 0, 0, 0, 0, 0, 0, 0, &local)) {
__LOG0();
        goto cleanup_and_error;
    }

    EXPLICIT_ACCESS ea;
    ZeroMemory(&ea, sizeof(EXPLICIT_ACCESS));
    ea.grfAccessPermissions = ~0;
    ea.grfAccessMode = SET_ACCESS;
    ea.grfInheritance = NO_INHERITANCE;
    ea.Trustee.pMultipleTrustee = NULL;
    ea.Trustee.MultipleTrusteeOperation = NO_MULTIPLE_TRUSTEE;
    ea.Trustee.TrusteeForm = TRUSTEE_IS_SID;
    ea.Trustee.TrusteeType = TRUSTEE_IS_GROUP;
    ea.Trustee.ptstrName = (LPTSTR)local;

    if (SetEntriesInAcl(1, &ea, NULL, &acl) != ERROR_SUCCESS) {
__LOG0();
        goto cleanup_and_error;
    }


#if 0
    PSID sid = (PSID)sidbuf;
    DWORD nsid = sizeof(sidbuf);
    char domain[256] = "\0";
    DWORD ndomain = sizeof(domain);
    SID_NAME_USE accounttype;
    if (!LookupAccountName(NULL, "everyone", sid, &nsid, domain, &ndomain, &accounttype)) {
        goto cleanup_and_error;
    }
#endif

#if 0
    char aclbuf[1024];
    PACL acl = (PACL)aclbuf;
    InitializeAcl(acl, sizeof(aclbuf), ACL_REVISION);
    char sidbuf[256];

    if (!AddAccessAllowedAce(acl, ACL_REVISION, ~0UL, sid)) { return INT2NUM(-1); }
#endif


    if (!SetSecurityDescriptorDacl(wdesc, TRUE, acl, FALSE)) {
__LOG0();
        goto cleanup_and_error;
    }

    if (*info & OWNER_SECURITY_INFORMATION) {
        if (!SetSecurityDescriptorOwner(wdesc, local, TRUE)) {
__LOG0();
            goto cleanup_and_error;
        }
    }

    if (*info & GROUP_SECURITY_INFORMATION) {
        if (!SetSecurityDescriptorGroup(wdesc, everyone, TRUE)) {
__LOG0();
            goto cleanup_and_error;
        }
    } 

    *nneed = GetSecurityDescriptorLength(wdesc);
    if (ndesc >= *nneed) {
        ZeroMemory(desc, ndesc);
        CopyMemory(desc, wdesc, *nneed);
        return INT2NUM(0);
    } else {
__LOG0();
        err = ERROR_INSUFFICIENT_BUFFER;
        goto cleanup_and_error;
    }

cleanup_and_error:
    if (!err) { err = GetLastError(); }

    if (wdesc) { LocalFree(wdesc); }
    if (acl) { LocalFree(acl); }
    if (everyone) { FreeSid(everyone); }
    if (self) { FreeSid(self); }
    if (local) { FreeSid(local); }

    {
        LPTSTR mesg;
        FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS, NULL, err, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR)&mesg, 0, NULL);
        LPTSTR p = mesg;
        for (; *p != '\0'; p ++) {
            if (p[0] == '\r' && p[1] == '\n' && p[2] == '\0') {
                *p = '\0';
                break;
            } else if (p[0] == '\n' && p[1] == '\0') {
                *p = '\0';
                break;
            }
        }
        __LOG("lasterror: %d (%s)", err, mesg);
        LocalFree(mesg);
    }

    return INT2NUM(-err);
}

// TODO: IMPLEMENT ME!
#warning IMPLEMENT ME! (wrapsecurity)
static VALUE
wrapsecurity(PSECURITY_INFORMATION secinfo, PSECURITY_DESCRIPTOR secdesc, ULONG secdescsize)
{
__LOG0();
    return INT2NUM(-1);
}


#define RETURN_IF_ERROR(STATUS)                               \
    ({                                                        \
        VALUE __status = (STATUS);                            \
        VALUE __status1 = utils_i_error_code(Qnil, __status); \
        if (!NIL_P(__status1)) { return __status1; }          \
        __status;                                             \
    })                                                        \


static VALUE
wrapCreateFile(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(CreateFile, nazuna, info,
                                ULONG2NUM(va_arg(args, DWORD)),
                                ULONG2NUM(va_arg(args, DWORD)),
                                ULONG2NUM(va_arg(args, DWORD)),
                                ULONG2NUM(va_arg(args, DWORD)));
    RETURN_IF_ERROR(status);
    int32_t s = status_getint(status);
    opinfo_refp(info)->dokan->Context = s;

    return INT2FIX(0);
}

static WINAPI int
dispatchCreateFile(LPCWSTR path,
                   DWORD access, DWORD share, DWORD disposition,
                   DWORD flags, PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(CreateFile, dokan, path,
                        access, share, disposition, flags);
}


static VALUE
wrapOpenDirectory(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(OpenDirectory, nazuna, info);
    RETURN_IF_ERROR(status);
    int32_t s = status_getint(status);
    opinfo_refp(info)->dokan->Context = s;
    return INT2FIX(0);
}

static WINAPI int
dispatchOpenDirectory(LPCWSTR path,
                      PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(OpenDirectory, dokan, path, NULL);
}


static VALUE
wrapCreateDirectory(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(CreateDirectory, nazuna, info,
                         wpath2str(va_arg(args, LPCWSTR)));
}

static WINAPI int
dispatchCreateDirectory(LPCWSTR path,
                        PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(CreateDirectory, dokan, NULL, path);
}


static VALUE
wrapCleanup(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(Cleanup, nazuna, info);
}

static WINAPI int
dispatchCleanup(LPCWSTR path,
                PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(Cleanup, dokan, path);
}


static VALUE
wrapCloseFile(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(CloseFile, nazuna, info);
}

static WINAPI int
dispatchCloseFile(LPCWSTR path,
                  PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(CloseFile, dokan, path, NULL);
}


static VALUE
wrapReadFile(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    DWORD size;
    VALUE status = CALLDISPATCH(ReadFile, nazuna, info,
                                LL2NUM(va_arg(args, LONGLONG)),         // offset
                                ULONG2NUM(size = va_arg(args, DWORD))); // size
    if (!rb_obj_is_kind_of(status, rb_cString)) { return status; }

    size_t s = RSTRING_LEN(status);
    if (s > size) {
        rb_raise(eStatusError, "%s", "read buffer size is too big");
    }

    char *buffer = va_arg(args, char *);
    LPDWORD nread = va_arg(args, LPDWORD);
    memcpy(buffer, RSTRING_PTR(status), s);
    if (nread) { *nread = s; }

    return INT2FIX(0);
}

static WINAPI int
dispatchReadFile(LPCWSTR path,
                 LPVOID buffer, DWORD size,
                 LPDWORD nread, LONGLONG offset,
                 PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(ReadFile, dokan, path, offset, size, buffer, nread);
}

// 評価順を左から右に固定する
#define COPYBUF(BUF, SIZE)                                  \
    ({                                                      \
        const char *_buf = (BUF);                           \
        size_t _size = (SIZE);                              \
        rb_str_buf_cat(rb_str_buf_new(_size), _buf, _size); \
    })                                                      \

static VALUE
wrapWriteFile(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    size_t size;
    VALUE status = CALLDISPATCH(WriteFile, nazuna, info,
                                LL2NUM(va_arg(args, LONGLONG)),       // offset
                                COPYBUF(va_arg(args, LPCVOID),        // buffer
                                        size = va_arg(args, DWORD))); // size
    if (!rb_obj_is_kind_of(status, rb_cFixnum)) { return status; }

    int64_t s = NUM2LL(status);
    if (s < 0) {
        return status;
    }
    if (s > size) {
        rb_raise(eStatusError, "%s", "write is buffer size over");
    }
    *va_arg(args, DWORD *) = s; // nwrote
    return INT2FIX(0);
}

static WINAPI int
dispatchWriteFile(LPCWSTR path,
                  LPCVOID buffer,
                  DWORD size,
                  LPDWORD nwrote,
                  LONGLONG offset,
                  PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(WriteFile, dokan, path, offset, buffer, size, nwrote);
}


static VALUE
wrapFlushFileBuffers(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(FlushFileBuffers, nazuna, info);
}

static WINAPI int
dispatchFlushFileBuffers(LPCWSTR path,
                         PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(FlushFileBuffers, dokan, path, NULL);
}


static VALUE
wrapGetFileInformation(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(GetFileInformation, nazuna, info);
    if (NIL_P(status) || rb_obj_is_kind_of(status, rb_cInteger)) {
        return status;
    }

    {
        LPBY_HANDLE_FILE_INFORMATION buffer = va_arg(args, LPBY_HANDLE_FILE_INFORMATION);
        uint64_t n;

        buffer->dwFileAttributes = NUM2UINT(getattr(status, IDflags));
        buffer->ftCreationTime = time2ftime(getattr(status, IDctime));
        buffer->ftLastAccessTime = time2ftime(getattr(status, IDatime));
        buffer->ftLastWriteTime = time2ftime(getattr(status, IDmtime));
        buffer->dwVolumeSerialNumber = NUM2UINT(getattr(status, IDdev));

        n = NUM2ULL(getattr(status, IDsize));
        buffer->nFileSizeHigh = (n >> 32) & 0xffffffff;
        buffer->nFileSizeLow = n & 0xffffffff;

        buffer->nNumberOfLinks = NUM2UINT(getattr(status, IDnlink));

        n = NUM2ULL(getattr(status, IDino));
        buffer->nFileIndexHigh = (n >> 32) & 0xffffffff;
        buffer->nFileIndexLow = n & 0xffffffff;

        return INT2FIX(0);
    }
}

static WINAPI int
dispatchGetFileInformation(LPCWSTR path,
                           LPBY_HANDLE_FILE_INFORMATION buffer,
                           PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(GetFileInformation, dokan, path, buffer);
}


struct findfiles_inblock
{
    PDOKAN_FILE_INFO dokan;
    PFillFindData report;
};

static VALUE
blockFindFiles(VALUE obj, struct findfiles_inblock *p, int argc, VALUE argv[])
{
    VALUE stat, name;
    switch (argc) {
    case 1:
        stat = argv[0];
        name = getattr(stat, IDname);
        break;
    case 2:
        name = argv[0];
        stat = argv[1];
        break;
    default:
        rb_scan_args(argc, argv, "11", NULL, NULL);
        name = stat = Qnil; // 到達し得ないんだけど、GCC がうるさくて・・・
        break;
    }

    WIN32_FIND_DATAW data;
    memset(&data, 0, sizeof(data));

    uint64_t n;

    data.dwFileAttributes = NUM2UINT(getattr(stat, IDflags));
    data.ftCreationTime = time2ftime(getattr(stat, IDctime));
    data.ftLastAccessTime = time2ftime(getattr(stat, IDatime));
    data.ftLastWriteTime = time2ftime(getattr(stat, IDmtime));

    n = NUM2ULL(getattr(stat, IDsize));
    data.nFileSizeHigh = (n >> 32) & 0xffffffff;
    data.nFileSizeLow = n & 0xffffffff;

    str2wpath(name, data.cFileName, data.cFileName + ELEMENTOF(data.cFileName));

    int status = p->report(&data, p->dokan);
    if (status) {
        return INT2NUM(status);
    } else {
        return Qnil;
    }
}

static VALUE
wrapFindFiles(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    struct findfiles_inblock p = {
        opinfo_refp(info)->dokan,
        va_arg(args, PFillFindData),
    };

    return CALLDISPATCH_WITHBLOCK(FindFiles, nazuna, blockFindFiles, &p,
                                  info);
}

static WINAPI int
dispatchFindFiles(LPCWSTR path,
                  PFillFindData report,
                  PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(FindFiles, dokan, path, report);
}


static VALUE
wrapSetFileAttributes(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(SetFileAttributes, nazuna, info,
                        UINT2NUM(va_arg(args, DWORD)));          // attr
}

static WINAPI int
dispatchSetFileAttributes(LPCWSTR path,
                          DWORD attr,
                          PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(SetFileAttributes, dokan, path, attr);
}


static VALUE
wrapSetFileTime(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(SetFileTime, nazuna, info,
                        ftime2time(va_arg(args, CONST FILETIME *)),  // ctime
                        ftime2time(va_arg(args, CONST FILETIME *)),  // atime
                        ftime2time(va_arg(args, CONST FILETIME *))); // mtime
}

static WINAPI int
dispatchSetFileTime(LPCWSTR path,
                    CONST FILETIME *ctime,
                    CONST FILETIME *atime,
                    CONST FILETIME *mtime,
                    PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(SetFileTime, dokan, path, ctime, atime, mtime);
}


static VALUE
wrapDeleteFile(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(DeleteFile, nazuna, info,
                        wpath2str(va_arg(args, const wchar_t *)));  // path
}

static WINAPI int
dispatchDeleteFile(LPCWSTR path,
                   PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(DeleteFile, dokan, NULL, path);
}


static VALUE
wrapDeleteDirectory(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(DeleteDirectory, nazuna, info,
                        wpath2str(va_arg(args, const wchar_t *)));  // path
}

static WINAPI int
dispatchDeleteDirectory(LPCWSTR path,
                        PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(DeleteDirectory, dokan, NULL, path);
}


static VALUE
wrapMoveFile(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(MoveFile, nazuna, info,
                        wpath2str(va_arg(args, LPCWSTR)),            // oldpath
                        wpath2str(va_arg(args, LPCWSTR)),            // newpath
                        (va_arg(args, BOOL) != 0) ? Qtrue : Qfalse); // forcereplace
}

static WINAPI int
dispatchMoveFile(LPCWSTR oldpath,
                 LPCWSTR newpath,
                 BOOL forcereplace,
                 PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(MoveFile, dokan, NULL, oldpath, newpath, forcereplace);
}


static VALUE
wrapSetEndOfFile(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(SetEndOfFile, nazuna, info,
                        LL2NUM(va_arg(args, LONGLONG)));         // length
}

static WINAPI int
dispatchSetEndOfFile(LPCWSTR path,
                     LONGLONG length,
                     PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(SetEndOfFile, dokan, path, length);
}


static VALUE
wrapSetAllocationSize(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(SetAllocationSize, nazuna, info,
                        LL2NUM(va_arg(args, LONGLONG)));         // length
}

static WINAPI int
dispatchSetAllocationSize(LPCWSTR path,
                          LONGLONG length,
                          PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(SetAllocationSize, dokan, path, length);
}


static VALUE
wrapLockFile(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(LockFile, nazuna, info,
                        LL2NUM(va_arg(args, LONGLONG)),          // offset
                        LL2NUM(va_arg(args, LONGLONG)));         // length
}

static WINAPI int
dispatchLockFile(LPCWSTR path,
                 LONGLONG offset,
                 LONGLONG length,
                 PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(LockFile, dokan, path, offset, length);
}


static VALUE
wrapUnlockFile(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(UnlockFile, nazuna, info,
                        LL2NUM(va_arg(args, LONGLONG)),          // offset
                        LL2NUM(va_arg(args, LONGLONG)));         // length
}

static WINAPI int
dispatchUnlockFile(LPCWSTR path,
                   LONGLONG offset,
                   LONGLONG length,
                   PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(UnlockFile, dokan, path, offset, length);
}


static inline void
setull(PULONGLONG dest, uint64_t num)
{
    if (dest) { *dest = num; }
}

static VALUE
wrapGetDiskFreeSpace(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(GetDiskFreeSpace, nazuna, info);
    if (! rb_obj_is_kind_of(status, rb_cArray)) { return status; }
    size_t len = RARRAY_LEN(status);
    VALUE *p = RARRAY_PTR(status);
    setull(va_arg(args, PULONGLONG), (len > 0) ? NUM2ULL(p[0]) : 0); // available
    setull(va_arg(args, PULONGLONG), (len > 1) ? NUM2ULL(p[1]) : 0); // total
    setull(va_arg(args, PULONGLONG), (len > 2) ? NUM2ULL(p[2]) : 0); // totalfree

    return INT2NUM(0);
}

static WINAPI int
dispatchGetDiskFreeSpace(PULONGLONG available,
                         PULONGLONG total,
                         PULONGLONG totalfree,
                         PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(GetDiskFreeSpace, dokan, NULL, available, total, totalfree);
}


static inline void
setwcs(VALUE str, wchar_t dest[], size_t size, const wchar_t alt[])
{
    if (!dest || !size) { return; }

    if (RTEST(str)) {
        str2wcs(str, dest, dest + size);
    } else {
        dest[0] = L'\0';
        if (alt) {
            wcsncat(dest, alt, size);
        }
    }
}

static inline void
setu32(LPDWORD dest, VALUE src)
{
    if (!dest) { return; }
    if (NIL_P(src)) {
        *dest = 0;
    } else {
        *dest = NUM2UINT(src);
    }
}

static VALUE
wrapGetVolumeInformation(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(GetVolumeInformation, nazuna, info);

    if (!rb_obj_is_kind_of(status, rb_cArray)) { return status; }
    size_t len = RARRAY_LEN(status);
    VALUE *p = RARRAY_PTR(status);

    {
        LPWSTR volumename = va_arg(args, LPWSTR);
        DWORD volumenamesize = va_arg(args, DWORD);
        setwcs((len > 0 ? p[0] : Qnil), volumename, volumenamesize, NULL);
    }

    setu32(va_arg(args, LPDWORD), (len > 1) ? p[1] : Qnil); // serial
    setu32(va_arg(args, LPDWORD), (len > 2) ? p[2] : Qnil); // maxcomponent
    setu32(va_arg(args, LPDWORD), (len > 3) ? p[3] : Qnil); // fsflags

    {
        LPWSTR fsname = va_arg(args, LPWSTR);
        DWORD fsnamesize = va_arg(args, DWORD);
        setwcs((len > 4 ? p[4] : Qnil), fsname, fsnamesize, L"nazuna");
    }

    return INT2NUM(0);
}

static WINAPI int
dispatchGetVolumeInformation(LPWSTR volumename,
                             DWORD volumenamesize,
                             LPDWORD serial,
                             LPDWORD maxcomponent,
                             LPDWORD fsflags,
                             LPWSTR fsname,
                             DWORD fsnamesize,
                             PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(GetVolumeInformation, dokan, NULL,
                        volumename, volumenamesize, serial, maxcomponent, fsflags, fsname, fsnamesize);
}


static VALUE
wrapUnmount(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    return CALLDISPATCH(Unmount, nazuna, info);
}

static WINAPI int
dispatchUnmount(PDOKAN_FILE_INFO dokan)
{
    return SEND_REQUEST(Unmount, dokan, NULL, NULL);
}


static VALUE
wrapGetFileSecurity(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(GetFileSecurity, nazuna, info);
    if (!rb_obj_is_kind_of(status, cSecDesc)) {
        return status;
    }

    PSECURITY_INFORMATION secinfo = va_arg(args, PSECURITY_INFORMATION);
    PSECURITY_DESCRIPTOR secdesc = va_arg(args, PSECURITY_DESCRIPTOR);
    ULONG secdescsize = va_arg(args, ULONG);
    PULONG needsize = va_arg(args, PULONG);
    return transportsecurity(status, secinfo, secdesc, secdescsize, needsize);
}

static WINAPI int
dispatchGetFileSecurity(LPCWSTR path,
                        PSECURITY_INFORMATION secinfo,
                        PSECURITY_DESCRIPTOR secdesc,
                        ULONG secdescsize,
                        PULONG needsize,
                        PDOKAN_FILE_INFO dokan)
{
    return -ERROR_INVALID_FUNCTION; // FIXME
    return SEND_REQUEST(GetFileSecurity, dokan, path, secinfo, secdesc, secdescsize, needsize);
}


static VALUE
wrapSetFileSecurity(struct nazuna *nazuna, VALUE info, VALUE refs[], va_list args)
{
    PSECURITY_INFORMATION secinfo = va_arg(args, PSECURITY_INFORMATION);
    PSECURITY_DESCRIPTOR secdesc = va_arg(args, PSECURITY_DESCRIPTOR);
    ULONG secdescsize = va_arg(args, ULONG);
    return CALLDISPATCH(SetFileSecurity, nazuna, info,
                        wrapsecurity(secinfo, secdesc, secdescsize));
}

static WINAPI int
dispatchSetFileSecurity(LPCWSTR path,
                        PSECURITY_INFORMATION secinfo,
                        PSECURITY_DESCRIPTOR secdesc,
                        ULONG secdescsize,
                        PDOKAN_FILE_INFO dokan)
{
    return -ERROR_INVALID_FUNCTION; // FIXME
    return SEND_REQUEST(SetFileSecurity, dokan, path, secinfo, secdesc, secdescsize);
}


DOKAN_OPERATIONS dispatch_table = {
    .CreateFile                 = dispatchCreateFile,
    .OpenDirectory              = dispatchOpenDirectory,
    .CreateDirectory            = dispatchCreateDirectory,
    .Cleanup                    = dispatchCleanup,
    .CloseFile                  = dispatchCloseFile,
    .ReadFile                   = dispatchReadFile,
    .WriteFile                  = dispatchWriteFile,
    .FlushFileBuffers           = dispatchFlushFileBuffers,
    .GetFileInformation         = dispatchGetFileInformation,
    .FindFiles                  = dispatchFindFiles,
    // NOTE: 直上のFindFilesを利用するため無効にしておく
    // .FindFilesWithPattern    = dispatchFindFilesWithPattern,
    .SetFileAttributes          = dispatchSetFileAttributes,
    .SetFileTime                = dispatchSetFileTime,
    .DeleteFile                 = dispatchDeleteFile,
    .DeleteDirectory            = dispatchDeleteDirectory,
    .MoveFile                   = dispatchMoveFile,
    .SetEndOfFile               = dispatchSetEndOfFile,
    .SetAllocationSize          = dispatchSetAllocationSize,
    .LockFile                   = dispatchLockFile,
    .UnlockFile                 = dispatchUnlockFile,
    .GetDiskFreeSpace           = dispatchGetDiskFreeSpace,
    .GetVolumeInformation       = dispatchGetVolumeInformation,
    .Unmount                    = dispatchUnmount,
    .GetFileSecurity            = dispatchGetFileSecurity,
    .SetFileSecurity            = dispatchSetFileSecurity,
};




static VALUE
dispatch_main_try_dispatch(struct request *r)
{
    VALUE status = r->dispatch(r->nazuna, r->accessor, r->refs, r->args);

    if (NIL_P(status)) {
        r->status = -ENOSYS;
    } else if (rb_obj_is_kind_of(status, rb_cInteger)) {
        r->status = NUM2INT(status);
    } else if (rb_obj_is_kind_of(status, rb_eSystemCallError)) {
        r->status = -NUM2INT(rb_funcall2(status, rb_intern("errno"), 0, NULL));
    } else if (rb_obj_is_kind_of(status, rb_cModule) &&
               RTEST(rb_class_inherited_p(status, rb_eSystemCallError))) {
        r->status = -NUM2INT(rb_const_get(status, rb_intern("Errno")));
    } else {
        rb_raise(eStatusError, "status error from %s", r->funcname);
    }

    return Qnil;
}

static VALUE
dispatch_main_try_cleanup(VALUE refs[])
{
    for (; *refs; refs ++) {
        if (!RTYPEDDATA_P(*refs)) {
            rb_bug("not a typeddata in %s(%s)\n"
                       "\tvalue: %d(%xh)\n"
                       "\ttype: %d(%xh)\n"
                       "\tclass name: %s\n",
                   __func__, __FILE__,
                   (uintptr_t)(*refs), (uintptr_t)(*refs),
                   BUILTIN_TYPE(*refs), BUILTIN_TYPE(*refs),
                   RSTRING_PTR(rb_class_name(*refs)));
        }
        RTYPEDDATA_DATA(*refs) = NULL;
    }
    return Qnil;
}

static VALUE
dispatch_main_try(struct request *r)
{
    VALUE refs[] = { opinfo_new(r->info), 0, 0, 0, 0, };
    r->accessor = refs[0];
    r->refs = refs + 1;
    rb_ensure(dispatch_main_try_dispatch, (VALUE)r,
              dispatch_main_try_cleanup, (VALUE)refs);
    r->done = 1;
    return Qnil;
}

static VALUE
dispatch_main_cleanup(struct request *r)
{
    if (!r->done) {
        if (r->nazuna->opt.MountPoint) {
            DokanRemoveMountPoint(r->nazuna->opt.MountPoint);
            r->nazuna->opt.MountPoint = NULL;
        }

        VALUE e = rb_errinfo();
        if (RTEST(e)) {
            VALUE bt = rb_funcall2(e, rb_intern("backtrace"), 0, NULL);
            if (NIL_P(bt)) {
                bt = rb_ary_new();
                rb_funcall2(e, rb_intern("set_backtrace"), 1, &bt);
            }
            rb_ary_push(bt, rb_sprintf("nazuna.c:in operation dispatch `%s'",
                                       r->funcname));
        }
    }
    finish_request(r);

    return Qnil;
}

static VALUE
dispatch_main(struct nazuna *nazuna)
{
    struct request *r = NULL;
    while ((r = pop_request(nazuna))) {
        r->done = 0;
        rb_ensure(dispatch_main_try, (VALUE)r,
                  dispatch_main_cleanup, (VALUE)r);
    }

    return Qnil;
}


// SECTION: Nazuna~::Constants

static VALUE iConstants; // ::Nazuna~::Constants


// SECTION: Nazuna~::Status

static VALUE mStatus;             // Nazuna~::Status
static VALUE mSuccess;            // Nazuna~::Status::Success
static VALUE mError;              // Nazuna~::Status::Error
static VALUE mDriveLetterError;   // Nazuna~::Status::DriveLetterError
static VALUE mDriverInstallError; // Nazuna~::Status::DriverInstallError
static VALUE mStartError;         // Nazuna~::Status::StartError
static VALUE mMountError;         // Nazuna~::Status::MountError
static VALUE mMountPointError;    // Nazuna~::Status::MountPointError


// SECTION: Nazuna~::Context

//static VALUE cContext;          // Nazuna~::Context

static inline const char *
dokanstatusmessage(int status)
{
    switch (status) {
    case DOKAN_SUCCESS:              return "Success";
    case DOKAN_ERROR:                return "General Error";
    case DOKAN_DRIVE_LETTER_ERROR:   return "Bad Drive letter";
    case DOKAN_DRIVER_INSTALL_ERROR: return "Can't install driver";
    case DOKAN_START_ERROR:          return "Driver something wrong";
    case DOKAN_MOUNT_ERROR:          return "Can't assign a drive letter or mount point";
    case DOKAN_MOUNT_POINT_ERROR:    return "Mount point is invalid";
    default:                         return "unknown error";
    }
}

// FIXME: もうちょっと変換先例外を整理したい
static inline VALUE
status2except(int status)
{
    switch (status) {
    case DOKAN_SUCCESS:              return Qnil;
    case DOKAN_ERROR:                return eMountError;
    case DOKAN_DRIVE_LETTER_ERROR:   return eMountError;
    case DOKAN_DRIVER_INSTALL_ERROR: return eDriverError;
    case DOKAN_START_ERROR:          return eDriverError;
    case DOKAN_MOUNT_ERROR:          return eMountError;
    case DOKAN_MOUNT_POINT_ERROR:    return eMountPointError;
    default:                         return eMountError;
    }
}




struct scan_hash
{
    const char *name;
    VALUE *ref;
    VALUE defaultvalue;
};

static inline void
scan_hash(VALUE hash, size_t nums, struct scan_hash params[])
{
    struct scan_hash *p = params;
    if (NIL_P(hash)) {
        for (; nums > 0; nums --) {
            *p->ref = p->defaultvalue;
        }
    } else {
        hash = rb_funcall3(hash, rb_intern("to_hash"), 0, 0);
        for (; nums > 0; nums --, p ++) {
            *p->ref = rb_hash_lookup2(hash, ID2SYM(rb_intern(p->name)), p->defaultvalue);
        }
    }
}

#define SCAN_HASH(HASH, ...)                            \
    ({                                                  \
        struct scan_hash _params[] = { __VA_ARGS__ };   \
        scan_hash((HASH), ELEMENTOF(_params), _params); \
    })                                                  \

#define SCAN_HASH_PARAM(NAME, DEST, DEFAULT) { (NAME), (DEST), (DEFAULT), }

static void
setup_context(struct nazuna *nazuna, int argc, VALUE argv[])
{
    VALUE dispatch, mountpoint, opts;
    rb_scan_args(argc, argv, "2:", &dispatch, &mountpoint, &opts);

    if (rb_safe_level() > 0 && OBJ_TAINTED(mountpoint)) {
        rb_raise(rb_eSecurityError, "mountpoint object is tainted");
    }

    VALUE nthread, network, removable, logger;
    SCAN_HASH(opts,
              SCAN_HASH_PARAM("nthread",   &nthread,   INT2FIX(5)),
              SCAN_HASH_PARAM("network",   &network,   Qfalse),
              SCAN_HASH_PARAM("removable", &removable, Qfalse),
              SCAN_HASH_PARAM("logger",    &logger,    Qnil));

    str2wpath(mountpoint, nazuna->mountpoint,
              nazuna->mountpoint + ELEMENTOF(nazuna->mountpoint));

    nazuna->nthread = NUM2INT(nthread);
    if (nazuna->nthread < 1) {
        nazuna->nthread = 5;
    } else if (nazuna->nthread > 20) {
        nazuna->nthread = 20;
    }

    nazuna->opt.GlobalContext = (ULONG64)(uintptr_t)(nazuna);
    nazuna->opt.Version = DOKAN_VERSION;
    nazuna->opt.ThreadCount = nazuna->nthread;
    nazuna->opt.MountPoint = nazuna->mountpoint;

    if (RTEST(removable)) { nazuna->opt.Options |= DOKAN_OPTION_REMOVABLE; }
    if (RTEST(network)) { nazuna->opt.Options |= DOKAN_OPTION_NETWORK; }

    nazuna->mounter = Qnil;
    nazuna->dispatch = dispatch;
    nazuna->threads = rb_ary_new();
    nazuna->logger = logger;

    nazuna->request = nazuna->request_tail = NULL;

    pthread_mutex_init(&nazuna->mutex, NULL);
    pthread_cond_init(&nazuna->cond, NULL);
}



static VALUE
mount_main_try_mount(struct nazuna *nazuna)
{
    int status = DokanMain(&nazuna->opt, &dispatch_table);
    nazuna->opt.MountPoint = NULL;

    return status;
}

static VALUE
mount_main_try(struct nazuna *nazuna)
{
    int i;
    for (i = nazuna->nthread; i > 0; i --) {
        rb_ary_push(nazuna->threads, rb_thread_create(dispatch_main, nazuna));
    }

    int status = rb_thread_blocking_region((VALUE (*)(void *))mount_main_try_mount,
                                           (void *)nazuna, RUBY_UBF_IO, NULL);

    if (status != DOKAN_SUCCESS) {
        const char *mesg = dokanstatusmessage(status);
        VALUE except = status2except(status);
        rb_raise(except, "%s (status: %d)", mesg, status);
    }

    return Qnil;
}

static VALUE
mount_main_cleanup_rescue(VALUE *err)
{
    if (!RTEST(*err)) {
        *err = rb_errinfo();
    }
    return Qnil;
}

static VALUE
mount_main_cleanup(struct nazuna *nazuna)
{
    VALUE *p = RARRAY_PTR(nazuna->threads);
    VALUE *end = p + RARRAY_LEN(nazuna->threads);
    for (; p < end; p ++) {
        rb_thread_kill(*p);
    }

    clearqueue(nazuna);

    pthread_mutex_lock(&nazuna->mutex);
    pthread_cond_broadcast(&nazuna->cond);
    pthread_mutex_unlock(&nazuna->mutex);

    VALUE err = Qnil;
    p = RARRAY_PTR(nazuna->threads);
    end = p + RARRAY_LEN(nazuna->threads);
    for (; p < end; p ++) {
        rb_rescue2(thread_join, (VALUE)*p,
                   mount_main_cleanup_rescue, (VALUE)&err,
                   rb_cObject, NULL);
    }

    pthread_cond_destroy(&nazuna->cond);
    pthread_mutex_destroy(&nazuna->mutex);

    nazuna->opt.MountPoint = NULL;

    if (RTEST(err)) { rb_exc_raise(err); }

    return Qnil;
}

static VALUE
mount_main(struct nazuna *nazuna)
{
    rb_ensure(mount_main_try, (VALUE)nazuna, mount_main_cleanup, (VALUE)nazuna);

    return Qnil;
}


static VALUE
context_i_mount_unmount(struct nazuna *nazuna)
{
    if (RTEST(nazuna->mounter)) {
        const wchar_t *mountpoint = nazuna->opt.MountPoint;
        nazuna->opt.MountPoint = NULL;
        if (mountpoint) {
            DokanRemoveMountPoint(mountpoint);
        }

        rb_thread_kill(nazuna->mounter);
        thread_join(nazuna->mounter);
    }

    return Qnil;
}

/*
 * call-seq:
 * Nazuna.mount(dispatch, mointpoint, nthread: 5, removable: false, network: false, safelevel: nil, logger: nil)
 *
 * ※ このメソッドは Nazuna 特異クラスの特異メソッドとして定義されます。上記の方法で直接呼び出すことは出来ません。
 *
 * 指定 mountpoint へマウントします。I/O 要求は dispatch オブジェクトの各メソッドを呼び出すことで受け渡されます。
 *
 * [RETURN]     常に nil が返ります。
 *
 * [mountpoint] ファイルシステム接続先パス位置を指定します。
 *
 *              汚染オブジェクトであれば $SAFE > 0 でセキュリティ例外が発生します。
 *
 * [dispatch]   ファイルシステムとしての I/O 要求を処理するオブジェクトを指定します。
 *
 * [nthread]    I/O要求処理のスレッド数を指定します。
 *
 * [removable]  着脱式デバイスとして認識させます。 FDD や CD/DVD ドライブのようなものです。
 *
 * [network]    ネットワークストレージとして認識させます。nfs や smbfs のようなものです。
 *
 * [safelevel]  I/O 要求処理スレッドのセーフレベルを指定します。既定値は #mount 呼び出し時の $SAFE。
 *
 * [logger]     I/O 要求や、処理完了時の戻り値を確認できます。デバッグ用途を想定しています。
 *
 *              <tt>yield(event, request, *args)</tt> のかたちでメソッドが呼ばれます。
 *
 * [EXCEPTION]  モジュール Nazuna::Exceptions 内で定義されているすべての例外と SystemCallError が発生します。
 *
 *              Nazuna の例外は ''rescue Nazuna::Exceptions'' でまとめて受けることが出来ます。
 *
 *              個々の例外についてはモジュール Nazuna::Exceptions 内にまとめてあります。
 *
 *              その他には、dispatch オブジェクトの I/O 処理メソッドで発生した例外もそのまま通知されます ($DEBUG の有効・無効は問いません)。
 */
static VALUE
nazuna_i_mount(int argc, VALUE argv[], VALUE klass)
{
    rb_secure(2);

    struct nazuna nazuna;
    memset(&nazuna, 0, sizeof(nazuna));
    setup_context(&nazuna, argc, argv);

    nazuna.mounter = rb_thread_create(mount_main, &nazuna);

    rb_ensure(thread_join, nazuna.mounter,
              context_i_mount_unmount, (VALUE)&nazuna);

    return Qnil;
}


static int
map_winerr(int err)
{
    switch (err) {
    case NOERROR:         return 0;
    case EPERM:           return ERROR_NOACCESS;
    case ENOENT:          return ERROR_FILE_NOT_FOUND;
    case ESRCH:           return -1;
    case EINTR:           return ERROR_OPERATION_ABORTED;
    case EIO:             return ERROR_IO_DEVICE;
    case ENXIO:           return ERROR_NOT_READY;
    case E2BIG:           return ERROR_BAD_ENVIRONMENT;
    case ENOEXEC:         return ERROR_BAD_FORMAT;
    case EBADF:           return ERROR_INVALID_HANDLE;
    case ECHILD:          return ERROR_CHILD_NOT_COMPLETE;
    case EAGAIN:          return ERROR_NOT_READY;
    case ENOMEM:          return ERROR_NOT_ENOUGH_MEMORY;
    case EACCES:          return ERROR_ACCESS_DENIED;
    case EFAULT:          return ERROR_NOACCESS;
    case EBUSY:           return ERROR_BUSY;
    case EEXIST:          return ERROR_FILE_EXISTS;
    case EXDEV:           return ERROR_NOT_SAME_DEVICE;
    case ENODEV:          return ERROR_BAD_UNIT;
    case ENOTDIR:         return ERROR_DIRECTORY;
    case EISDIR:          return ERROR_DIRECTORY;
    case EINVAL:          return ERROR_INVALID_PARAMETER;
    case ENFILE:          return ERROR_SHARING_BUFFER_EXCEEDED;
    case EMFILE:          return ERROR_TOO_MANY_OPEN_FILES;
    case ENOTTY:          return -1;
    case EFBIG:           return -1;
    case ENOSPC:          return ERROR_DISK_FULL;
    case ESPIPE:          return ERROR_SEEK_ON_DEVICE;
    case EROFS:           return ERROR_WRITE_PROTECT;
    case EMLINK:          return -1;
    case EPIPE:           return ERROR_BAD_PIPE;
    case EDOM:            return -1;
    case ERANGE:          return ERROR_ARITHMETIC_OVERFLOW;
    case EDEADLK:         return -1;
    case ENAMETOOLONG:    return ERROR_FILENAME_EXCED_RANGE;
    case ENOLCK:          return -1;
    case ENOSYS:          return ERROR_CALL_NOT_IMPLEMENTED;
    case ENOTEMPTY:       return ERROR_DIR_NOT_EMPTY;
    case ELOOP:           return WSAELOOP;
    case EWOULDBLOCK:     return WSAEWOULDBLOCK;
    case EREMOTE:         return WSAEREMOTE;
    case EILSEQ:          return -1;
    case EUSERS:          return WSAEUSERS;
    case ENOTSOCK:        return WSAENOTSOCK;
    case EDESTADDRREQ:    return WSAEDESTADDRREQ;
    case EMSGSIZE:        return WSAEMSGSIZE;
    case EPROTOTYPE:      return WSAEPROTOTYPE;
    case ENOPROTOOPT:     return WSAENOPROTOOPT;
    case EPROTONOSUPPORT: return WSAEPROTONOSUPPORT;
    case ESOCKTNOSUPPORT: return WSAESOCKTNOSUPPORT;
    case EOPNOTSUPP:      return WSAEOPNOTSUPP;
    case EPFNOSUPPORT:    return WSAEPFNOSUPPORT;
    case EAFNOSUPPORT:    return WSAEAFNOSUPPORT;
    case EADDRINUSE:      return WSAEADDRINUSE;
    case EADDRNOTAVAIL:   return WSAEADDRNOTAVAIL;
    case ENETDOWN:        return WSAENETDOWN;
    case ENETUNREACH:     return WSAENETUNREACH;
    case ENETRESET:       return WSAENETRESET;
    case ECONNABORTED:    return WSAECONNABORTED;
    case ECONNRESET:      return WSAECONNRESET;
    case ENOBUFS:         return WSAENOBUFS;
    case EISCONN:         return WSAEISCONN;
    case ENOTCONN:        return WSAENOTCONN;
    case ESHUTDOWN:       return WSAESHUTDOWN;
    case ETOOMANYREFS:    return WSAETOOMANYREFS;
    case ETIMEDOUT:       return WSAETIMEDOUT;
    case ECONNREFUSED:    return WSAECONNREFUSED;
    case EHOSTDOWN:       return WSAEHOSTDOWN;
    case EHOSTUNREACH:    return WSAEHOSTUNREACH;
    case EALREADY:        return WSAEALREADY;
    case EINPROGRESS:     return WSAEINPROGRESS;
    case ESTALE:          return WSAESTALE;
    case EDQUOT:          return WSAEDQUOT;
    case EPROCLIM:        return WSAEPROCLIM;
    default:              return -1;
    }
}

/*
 * call-seq:
 * Nazuna::Utils.map_winerr(errno) -> winerr
 *
 * errno を Windows error code に変換します。
 *
 * [RETURN]     Win32 API の ステータスコード
 *
 * [errno]      標準 C 言語レベルでの errno
 *
 * [EXCEPTIONS] 引数 errno が整数値に変換出来ない場合、TypeError が発生します。
 *
 *              また、32ビットで表現できる無符号整数に変換できない場合は RangeError が発生します。
 */
static VALUE
utils_i_map_winerr(VALUE obj, VALUE err)
{
    int e = map_winerr(NUM2UINT(err));
    if (e < 0) {
        return INT2NUM(ERROR_INVALID_FUNCTION);
    } else {
        return INT2NUM(e);
    }
}

static VALUE
utils_i_map_file_mode(VALUE obj, VALUE vaccess, VALUE vshare, VALUE vdisposition, VALUE vflags)
{
    uint32_t mode = 0;
    uint32_t access = NUM2UINT(vaccess);
    uint32_t share = NUM2UINT(vshare);
    uint32_t disposition = NUM2UINT(vdisposition);
    uint32_t flags = NUM2UINT(vflags);

    switch (access & (FILE_READ_DATA | FILE_WRITE_DATA)) {
    case 0:
        mode |= O_RDONLY ^ O_WRONLY ^ O_RDWR;
        break;
    case FILE_READ_DATA:
        mode |= O_RDONLY;
        break;
    case FILE_WRITE_DATA:
        mode |= O_WRONLY;
        break;
    case (FILE_READ_DATA | FILE_WRITE_DATA):
        mode |= O_RDWR;
        break;
    }


//#if 0
// #define DELETE                           (0x00010000L)
// #define READ_CONTROL                     (0x00020000L)
// #define WRITE_DAC                        (0x00040000L)
// #define WRITE_OWNER                      (0x00080000L)
// 
// #define STANDARD_RIGHTS_REQUIRED         (0x000F0000L)
// 
// #define STANDARD_RIGHTS_READ             (READ_CONTROL)
// #define STANDARD_RIGHTS_WRITE            (READ_CONTROL)
// #define STANDARD_RIGHTS_EXECUTE          (READ_CONTROL)
// 
// #define STANDARD_RIGHTS_ALL              (0x001F0000L)
// 
// #define SPECIFIC_RIGHTS_ALL              (0x0000FFFFL)

    if ((access & GENERIC_EXECUTE) == GENERIC_EXECUTE) {
        mode |= O_EXEC;
    }

//           O_NONBLOCK      オープンするときにブロックしない

//           O_APPEND        書み込みのたびに末尾に追加する

    switch (disposition) {
    case CREATE_ALWAYS:
    case OPEN_ALWAYS:
        mode |= O_CREAT;
        break;
    case TRUNCATE_EXISTING:
        mode |= O_TRUNC;
        break;
    case CREATE_NEW:
        mode |= O_EXCL;
        break;
    case OPEN_EXISTING:
        // TODO: どうするべきかな。。。
        break;
    default:
        // TODO:
        break;
    }


//share   FILE_SHARE_WRITE    0x00000002
//share   FILE_SHARE_DELETE   0x00000004

    if ((share & FILE_SHARE_READ) == FILE_SHARE_READ) {
        mode |= O_SHLOCK;
    }

    if ((share & (FILE_SHARE_WRITE | FILE_SHARE_DELETE | FILE_SHARE_READ)) == 0) {
        mode |= O_EXLOCK;
    }

    if ((flags & FILE_FLAG_NO_BUFFERING) == FILE_FLAG_NO_BUFFERING) {
        mode |= O_DIRECT;
    }

    if ((access & SYNCHRONIZE) == SYNCHRONIZE) {
        mode |= O_FSYNC;
    }

//#define O_NOFOLLOW
    if ((flags & FILE_FLAG_OPEN_REPARSE_POINT) & FILE_FLAG_OPEN_REPARSE_POINT) {
        mode |= O_NOFOLLOW;
    }

//           O_DIRECTORY     ファイルがディレクトリでなければ、エラーとする
//           O_CLOEXEC       オープン時に FD_CLOEXEC を設定する

//    if ((flags & O_ASYNC) == O_ASYNC)
//      dwOpenMode |= FILE_FLAG_OVERLAPPED;

//#endif
    return UINT2NUM(mode);
}

static VALUE
utils_i_map_create_flags(VALUE obj, VALUE vaccess, VALUE vshare, VALUE vdisposition, VALUE vflags)
{
// flags   FILE_ATTRIBUTE_ARCHIVE          0x00000020
// flags   FILE_ATTRIBUTE_ENCRYPTED        0x00004000
// flags   FILE_ATTRIBUTE_HIDDEN           0x00000002
// flags   FILE_ATTRIBUTE_NORMAL           0x00000080
// flags   FILE_ATTRIBUTE_OFFLINE          0x00001000
// flags   FILE_ATTRIBUTE_READONLY         0x00000001
// flags   FILE_ATTRIBUTE_SYSTEM           0x00000004
// flags   FILE_ATTRIBUTE_TEMPORARY        0x00000100
// flags   FILE_FLAG_BACKUP_SEMANTICS      0x02000000
// flags   FILE_FLAG_DELETE_ON_CLOSE       0x04000000
// flags   FILE_FLAG_OPEN_NO_RECALL        0x00100000
// flags   FILE_FLAG_OVERLAPPED            0x40000000
// flags   FILE_FLAG_POSIX_SEMANTICS       0x01000000
// flags   FILE_FLAG_RANDOM_ACCESS         0x10000000
// flags   FILE_FLAG_SEQUENTIAL_SCAN       0x08000000
// flags   FILE_FLAG_WRITE_THROUGH         0x80000000
    return Qnil;
}


// SECTION: Nazuna

void
Init_nazuna(void)
{
    mNazuna = rb_define_module("Nazuna");
    rb_define_const(mNazuna, "Nazuna", mNazuna); // Nazuna定数が再定義されても、内部参照で狂わないように

    VALUE mConstants = rb_define_module_under(mNazuna, "Constants");
    rb_include_module(mNazuna, mConstants);
    rb_define_const(mConstants, "BLOCK_SIZE", INT2FIX(512));

    mExceptions = rb_define_module_under(mNazuna, "Exceptions");
    rb_include_module(mNazuna, mExceptions);

    eMountError = rb_define_class_under(mExceptions, "MountError", rb_eStandardError);
    rb_include_module(eMountError, mExceptions);

    eMountPointError = rb_define_class_under(mExceptions, "MountPointError", eMountError);
    rb_include_module(eMountError, mExceptions);

    eBusyMountPoint = rb_define_class_under(mExceptions, "BusyMountPoint", eMountError);
    rb_include_module(eMountError, mExceptions);

    eDriverError = rb_define_class_under(mExceptions, "DriverError", eMountError);
    rb_include_module(eMountError, mExceptions);

    eStatusError = rb_define_class_under(mExceptions, "StatusError", rb_eStandardError);
    rb_include_module(eStatusError, mExceptions);

    eBadReference = rb_define_class_under(mExceptions, "BadReference", rb_eStandardError);
    rb_include_module(eBadReference, mExceptions);


    IDflags = rb_intern("flags");
    IDctime = rb_intern("ctime");
    IDatime = rb_intern("atime");
    IDmtime = rb_intern("mtime");
    IDdev = rb_intern("dev");
    IDsize = rb_intern("size");
    IDnlink = rb_intern("nlink");
    IDino = rb_intern("ino");

    IDname = rb_intern("name");

    IDtv_sec = rb_intern("tv_sec");
    IDtv_nsec = rb_intern("tv_nsec");

    IDCreateFile           = rb_intern("CreateFile");
    IDOpenDirectory        = rb_intern("OpenDirectory");
    IDCreateDirectory      = rb_intern("CreateDirectory");
    IDCleanup              = rb_intern("Cleanup");
    IDCloseFile            = rb_intern("CloseFile");
    IDReadFile             = rb_intern("ReadFile");
    IDWriteFile            = rb_intern("WriteFile");
    IDFlushFileBuffers     = rb_intern("FlushFileBuffers");
    IDGetFileInformation   = rb_intern("GetFileInformation");
    IDFindFiles            = rb_intern("FindFiles");
    IDFindFilesWithPattern = rb_intern("FindFilesWithPattern");
    IDSetFileAttributes    = rb_intern("SetFileAttributes");
    IDSetFileTime          = rb_intern("SetFileTime");
    IDDeleteFile           = rb_intern("DeleteFile");
    IDDeleteDirectory      = rb_intern("DeleteDirectory");
    IDMoveFile             = rb_intern("MoveFile");
    IDSetEndOfFile         = rb_intern("SetEndOfFile");
    IDSetAllocationSize    = rb_intern("SetAllocationSize");
    IDLockFile             = rb_intern("LockFile");
    IDUnlockFile           = rb_intern("UnlockFile");
    IDGetDiskFreeSpace     = rb_intern("GetDiskFreeSpace");
    IDGetVolumeInformation = rb_intern("GetVolumeInformation");
    IDUnmount              = rb_intern("Unmount");
    IDGetFileSecurity      = rb_intern("GetFileSecurity");
    IDSetFileSecurity      = rb_intern("SetFileSecurity");


    ENCutf8p = rb_enc_find("UTF-8");
    ENCutf8 = rb_enc_from_encoding(ENCutf8p);
    rb_gc_register_address(&ENCutf8);

    ENCutf16lep = rb_enc_find("UTF-16LE");
    ENCutf16le = rb_enc_from_encoding(ENCutf16lep);
    rb_gc_register_address(&ENCutf16le);

    /*
     * Document-class: Nazuna
     *
     * nazuna 舞台裏で利用される。
     */

    VALUE iNazuna = rb_singleton_class(mNazuna);
    rb_define_const(iNazuna, "Nazuna", mNazuna); // Nazuna定数が再定義されても、内部参照で狂わないように

    /*
     * Document-class: Nazuna::Dokan
     *
     * nazuna 舞台裏で利用される実装本体。
     *
     * Windows 向けには Dokan-lib を用いて実装される。
     */

    VALUE iDokan = rb_define_module_under(iNazuna, "Dokan");
    rb_define_singleton_method(iDokan, "mount", RUBY_METHOD_FUNC(nazuna_i_mount), -1);

    rb_include_module(iNazuna, iDokan);

    /*
     * Document-class: Nazuna::Dokan::Utils
     *
     * nazuna 舞台裏で利用される実装本体。
     *
     * Windows 向けには Dokan-lib を用いて実装される。
     */

    VALUE iUtils = rb_define_module_under(iDokan, "Utils");
    rb_include_module(iDokan, iUtils);
    rb_define_module_function(iUtils, "error_code", RUBY_METHOD_FUNC(utils_i_error_code), 1);
    rb_define_module_function(iUtils, "error?", RUBY_METHOD_FUNC(utils_i_is_error), 1);
    rb_define_module_function(iUtils, "map_winerr", RUBY_METHOD_FUNC(utils_i_map_winerr), 1);
    rb_define_module_function(iUtils, "map_file_mode", RUBY_METHOD_FUNC(utils_i_map_file_mode), 4);
    rb_define_module_function(iUtils, "map_create_flags", RUBY_METHOD_FUNC(utils_i_map_create_flags), 4);

    /*
     * Document-class: Nazuna::Dokan::Constants
     *
     * Nazuna舞台裏で利用される、定数群。
     *
     * Windows上で利用される定数が定義される。
     */

    iConstants = rb_define_module_under(iDokan, "Constants");
    rb_include_module(iNazuna, iConstants);
    rb_include_module(iDokan, iConstants);
    rb_define_const(iConstants, "IMPLEMENT", rb_str_new_cstr("nazuna-dokan"));

    rb_define_const(iConstants, "DOKAN_VERSION",           INT2FIX(DOKAN_VERSION));
    rb_define_const(iConstants, "DOKAN_OPTION_DEBUG",      INT2FIX(DOKAN_OPTION_DEBUG));
    rb_define_const(iConstants, "DOKAN_OPTION_STDERR",     INT2FIX(DOKAN_OPTION_STDERR));
    rb_define_const(iConstants, "DOKAN_OPTION_ALT_STREAM", INT2FIX(DOKAN_OPTION_ALT_STREAM));
    rb_define_const(iConstants, "DOKAN_OPTION_KEEP_ALIVE", INT2FIX(DOKAN_OPTION_KEEP_ALIVE));
    rb_define_const(iConstants, "DOKAN_OPTION_NETWORK",    INT2FIX(DOKAN_OPTION_NETWORK));
    rb_define_const(iConstants, "DOKAN_OPTION_REMOVABLE",  INT2FIX(DOKAN_OPTION_REMOVABLE));

    rb_define_const(iConstants, "DELETE",                             UINT2NUM(DELETE));
    rb_define_const(iConstants, "READ_CONTROL",                       UINT2NUM(READ_CONTROL));
    rb_define_const(iConstants, "WRITE_DAC",                          UINT2NUM(WRITE_DAC));
    rb_define_const(iConstants, "WRITE_OWNER",                        UINT2NUM(WRITE_OWNER));
    rb_define_const(iConstants, "SYNCHRONIZE",                        UINT2NUM(SYNCHRONIZE));
    rb_define_const(iConstants, "STANDARD_RIGHTS_REQUIRED",           UINT2NUM(STANDARD_RIGHTS_REQUIRED));
    rb_define_const(iConstants, "STANDARD_RIGHTS_READ",               UINT2NUM(STANDARD_RIGHTS_READ));
    rb_define_const(iConstants, "STANDARD_RIGHTS_WRITE",              UINT2NUM(STANDARD_RIGHTS_WRITE));
    rb_define_const(iConstants, "STANDARD_RIGHTS_EXECUTE",            UINT2NUM(STANDARD_RIGHTS_EXECUTE));
    rb_define_const(iConstants, "STANDARD_RIGHTS_ALL",                UINT2NUM(STANDARD_RIGHTS_ALL));
    rb_define_const(iConstants, "SPECIFIC_RIGHTS_ALL",                UINT2NUM(SPECIFIC_RIGHTS_ALL));
    rb_define_const(iConstants, "ACCESS_SYSTEM_SECURITY",             UINT2NUM(ACCESS_SYSTEM_SECURITY));
    rb_define_const(iConstants, "MAXIMUM_ALLOWED",                    UINT2NUM(MAXIMUM_ALLOWED));
    rb_define_const(iConstants, "GENERIC_READ",                       UINT2NUM(GENERIC_READ));
    rb_define_const(iConstants, "GENERIC_WRITE",                      UINT2NUM(GENERIC_WRITE));
    rb_define_const(iConstants, "GENERIC_EXECUTE",                    UINT2NUM(GENERIC_EXECUTE));
    rb_define_const(iConstants, "GENERIC_ALL",                        UINT2NUM(GENERIC_ALL));

    rb_define_const(iConstants, "FILE_SHARE_DELETE",                  UINT2NUM(FILE_SHARE_DELETE));
    rb_define_const(iConstants, "FILE_SHARE_READ",                    UINT2NUM(FILE_SHARE_READ));
    rb_define_const(iConstants, "FILE_SHARE_WRITE",                   UINT2NUM(FILE_SHARE_WRITE));
    rb_define_const(iConstants, "CREATE_NEW",                         UINT2NUM(CREATE_NEW));
    rb_define_const(iConstants, "CREATE_ALWAYS",                      UINT2NUM(CREATE_ALWAYS));
    rb_define_const(iConstants, "OPEN_EXISTING",                      UINT2NUM(OPEN_EXISTING));
    rb_define_const(iConstants, "OPEN_ALWAYS",                        UINT2NUM(OPEN_ALWAYS));
    rb_define_const(iConstants, "TRUNCATE_EXISTING",                  UINT2NUM(TRUNCATE_EXISTING));

    rb_define_const(iConstants, "FILE_ATTRIBUTE_ARCHIVE",             UINT2NUM(FILE_ATTRIBUTE_ARCHIVE));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_COMPRESSED",          UINT2NUM(FILE_ATTRIBUTE_COMPRESSED));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_DEVICE",              UINT2NUM(FILE_ATTRIBUTE_DEVICE));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_DIRECTORY",           UINT2NUM(FILE_ATTRIBUTE_DIRECTORY));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_ENCRYPTED",           UINT2NUM(FILE_ATTRIBUTE_ENCRYPTED));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_HIDDEN",              UINT2NUM(FILE_ATTRIBUTE_HIDDEN));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_NORMAL",              UINT2NUM(FILE_ATTRIBUTE_NORMAL));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_NOT_CONTENT_INDEXED", UINT2NUM(FILE_ATTRIBUTE_NOT_CONTENT_INDEXED));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_OFFLINE",             UINT2NUM(FILE_ATTRIBUTE_OFFLINE));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_READONLY",            UINT2NUM(FILE_ATTRIBUTE_READONLY));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_REPARSE_POINT",       UINT2NUM(FILE_ATTRIBUTE_REPARSE_POINT));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_SPARSE_FILE",         UINT2NUM(FILE_ATTRIBUTE_SPARSE_FILE));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_SYSTEM",              UINT2NUM(FILE_ATTRIBUTE_SYSTEM));
    rb_define_const(iConstants, "FILE_ATTRIBUTE_TEMPORARY",           UINT2NUM(FILE_ATTRIBUTE_TEMPORARY));

    // http://msdn.microsoft.com/en-us/library/gg258117%28v=vs.85%29.aspx
    #ifndef FILE_ATTRIBUTE_VIRTUAL
    #   define FILE_ATTRIBUTE_VIRTUAL   0x10000
    #endif
    rb_define_const(iConstants, "FILE_ATTRIBUTE_VIRTUAL",             UINT2NUM(FILE_ATTRIBUTE_VIRTUAL));

    rb_define_const(iConstants, "FILE_FLAG_WRITE_THROUGH",            UINT2NUM(FILE_FLAG_WRITE_THROUGH));
    rb_define_const(iConstants, "FILE_FLAG_OVERLAPPED",               UINT2NUM(FILE_FLAG_OVERLAPPED));
    rb_define_const(iConstants, "FILE_FLAG_NO_BUFFERING",             UINT2NUM(FILE_FLAG_NO_BUFFERING));
    rb_define_const(iConstants, "FILE_FLAG_RANDOM_ACCESS",            UINT2NUM(FILE_FLAG_RANDOM_ACCESS));
    rb_define_const(iConstants, "FILE_FLAG_SEQUENTIAL_SCAN",          UINT2NUM(FILE_FLAG_SEQUENTIAL_SCAN));
    rb_define_const(iConstants, "FILE_FLAG_DELETE_ON_CLOSE",          UINT2NUM(FILE_FLAG_DELETE_ON_CLOSE));
    rb_define_const(iConstants, "FILE_FLAG_BACKUP_SEMANTICS",         UINT2NUM(FILE_FLAG_BACKUP_SEMANTICS));
    rb_define_const(iConstants, "FILE_FLAG_POSIX_SEMANTICS",          UINT2NUM(FILE_FLAG_POSIX_SEMANTICS));
    rb_define_const(iConstants, "FILE_FLAG_OPEN_REPARSE_POINT",       UINT2NUM(FILE_FLAG_OPEN_REPARSE_POINT));
    rb_define_const(iConstants, "FILE_FLAG_OPEN_NO_RECALL",           UINT2NUM(FILE_FLAG_OPEN_NO_RECALL));
    rb_define_const(iConstants, "SECURITY_ANONYMOUS",                 UINT2NUM(SECURITY_ANONYMOUS));
    rb_define_const(iConstants, "SECURITY_IDENTIFICATION",            UINT2NUM(SECURITY_IDENTIFICATION));
    rb_define_const(iConstants, "SECURITY_IMPERSONATION",             UINT2NUM(SECURITY_IMPERSONATION));
    rb_define_const(iConstants, "SECURITY_DELEGATION",                UINT2NUM(SECURITY_DELEGATION));
    rb_define_const(iConstants, "SECURITY_CONTEXT_TRACKING",          UINT2NUM(SECURITY_CONTEXT_TRACKING));
    rb_define_const(iConstants, "SECURITY_EFFECTIVE_ONLY",            UINT2NUM(SECURITY_EFFECTIVE_ONLY));

    rb_define_const(iConstants, "FS_CASE_IS_PRESERVED",         UINT2NUM(FS_CASE_IS_PRESERVED));
    rb_define_const(iConstants, "FS_CASE_SENSITIVE",            UINT2NUM(FS_CASE_SENSITIVE));
    rb_define_const(iConstants, "FS_UNICODE_STORED_ON_DISK",    UINT2NUM(FS_UNICODE_STORED_ON_DISK));
    rb_define_const(iConstants, "FS_PERSISTENT_ACLS",           UINT2NUM(FS_PERSISTENT_ACLS));
    rb_define_const(iConstants, "FS_FILE_COMPRESSION",          UINT2NUM(FS_FILE_COMPRESSION));
    rb_define_const(iConstants, "FS_VOL_IS_COMPRESSED",         UINT2NUM(FS_VOL_IS_COMPRESSED));
    rb_define_const(iConstants, "FILE_NAMED_STREAMS",           UINT2NUM(FILE_NAMED_STREAMS));
    rb_define_const(iConstants, "FILE_SUPPORTS_ENCRYPTION",     UINT2NUM(FILE_SUPPORTS_ENCRYPTION));
    rb_define_const(iConstants, "FILE_SUPPORTS_OBJECT_IDS",     UINT2NUM(FILE_SUPPORTS_OBJECT_IDS));
    rb_define_const(iConstants, "FILE_SUPPORTS_REPARSE_POINTS", UINT2NUM(FILE_SUPPORTS_REPARSE_POINTS));
    rb_define_const(iConstants, "FILE_SUPPORTS_SPARSE_FILES",   UINT2NUM(FILE_SUPPORTS_SPARSE_FILES));
    rb_define_const(iConstants, "FILE_VOLUME_QUOTAS",           UINT2NUM(FILE_VOLUME_QUOTAS));


    mStatus             = rb_define_module_under(iNazuna, "Status");
    mSuccess            = rb_define_module_under(mStatus, "Success");
    mError              = rb_define_module_under(mStatus, "Error");
    mDriveLetterError   = rb_define_module_under(mStatus, "DriveLetterError");
    mDriverInstallError = rb_define_module_under(mStatus, "DriverInstallError");
    mStartError         = rb_define_module_under(mStatus, "StartError");
    mMountError         = rb_define_module_under(mStatus, "MountError");
    mMountPointError    = rb_define_module_under(mStatus, "MountPointError");
    rb_define_const(mSuccess,            "CODE", INT2FIX(DOKAN_SUCCESS));
    rb_define_const(mError,              "CODE", INT2FIX(DOKAN_ERROR));
    rb_define_const(mDriveLetterError,   "CODE", INT2FIX(DOKAN_DRIVE_LETTER_ERROR));
    rb_define_const(mDriverInstallError, "CODE", INT2FIX(DOKAN_DRIVER_INSTALL_ERROR));
    rb_define_const(mStartError,         "CODE", INT2FIX(DOKAN_START_ERROR));
    rb_define_const(mMountError,         "CODE", INT2FIX(DOKAN_MOUNT_ERROR));
    rb_define_const(mMountPointError,    "CODE", INT2FIX(DOKAN_MOUNT_POINT_ERROR));
    rb_define_const(mSuccess,            "MESSAGE", frozen_str("Successed"));
    rb_define_const(mError,              "MESSAGE", frozen_str("General Error"));
    rb_define_const(mDriveLetterError,   "MESSAGE", frozen_str("Bad Drive letter"));
    rb_define_const(mDriverInstallError, "MESSAGE", frozen_str("Can't install driver"));
    rb_define_const(mStartError,         "MESSAGE", frozen_str("Driver something wrong"));
    rb_define_const(mMountError,         "MESSAGE", frozen_str("Can't assign a drive letter or mount point"));
    rb_define_const(mMountPointError,    "MESSAGE", frozen_str("Mount point is invalid"));

    /*
     * Document-class: Nazuna::Dokan::OperationInfo
     *
     * ナズナ舞台裏で利用されるクラス。
     *
     * I/O 要求時に付随する情報を取りまとめる。
     */

    iOpInfo = rb_define_class_under(iDokan, "OperationInfo", rb_cObject);
    rb_undef_alloc_func(iOpInfo);

    rb_define_method(iOpInfo, "path",           RUBY_METHOD_FUNC(opinfo_get_path), 0);
    rb_define_method(iOpInfo, "context",        RUBY_METHOD_FUNC(opinfo_get_context), 0);
    rb_define_method(iOpInfo, "directory?",     RUBY_METHOD_FUNC(opinfo_is_directory), 0);
    rb_define_method(iOpInfo, "deleteonclose?", RUBY_METHOD_FUNC(opinfo_is_delete_on_close), 0);
    rb_define_method(iOpInfo, "pagingio?",      RUBY_METHOD_FUNC(opinfo_is_paging_io), 0);
    rb_define_method(iOpInfo, "synchronousio?", RUBY_METHOD_FUNC(opinfo_is_synchronous_io), 0);
    rb_define_method(iOpInfo, "nocache?",       RUBY_METHOD_FUNC(opinfo_is_no_cache), 0);
    rb_define_method(iOpInfo, "write_to_eof?",  RUBY_METHOD_FUNC(opinfo_is_write_to_end_of_file), 0);
    rb_define_method(iOpInfo, "process",        RUBY_METHOD_FUNC(opinfo_get_process), 0);

    /*
     * Document-class: Nazuna::Dokan::ProcessInfo
     *
     * ナズナ舞台裏で利用されるクラス。
     *
     * I/O 要求時に付随するプロセス情報を取りまとめる。
     */

    cProcessInfo = rb_define_class_under(iDokan, "ProcessInfo", rb_cObject);
    rb_undef_alloc_func(cProcessInfo);
    rb_define_method(cProcessInfo, "pid",      RUBY_METHOD_FUNC(processinfo_pid), 0);
    rb_define_method(cProcessInfo, "userinfo", RUBY_METHOD_FUNC(processinfo_userinfo), 0);
}

#endif /* defined(HAVE_DOKAN_H) */
