// encoding:UTF-8

/*
 * nazuna.c - Filesystem in userspace for ruby/fuse
 *
 * - AUTHOR : dearblue <dearblue@users.sourceforge.jp>
 * - PROJECTPAGE : http://sourceforge.jp/projects/rutsubo/
 * - LICENSE : 2-clause BSD License
 */

#if defined(HAVE_FUSE_H) && !defined(HAVE_DOKAN_H)

//#define NAZUNA_DEBUG_OUT 1

#include "nazuna.h"
#include <errno.h>
#include <fuse.h>


#define PTHREAD_CLEANUP_FUNC(FUNC) ((void (*)(void *))(FUNC))
#define RUBY_BLOCKING_FUNC(FUNC) ((rb_blocking_function_t *)(FUNC))


enum {
    MAX_DISPATCHTHREADS         = 20,
    DEFAULT_DISPATCH_THREADS    = 1,
    DEFAULT_DISPATCH_SAFELEVEL  = 0,
};

// SECTION: Nazuna

struct nazuna;
struct request;


typedef int dispatch_f(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args);


struct nazuna
{
    VALUE mounter;  // fuse_mainを監視するスレッド
    VALUE dispatch; // Nazuna::~.mountのdispatch引数と等価
    VALUE threads;  // I/O要求の処理スレッド (配列にスレッドを詰める)
    VALUE mountpoint; // マウントポイント。Nazuna::~.mountのmountpoint引数と等価
    VALUE logger; // Nazuna::~.mountのlogger引数と等価

    struct request *request;    // fuseから渡されたI/O要求をrubyに渡すための、片方向リスト構造
    struct request *request_tail; // requestを辿って行った、最後尾

    // requestをpost/popするための同期機構
    pthread_mutex_t mutex;
    pthread_cond_t cond;

    // 以下Context.mountしたときの任意引数

    int nthread;    // I/O要求の処理スレッド数。
    int network;    // ストレージはネットワーク接続されているか。
    int removable;  // ストレージは着脱式か。
    int safelevel;  // I/O要求処理スレッドのrubyセキュリティレベル。
    int cleanup;    // 終了処理中であれば0以外。
};

struct request
{
    struct request *next;

    pthread_mutex_t mutex;
    pthread_cond_t cond;

    intptr_t status;
    struct nazuna *nazuna;
    const char *funcname;
    dispatch_f *dispatch;
    struct fuse_context *context;

    int done;
    VALUE refs[8];
    va_list args;
};


static VALUE nazuna_debug_log;
static VALUE eReferenceError, eMountError,
             eReadTooLarge, eStatusError, eStatusType;

static ID IDgetattr, IDreadlink, IDmknod, IDmkdir, IDunlink, IDrmdir, IDsymlink, IDrename, IDlink, IDchmod, IDchown, IDtruncate, IDutime;
static ID IDopen, IDread, IDwrite, IDstatfs, IDflush, IDrelease, IDfsync;
static ID IDopendir, IDreaddir, IDreleasedir, IDfsyncdir;
static ID IDaccess, IDcreate, IDftruncate, IDfgetattr, IDlock, IDutime, IDbmap, IDioctl, IDpoll;
static ID IDbavail, IDbfree, IDblocks, IDfavail, IDffree, IDfiles, IDbsize, IDflag, IDfrsize, IDfsid, IDnamemax;
static ID IDname;
static ID IDnthread, IDnetwork, IDremovable, IDlogger;


static inline void
nazuna_debug_log_call(const char filename[], int line, const char funcname[], ID type, VALUE obj)
{
    VALUE args[] = {
        rb_str_new_cstr(filename),
        INT2FIX(line),
        rb_str_new_cstr(funcname),
        obj,
    };
    rb_funcall3(nazuna_debug_log, type, ELEMENTOF(args), args);
}

#define NAZUNA_DEBUG_LOG_CALL(TYPE, OBJ) (!NIL_P(nazuna_debug_log) ? nazuna_debug_log_call(__FILE__, __LINE__, __func__, (TYPE), (OBJ)) : NULL)
#define RBLOG0()        NAZUNA_DEBUG_LOG_CALL(rb_intern("call"), Qnil)
#define RBLOGS(STR)     NAZUNA_DEBUG_LOG_CALL(rb_intern("call"), rb_str_new_cstr((STR)))
#define RBLOGF(...)     NAZUNA_DEBUG_LOG_CALL(rb_intern("call"), rb_sprintf(__VA_ARGS__))
#define RBLOGOBJ(OBJ)   NAZUNA_DEBUG_LOG_CALL(rb_intern("call"), (OBJ))


static struct request *
pop_request_pop(struct nazuna *nazuna)
{
__LOG0();
    struct request *req = NULL;

    if (pthread_mutex_lock(&nazuna->mutex) == 0) {
        pthread_cleanup_push(PTHREAD_CLEANUP_FUNC(pthread_mutex_unlock), &nazuna->mutex);
        if (!nazuna->cleanup && pthread_cond_wait(&nazuna->cond, &nazuna->mutex) == 0) {
            req = nazuna->request;
            if (req) {
                nazuna->request = req->next;
                req->next = NULL;
                if (!nazuna->request) {
                    nazuna->request_tail = NULL;
                }
            }
        }
        pthread_cleanup_pop(1);
    }

__LOG("request: %p", req);
    return req;
}

/*
 * ruby thread から呼び出される
 */
static inline struct request *
pop_request(struct nazuna *nazuna)
{
RBLOG0();
    return (struct request *)rb_thread_blocking_region((rb_blocking_function_t *)pop_request_pop,
                                                       nazuna, RUBY_UBF_IO, NULL);
}


static void
send_request_cleanup(struct request *req)
{
    pthread_mutex_unlock(&req->mutex);
    pthread_mutex_destroy(&req->mutex);
    pthread_cond_destroy(&req->cond);
}

/*
 * FUSE Managed Threadからのみ呼び出し可能
 */
static int
send_request(const char funcname[], dispatch_f *dispatch, ...)
{
__LOG("enter %s", funcname);
    struct fuse_context *context = fuse_get_context();
    struct nazuna *nazuna = (struct nazuna *)(context->private_data);
    if (nazuna->cleanup) { return -1; }

    {
        va_list args;
        va_start(args, dispatch);

        struct request req = {
            .next       = NULL,
            .mutex      = PTHREAD_MUTEX_INITIALIZER,
            .cond       = PTHREAD_COND_INITIALIZER,
            .status     = -1,
            .nazuna     = nazuna,
            .funcname   = funcname,
            .dispatch   = dispatch,
            .context    = context,
            .done       = 0,
            .refs       = { 0 },
            .args       = args,
        };

__LOG0();
        if (pthread_mutex_lock(&req.mutex) == 0) {
            pthread_cleanup_push(PTHREAD_CLEANUP_FUNC(send_request_cleanup), &req);
            if (pthread_mutex_lock(&nazuna->mutex) == 0) {
//                if (!nazuna->cleanup) {
                    pthread_cleanup_push(PTHREAD_CLEANUP_FUNC(pthread_mutex_unlock), &nazuna->mutex);
                    struct request *p = nazuna->request_tail;
                    if (p) {
                        p->next = nazuna->request_tail = &req;
                    } else {
                        nazuna->request = nazuna->request_tail = &req;
                    }
__LOG0();
                    pthread_cond_signal(&nazuna->cond);
//                }
                pthread_cleanup_pop(1);
            }
__LOG0();
//            if (!nazuna->cleanup) {
                pthread_cond_wait(&req.cond, &req.mutex);
//            }
            pthread_cleanup_pop(1);
        }

__LOG("leave %s, status: %d", funcname, req.status);
        return req.status;
    }
}

static void
finish_request(struct request *r)
{
    if (pthread_mutex_lock(&r->mutex) == 0) {
        pthread_cond_signal(&r->cond);
        pthread_mutex_unlock(&r->mutex);
    }
}

static void
clear_queue(struct nazuna *nazuna)
{
    nazuna->cleanup = 1;

    if (pthread_mutex_lock(&nazuna->mutex) == 0) {
        struct request *r = nazuna->request;
        nazuna->request = nazuna->request_tail = NULL;
        while (r) {
            pthread_mutex_lock(&r->mutex);

            r->status = -1;
            struct request *rr = r->next;
            r->next = NULL;
            pthread_cond_signal(&r->cond);
            pthread_mutex_unlock(&r->mutex);
            r = rr;
        }
        pthread_cond_broadcast(&nazuna->cond);
        pthread_mutex_unlock(&nazuna->mutex);
    }
}

static inline void
unlink_references(VALUE *p, const VALUE *end)
{
    for (; p < end && *p; p ++) {
        RTYPEDDATA_DATA(*p) = NULL;
    }
}

static VALUE
request_dispatch_call(struct request *r)
{
    r->status = r->dispatch(r->nazuna, Qnil, r->refs, r->args);
    r->done = 1;

    return Qnil;
}

static VALUE
request_dispatch_cleanup(struct request *r)
{
    unlink_references(r->refs, r->refs + ELEMENTOF(r->refs));

    if (!r->done) {
__LOG0();
        r->status = -EDOOFUS;
        r->nazuna->cleanup = 1;
        rb_thread_kill(r->nazuna->mounter);

        VALUE err = rb_errinfo();
        if (!NIL_P(err)) {
            VALUE bt = rb_funcall2(err, rb_intern("backtrace"), 0, NULL);
            if (NIL_P(bt)) {
                bt = rb_ary_new();
                rb_funcall2(err, rb_intern("set_backtrace"), 1, &bt);
            }

            // NOTE: "nazuna-fuse.c" にしている理由は、__FILE__ だと
            //       ビルド環境時のパス構成が漏洩する事が考えられるため。
            rb_ary_push(bt, rb_sprintf("%s:in operation `%s'",
                            "nazuna-fuse.c", r->funcname));
        }
    }

    finish_request(r);

    return Qnil;
}

static VALUE
request_dispatch(struct nazuna *nazuna)
{
    rb_set_safe_level(nazuna->safelevel);

    struct request *r;

__LOG("enter %s", __func__);
    while ((r = pop_request(nazuna)) != NULL) {
        rb_ensure(RUBY_METHOD_FUNC(request_dispatch_call), (VALUE)(r),
                  RUBY_METHOD_FUNC(request_dispatch_cleanup), (VALUE)(r));
    }
__LOG("leave %s", __func__);

    return Qnil;
}

#define SEND_REQUEST(METHOD, ...) send_request(# METHOD, wrap_ ## METHOD, __VA_ARGS__)


// SECTION : FileInfo

static VALUE cRefInfo;

struct nazuna_fileinfo
{
    const char *path;
    struct fuse_file_info *fuse_fileinfo;
};

static size_t
refinfo_memsize(const void *p)
{
    return 0;
}

static const rb_data_type_t refinfo_datatype = {
    .wrap_struct_name = "nazuna.file_info",
    .function.dmark = NULL,
    .function.dfree = NULL,
    .function.dsize = refinfo_memsize,
    .parent = NULL,
    .data = NULL,
};


static inline VALUE
refinfo(struct nazuna_fileinfo *info)
{
    return TypedData_Wrap_Struct(cRefInfo, &refinfo_datatype, info);
}

static inline struct nazuna_fileinfo *
refinfo_refp(VALUE obj)
{
    struct nazuna_fileinfo *info = (struct nazuna_fileinfo *)RTYPEDDATA_DATA(obj);
    if (!info) {
        rb_raise(eReferenceError, "null reference to file information");
    }
    return info;
}

static inline struct nazuna_fileinfo *
refinfo_ref(VALUE obj)
{
    rb_check_typeddata(obj, &refinfo_datatype);
    return refinfo_refp(obj);
}

static VALUE
refinfo_getpath(VALUE obj)
{
    struct nazuna_fileinfo *info = refinfo_ref(obj);
    return rb_str_new_cstr(info->path);
}

static VALUE
refinfo_getfd(VALUE obj)
{
    struct nazuna_fileinfo *info = refinfo_ref(obj);
    return UINT2NUM(info->fuse_fileinfo->fh);
}

static VALUE
refinfo_getflags(VALUE obj)
{
    struct nazuna_fileinfo *info = refinfo_ref(obj);
    return UINT2NUM(info->fuse_fileinfo->flags);
}

static VALUE
refinfo_getdirect_io(VALUE obj)
{
    struct nazuna_fileinfo *info = refinfo_ref(obj);
    return info->fuse_fileinfo->direct_io ? Qtrue : Qfalse;
}

static VALUE
refinfo_getkeep_cache(VALUE obj)
{
    struct nazuna_fileinfo *info = refinfo_ref(obj);
    return info->fuse_fileinfo->keep_cache ? Qtrue : Qfalse;
}

static inline VALUE
refinfo_inspect(VALUE obj)
{
    rb_check_typeddata(obj, &refinfo_datatype);
    struct nazuna_fileinfo *info = (struct nazuna_fileinfo *)RTYPEDDATA_DATA(obj);

    if (info) {
        VALUE inspected = rb_inspect(rb_str_new_cstr(info->path));
        return rb_sprintf("#<Nazuna::FileInfo %s [fh=%d, flags=%08x, direct_io?=%s, keep_cache?=%s]>",
                          StringValueCStr(inspected),
                          (int)info->fuse_fileinfo->fh,
                          info->fuse_fileinfo->flags,
                          info->fuse_fileinfo->direct_io ? "true" : "false",
                          info->fuse_fileinfo->keep_cache ? "true" : "false");
    } else {
        return rb_str_new_cstr("#<Nazuna::FileInfo *INVALID REFERENCE*>");
    }
}

static inline VALUE
refinfo_to_s(VALUE obj)
{
    struct nazuna_fileinfo *info = refinfo_ref(obj);
    return rb_str_new_cstr(info->path);
}

#if 0
static VALUE cRefStr;

static size_t
refstr_memsize(const void *p)
{
    return 0;
}

static const rb_data_type_t refstr_datatype = {
    .wrap_struct_name = "nazuna.referenced_string",
    .function.dmark = NULL,
    .function.dfree = NULL,
    .function.dsize = refstr_memsize,
    .parent = NULL,
    .data = NULL,
};


static inline VALUE
refstr(const char str[])
{
    return TypedData_Wrap_Struct(cRefStr, &refstr_datatype, (void *)str);
}

static inline const char *
refstr_refp(VALUE obj)
{
    const char *str = (const char *)RTYPEDDATA_DATA(obj);
    if (!str) {
        rb_raise(eReferenceError, "invalid reference for string");
    }
    return str;
}

static inline const char *
refstr_ref(VALUE obj)
{
    rb_check_typeddata(obj, &refstr_datatype);
    return refstr_refp(obj);
}

static inline VALUE
refstr_inspect(VALUE obj)
{
    rb_check_typeddata(obj, &refstr_datatype);
    const char *str = (const char *)RTYPEDDATA_DATA(obj);

    if (str) {
        VALUE inspected = rb_inspect(rb_str_new_cstr(str));
        return rb_sprintf("#<Nazuna::ReferenceString %s>", StringValueCStr(inspected));
    } else {
        return rb_str_new_cstr("#<Nazuna::ReferenceString *INVALID REFERENCE*>");
    }
}

static inline VALUE
refstr_to_s(VALUE obj)
{
    const char *str = refstr_ref(obj);
    return rb_str_new_cstr(str);
}
#endif



static VALUE cBMAP;

struct nazuna_bmap
{
    uint64_t *index;
    size_t blocksize;
};

static size_t
refbmap_memsize(const void *p)
{
    return 0;
}

static const rb_data_type_t refbmap_datatype = {
    .wrap_struct_name = "nazuna.bmap",
    .function.dmark = NULL,
    .function.dfree = NULL,
    .function.dsize = refbmap_memsize,
    .parent = NULL,
    .data = NULL,
};


static inline VALUE
refbmap(struct nazuna_refbmap *bmap)
{
    return TypedData_Wrap_Struct(cBMAP, &refbmap_datatype, bmap);
}

static inline struct nazuna_refbmap *
refbmap_refp(VALUE obj)
{
    struct nazuna_refbmap *bmap = (struct nazuna_refbmap *)RTYPEDDATA_DATA(obj);
    if (!bmap) {
        rb_raise(eReferenceError, "null reference to bmap");
    }
    return bmap;
}

static inline struct nazuna_refbmap *
refbmap_ref(VALUE obj)
{
    rb_check_typeddata(obj, &refbmap_datatype);
    return refbmap_refp(obj);
}


// SECTION : dispatch

// NOTE: REFSTR、REFINFO が next_ref を通しているのは、refs++ がうまくいかないから
static inline VALUE next_ref(VALUE **refs, VALUE ref) { return *(*refs) ++ = ref; }
//#define REFSTR(REFS, STR) next_ref(&(REFS), refstr((STR)))
#define REFINFO(REFS, STR) next_ref(&(REFS), refinfo((STR)))
//#define REFSTR(REFS, STR) (*(REFS) ++ = refstr((STR)))
//#define REFINFO(REFS, FILEINFO) (*(REFS) ++ = refinfo((FILEINFO)))

#define COPYBUF(BUF, SIZE)         \
    ({                             \
        const char *__buf = (BUF); \
        size_t __size = (SIZE);    \
        rb_str_new(__buf, __size); \
    })                             \

#define REFLOCK(REFS, FLOCK) (*(REFS) ++ = flock_wrap((FLOCK)))

#define TIMESPECNEW(TV)                                 \
    ({                                                  \
        const struct timespec *__tv = (TV);             \
        rb_time_nano_new(__tv->tv_sec, __tv->tv_nsec);  \
    })                                                  \


#define REFBMAP(REFS, BMAP)           \
    ({                                \
        *(REFS) ++ = refbmap((BMAP)); \
    })                                \


static inline struct stat *
assignstat(struct stat *st, VALUE src)
{
    static ID IDdev, IDino, IDmode, IDnlink, IDuid, IDgid, IDrdev,
              IDatime, IDmtime, IDctime, IDsize, IDblocks, IDblksize,
              IDflags, IDgen, IDlspare, IDbirthtime;
    if (!IDdev) {
        IDdev = rb_intern("dev");
        IDino = rb_intern("ino");
        IDmode = rb_intern("mode");
        IDnlink = rb_intern("nlink");
        IDuid = rb_intern("uid");
        IDgid = rb_intern("gid");
        IDrdev = rb_intern("rdev");
        IDatime = rb_intern("atime");
        IDmtime = rb_intern("mtime");
        IDctime = rb_intern("ctime");
        IDsize = rb_intern("size");
        IDblocks = rb_intern("blocks");
        IDblksize = rb_intern("blksize");
        IDflags = rb_intern("flags");
        IDgen = rb_intern("gen");
        IDlspare = rb_intern("lspare");
        IDbirthtime = rb_intern("birthtime");
    }

    memset(st, 0, sizeof(*st));
    st->st_dev = NUM2UINT(getattr(src, IDdev));
    st->st_ino = NUM2UINT(getattr(src, IDino));
    st->st_mode = NUM2UINT(getattr(src, IDmode));
    st->st_nlink = NUM2UINT(getattr(src, IDnlink));
    st->st_uid = NUM2UINT(getattr(src, IDuid));
    st->st_gid = NUM2UINT(getattr(src, IDgid));
    st->st_rdev = NUM2UINT(getattr(src, IDrdev));
    st->st_atim = rb_time_timespec(getattr(src, IDatime));
    st->st_mtim = rb_time_timespec(getattr(src, IDmtime));
    st->st_ctim = rb_time_timespec(getattr(src, IDctime));
    st->st_size = NUM2ULL(getattr(src, IDsize));
    st->st_blocks = NUM2UINT(getattr(src, IDblocks));
    st->st_blksize = NUM2UINT(getattr(src, IDblksize));

    if (rb_respond_to(src, IDflags)) {
        st->st_flags = NUM2UINT(getattr(src, IDflags));
    } else {        
        st->st_flags = 0;
    }

    if (rb_respond_to(src, IDgen)) {
        st->st_gen = NUM2UINT(getattr(src, IDgen));
    } else {
        st->st_gen = 0;
    }

    if (rb_respond_to(src, IDlspare)) {
        st->st_lspare = NUM2INT(getattr(src, IDlspare));
    } else {
        st->st_lspare = 0;
    }

    if (rb_respond_to(src, IDbirthtime)) {
        st->st_birthtim = rb_time_timespec(getattr(src, IDbirthtime));
    } else {
        st->st_birthtim = st->st_ctim;
    }

    return st;
}


static inline int
isasyserr(VALUE status)
{
    if (rb_obj_is_kind_of(status, rb_eSystemCallError))
        return 1;
    if (rb_obj_is_kind_of(status, rb_cModule) && RTEST(rb_class_inherited_p(status, rb_eSystemCallError)))
        return 1;
    return 0;
}


static inline ssize_t
status2int(VALUE status, int *isinteger)
{
    if (isinteger) { *isinteger = 1; }

    if (NIL_P(status)) {
        return -ENOSYS;
    } else if (rb_obj_is_kind_of(status, rb_eSystemCallError)) {
        return -NUM2INT(getattr(status, rb_intern("errno")));
    } else if (rb_obj_is_kind_of(status, rb_cModule) &&
               RTEST(rb_class_inherited_p(status, rb_eSystemCallError))) {
        return -NUM2INT(rb_const_get(status, rb_intern("Errno")));
    } else if (rb_obj_is_kind_of(status, rb_cInteger)) {
        return NUM2SSIZET(status);
    } else {
        if (isinteger) { *isinteger = 0; }
        return 0;
    }
}

#define RETURN_IF_ERROR(STATUS)                    \
    {                                              \
        ssize_t _err = status2int((STATUS), NULL); \
        if (_err < 0) { return _err; }             \
    }                                              \

#define RETURN_STATUS(STATUS)                                       \
    {                                                               \
        int _isinteger;                                             \
        ssize_t _err = status2int((STATUS), &_isinteger);           \
                                                                    \
        if (!_isinteger) {                                          \
            check_status_type((STATUS), rb_cInteger, "Integer");    \
        }                                                           \
        if (_err > 0) {                                             \
            rb_raise(eStatusError,                                  \
                     "dispatch returned status is not negative");   \
        }                                                           \
        return _err;                                                \
    }                                                               \


static inline void
check_status_type(VALUE status, VALUE type, const char *typename)
{
    if (!rb_obj_is_kind_of(status, type)) {
        if (typename) {
            rb_raise(eStatusType,
                     "operation returned wrong status type (not a %s)",
                     typename);
        } else {
            rb_raise(eStatusType,
                     "operation returned wrong status type");
        }
    }
}


static int
wrap_getattr(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
__LOG0();
    VALUE status = CALLDISPATCH(getattr, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)));

__LOG0();
    RETURN_IF_ERROR(status);

__LOG0();
    assignstat(va_arg(args, struct stat *), status);

__LOG0();
    return 0;
}

static int
dispatch_getattr(const char path[], struct stat *st)
{
__LOG0();
    return SEND_REQUEST(getattr, path, st);
}


static int
wrap_readlink(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    size_t size;
    VALUE status = CALLDISPATCH(readlink, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                SIZET2NUM(size = va_arg(args, size_t)));

    RETURN_IF_ERROR(status);
    check_status_type(status, rb_cString, "String");

    size_t len = RSTRING_LEN(status);
    if (len >= size)
        rb_raise(eReadTooLarge,
                 "readlink operation too large (%d for %d bytes)",
                 len, size - 1);

    char *ptr = va_arg(args, char *);

    memcpy(ptr, RSTRING_PTR(status), len);
    ptr[len] = '\0';

    return 0;
}

static int
dispatch_readlink(const char path[], char ptr[], size_t size)
{
__LOG0();
    return SEND_REQUEST(readlink, path, size, ptr);
}


static int
wrap_mknod(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(mknod, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                UINT2NUM(va_arg(args, int)),
                                UINT2NUM(va_arg(args, dev_t)));
    RETURN_STATUS(status);
}

static int
dispatch_mknod(const char path[], mode_t mode, dev_t dev)
{
__LOG0();
    return SEND_REQUEST(mknod, path, mode, dev);
}


static int
wrap_mkdir(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(mkdir, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                UINT2NUM(va_arg(args, int)));

    RETURN_STATUS(status);
}

static int
dispatch_mkdir(const char path[], mode_t mode)
{
__LOG0();
    return SEND_REQUEST(mkdir, path, mode);
}


static int
wrap_unlink(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(unlink, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)));
    RETURN_STATUS(status);
}

static int
dispatch_unlink(const char path[])
{
__LOG0();
    return SEND_REQUEST(unlink, path);
}


static int
wrap_rmdir(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(rmdir, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)));
    RETURN_STATUS(status);
}

static int
dispatch_rmdir(const char path[])
{
__LOG0();
    return SEND_REQUEST(rmdir, path);
}


static int
wrap_symlink(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(symlink, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                rb_str_new_cstr(va_arg(args, const char *)));
    RETURN_STATUS(status);
}

static int
dispatch_symlink(const char target[], const char linkname[])
{
__LOG0();
    return SEND_REQUEST(symlink, target, linkname);
}


static int
wrap_rename(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(rename, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                rb_str_new_cstr(va_arg(args, const char *)));
    RETURN_STATUS(status);
}

static int
dispatch_rename(const char oldpath[], const char newpath[])
{
__LOG0();
    return SEND_REQUEST(rename, oldpath, newpath);
}


static int
wrap_link(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(link, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                rb_str_new_cstr(va_arg(args, const char *)));
    RETURN_STATUS(status);
}

static int
dispatch_link(const char target[], const char linkname[])
{
__LOG0();
    return SEND_REQUEST(link, target, linkname);
}


static int
wrap_chmod(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(chmod, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                UINT2NUM(va_arg(args, int)));
    RETURN_STATUS(status);
}

static int
dispatch_chmod(const char path[], mode_t mode)
{
__LOG0();
    return SEND_REQUEST(chmod, path, mode);
}


static int
wrap_chown(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(chown, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                UINT2NUM(va_arg(args, uid_t)),
                                UINT2NUM(va_arg(args, gid_t)));
    RETURN_STATUS(status);
}

static int
dispatch_chown(const char path[], uid_t uid, gid_t gid)
{
__LOG0();
    return SEND_REQUEST(chown, path, uid, gid);
}


static int
wrap_truncate(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(truncate, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                ULL2NUM(va_arg(args, off_t)));
    RETURN_STATUS(status);
}

static int
dispatch_truncate(const char path[], off_t size)
{
__LOG0();
    return SEND_REQUEST(truncate, path, size);
}


static int
wrap_utime(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(utime, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                rb_time_new(va_arg(args, time_t), 0),
                                rb_time_new(va_arg(args, time_t), 0));
    RETURN_STATUS(status);
}

static int
dispatch_utime(const char path[], struct utimbuf *timebuf)
{
__LOG0();
    if (timebuf) {
        return SEND_REQUEST(utime, path, timebuf->actime, timebuf->modtime);
    } else {
        // もしかしたら必要ないかもしれないけれど、保険として記述しておく
        time_t t = time(NULL);
        return SEND_REQUEST(utime, path, t, t);
    }
}

static int
wrap_open(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    struct nazuna_fileinfo *info = va_arg(args, struct nazuna_fileinfo *);
    VALUE status = CALLDISPATCH(open, nazuna,
                                REFINFO(refs, info));

    RETURN_IF_ERROR(status);

    int64_t fd = NUM2LL(status);
    if (fd < 0)
        return fd;

    info->fuse_fileinfo->fh = fd;

    return 0;
}

static int
dispatch_open(const char path[], struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(open, &info2);
}


static int
wrap_read(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    size_t size;
    VALUE status = CALLDISPATCH(read, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)),
                                ULL2NUM(va_arg(args, off_t)),
                                SIZET2NUM(size = va_arg(args, size_t)));

    if (NIL_P(status)) { return 0; }

    RETURN_IF_ERROR(status);
    check_status_type(status, rb_cString, "String");

    {
        size_t len = RSTRING_LEN(status);
        if (len > size) {
            rb_raise(eReadTooLarge,
                     "read operation too large (%d for %d octets)",
                     len, size);
        }

        memcpy(va_arg(args, char *), RSTRING_PTR(status), size);

        return len;
    }
}

static int
dispatch_read(const char path[], char buf[], size_t size, off_t offset, struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(read, &info2, offset, size, buf);
}


static int
wrap_write(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(write, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)),
                                ULL2NUM(va_arg(args, off_t)),
                                COPYBUF(va_arg(args, const char *), va_arg(args, size_t)));

    RETURN_STATUS(status);
}

static int
dispatch_write(const char path[], const char buf[], size_t size, off_t offset, struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(write, &info2, offset, buf, size);
}


static int
wrap_statfs(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(statfs, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)));

    RETURN_IF_ERROR(status);

    struct statvfs *statvfs = va_arg(args, struct statvfs *);

    memset(statvfs, 0, sizeof(*statvfs));

    statvfs->f_bavail  = NUM2ULL(rb_funcall2(status, IDbavail,  0, NULL));
    statvfs->f_bfree   = NUM2ULL(rb_funcall2(status, IDbfree,   0, NULL));
    statvfs->f_blocks  = NUM2ULL(rb_funcall2(status, IDblocks,  0, NULL));
    statvfs->f_favail  = NUM2ULL(rb_funcall2(status, IDfavail,  0, NULL));
    statvfs->f_ffree   = NUM2ULL(rb_funcall2(status, IDffree,   0, NULL));
    statvfs->f_files   = NUM2ULL(rb_funcall2(status, IDfiles,   0, NULL));
    statvfs->f_bsize   = NUM2ULL(rb_funcall2(status, IDbsize,   0, NULL));
    statvfs->f_flag    = NUM2ULL(rb_funcall2(status, IDflag,    0, NULL));
    statvfs->f_frsize  = NUM2ULL(rb_funcall2(status, IDfrsize,  0, NULL));
    statvfs->f_fsid    = NUM2ULL(rb_funcall2(status, IDfsid,    0, NULL));
    statvfs->f_namemax = NUM2ULL(rb_funcall2(status, IDnamemax, 0, NULL));

    return 0;
}

static int
dispatch_statfs(const char path[], struct statvfs *statvfs)
{
__LOG0();
    return SEND_REQUEST(statfs, path, statvfs);
}


static int
wrap_flush(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(flush, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)));
    RETURN_STATUS(status);
}

static int
dispatch_flush(const char path[], struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(flush, &info2);
}


static int
wrap_release(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(release, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)));
    RETURN_STATUS(status);
}

static int
dispatch_release(const char path[], struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(release, &info2);
}


static int
wrap_fsync(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(fsync, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)),
                                INT2NUM(va_arg(args, int)));
    RETURN_STATUS(status);
}

static int
dispatch_fsync(const char path[], int isdatasync, struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(fsync, &info2, isdatasync);
}

#if 0
static int
dispatch_setxattr(const char *, const char *, const char *, size_t, int)
{
    return RUBY_BEGIN_DISPATCH(struct nazuna *nazuna, VALUE refs[]) {
        VALUE args[] = { , };
        VALUE status = CALLDISPATCH(nazuna->logger, nazuna->dispatch, , args);
        return status;
    } RUBY_END_DISPATCH();
}

static int
dispatch_getxattr(const char *, const char *, char *, size_t)
{
    return RUBY_BEGIN_DISPATCH(struct nazuna *nazuna, VALUE refs[]) {
        VALUE args[] = { , };
        VALUE status = CALLDISPATCH(nazuna->logger, nazuna->dispatch, , args);
        return status;
    } RUBY_END_DISPATCH();
}

static int
dispatch_listxattr(const char *, char *, size_t)
{
    return RUBY_BEGIN_DISPATCH(struct nazuna *nazuna, VALUE refs[]) {
        VALUE args[] = { , };
        VALUE status = CALLDISPATCH(nazuna->logger, nazuna->dispatch, , args);
        return status;
    } RUBY_END_DISPATCH();
}

static int
dispatch_removexattr(const char *, const char *)
{
    return RUBY_BEGIN_DISPATCH(struct nazuna *nazuna, VALUE refs[]) {
        VALUE args[] = { , };
        VALUE status = CALLDISPATCH(nazuna->logger, nazuna->dispatch, , args);
        return status;
    } RUBY_END_DISPATCH();
}
#endif


static int
wrap_opendir(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    struct nazuna_fileinfo *info = va_arg(args, struct nazuna_fileinfo *);
    VALUE status = CALLDISPATCH(opendir, nazuna,
                                REFINFO(refs, info));

    RETURN_IF_ERROR(status);

    int64_t fd = NUM2LL(status);
    if (fd < 0)
        return fd;

    info->fuse_fileinfo->fh = fd;

    return 0;
}

static int
dispatch_opendir(const char path[], struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(opendir, &info2);
}


struct wrap_readdir_args
{
    void *buf;
    fuse_fill_dir_t filler;
};

static VALUE
wrap_readdir_block(VALUE obj, struct wrap_readdir_args *args, int argc, VALUE argv[])
{
    VALUE name, stat;

    switch (argc) {
    case 1:
        stat = argv[0];
        name = rb_funcall3(stat, IDname, 0, NULL);
        break;
    case 2:
        name = argv[0];
        stat = argv[1];
        break;
    default:
        rb_scan_args(argc, argv, "11", &name, &stat);
        break;
    }

    struct stat st;
    assignstat(&st, stat);

    int cont = args->filler(args->buf, StringValueCStr(name), &st, 0);
    if (cont) {
        return INT2NUM(cont);
    } else {
        return Qnil;
    }
}

static int
wrap_readdir(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    struct wrap_readdir_args blockargs = {
        .buf = va_arg(args, void *),
        .filler = va_arg(args, fuse_fill_dir_t),
    };

    VALUE status = CALLDISPATCH_WITHBLOCK(readdir, nazuna,
                                          wrap_readdir_block, &blockargs,
                                          REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)),
                                          ULL2NUM(va_arg(args, off_t)));

    RETURN_STATUS(status);
}

static int
dispatch_readdir(const char path[], void *buf, fuse_fill_dir_t filler, off_t offset, struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(readdir, buf, filler, &info2, offset);
}


static int
wrap_releasedir(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(releasedir, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)));

    RETURN_STATUS(status);
}

static int
dispatch_releasedir(const char path[], struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(releasedir, &info2);
}


static int
wrap_fsyncdir(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(fsyncdir, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)),
                                INT2NUM(va_arg(args, int)));
    RETURN_STATUS(status);
}

static int
dispatch_fsyncdir(const char path[], int datasync, struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(fsyncdir, &info2, datasync);
}

#if 0
static void *
dispatch_init(struct fuse_conn_info *conn)
{
    return NULL;
}


static void
dispatch_destroy(void *user)
{
}
#endif

static int
wrap_access(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(access, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                INT2NUM(va_arg(args, int)));
    RETURN_STATUS(status);
}

static int
dispatch_access(const char path[], int mask)
{
__LOG0();
    return SEND_REQUEST(access, path, mask);
}


static int
wrap_create(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(create, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)),
                                UINT2NUM(va_arg(args, int)));
    RETURN_STATUS(status);
}

static int
dispatch_create(const char path[], mode_t mode, struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(create, &info2, mode);
}


static int
wrap_ftruncate(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(ftruncate, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)),
                                ULL2NUM(va_arg(args, off_t)));
    RETURN_STATUS(status);
}

static int
dispatch_ftruncate(const char path[], off_t offset, struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(ftruncate, &info2, offset);
}


static int
wrap_fgetattr(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(fgetattr, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)));

    RETURN_IF_ERROR(status);

    struct stat *st = va_arg(args, struct stat *);

    assignstat(st, status);

    return 0;
}

static int
dispatch_fgetattr(const char path[], struct stat *st, struct fuse_file_info *info)
{
__LOG0();
    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(fgetattr, &info2, st);
}


static inline VALUE
flock_wrap(struct flock *flock)
{
    rb_raise(rb_eNotImpError, "implement me - flock");
}
// fcntl.h
// struct flock {
// 	off_t	l_start;	/* starting offset */
// 	off_t	l_len;		/* len = 0 means until end of file */
// 	pid_t	l_pid;		/* lock owner */
// 	short	l_type;		/* lock type: read/write, etc. */
// 	short	l_whence;	/* type of l_start */
// };
static int
wrap_lock(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(lock, nazuna,
                                REFINFO(refs, va_arg(args, struct nazuna_fileinfo *)),
                                INT2NUM(va_arg(args, int)),
                                REFLOCK(refs, va_arg(args, struct flock *)));
    RETURN_STATUS(status);
}

static int
dispatch_lock(const char path[], struct fuse_file_info *info, int cmd, struct flock *flock)
{
__LOGS("implement me");
    return -ENOSYS; // FIXME

    struct nazuna_fileinfo info2 = { .path = path, .fuse_fileinfo = info, };
    return SEND_REQUEST(lock, &info2, cmd, flock);
}


static int
wrap_utimens(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(utime, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                TIMESPECNEW(va_arg(args, const struct timespec *)),
                                TIMESPECNEW(va_arg(args, const struct timespec *)));
    RETURN_STATUS(status);
}

static int
dispatch_utimens(const char path[], const struct timespec tv[2])
{
__LOG0();
    return SEND_REQUEST(utimens, path, tv[0], tv[1]);
}


static int
wrap_bmap(struct nazuna *nazuna, VALUE fusecontext, VALUE refs[], va_list args)
{
    VALUE status = CALLDISPATCH(bmap, nazuna,
                                rb_str_new_cstr(va_arg(args, const char *)),
                                REFBMAP(refs, va_arg(args, struct nazuna_bmap *)));
    RETURN_STATUS(status);
}

static int
dispatch_bmap(const char path[], size_t blocksize, uint64_t *index)
{
    struct nazuna_bmap bmap = { .index = index, .blocksize = blocksize, };

    return SEND_REQUEST(bmap, path, &bmap);
}




static struct fuse_operations nazuna_dispatch = {
    .opendir = dispatch_opendir,
    .readdir = dispatch_readdir,
    .releasedir = dispatch_releasedir,
    .fsyncdir = dispatch_fsyncdir,

    .getattr = dispatch_getattr,
    .readlink = dispatch_readlink,
    .mknod = dispatch_mknod,
    .mkdir = dispatch_mkdir,
    .unlink = dispatch_unlink,
    .rmdir = dispatch_rmdir,
    .symlink = dispatch_symlink,
    .rename = dispatch_rename,
    .link = dispatch_link,
    .chmod = dispatch_chmod,
    .chown = dispatch_chown,
    .truncate = dispatch_truncate,
    .utime = dispatch_utime,
    .open = dispatch_open,
    .read = dispatch_read,
    .write = dispatch_write,
    .statfs = dispatch_statfs,
    .flush = dispatch_flush,
    .release = dispatch_release,
    .fsync = dispatch_fsync,
    //.setxattr = dispatch_setxattr,
    //.getxattr = dispatch_getxattr,
    //.listxattr = NULL,
    //.removexattr = dispatch_removexattr,
    //.init = dispatch_init,
    //.destroy = dispatch_destroy,
    .access = dispatch_access,
    .create = dispatch_create,
    .ftruncate = dispatch_ftruncate,
    .fgetattr = dispatch_fgetattr,
    .lock = dispatch_lock,
    .utimens = dispatch_utimens,
    .bmap = dispatch_bmap,
};


static void
setup_context(struct nazuna *nazuna, int argc, VALUE argv[])
{
    VALUE mountpoint, dispatch, opts;
    VALUE nthread, network, removable, logger, safelevel;

    rb_scan_args(argc, argv, "2:", &dispatch, &mountpoint, &opts);

    if (rb_safe_level() > 0 && OBJ_TAINTED(mountpoint)) {
        rb_raise(rb_eSecurityError, "mountpoint object is tainted");
    }

    if (NIL_P(opts)) {
        nthread   = INT2FIX(DEFAULT_DISPATCH_THREADS);
        network   = Qfalse;
        removable = Qfalse;
        logger    = Qnil;
        safelevel = Qnil;
    } else {
        nthread   = rb_hash_lookup2(opts, ID2SYM(IDnthread),   INT2FIX(DEFAULT_DISPATCH_THREADS));
        network   = rb_hash_lookup2(opts, ID2SYM(IDnetwork),   Qfalse);
        removable = rb_hash_lookup2(opts, ID2SYM(IDremovable), Qfalse);
        logger    = rb_hash_lookup2(opts, ID2SYM(IDlogger),    Qnil);
        safelevel = rb_hash_lookup2(opts, ID2SYM(rb_intern("safelevel")), Qnil);
    }

    memset(nazuna, 0, sizeof(*nazuna));

    nazuna->mounter = Qnil;
    nazuna->dispatch = dispatch;
    nazuna->threads = rb_ary_new();
    nazuna->mountpoint = rb_obj_dup(rb_String(mountpoint));
    nazuna->logger = logger;
    nazuna->request = NULL;
    nazuna->nthread = NUM2UINT(nthread);
    if (nazuna->nthread > MAX_DISPATCHTHREADS) {
        rb_raise(rb_eArgError, "given to nthread too large (%d for %d or under)",
                 nazuna->nthread, MAX_DISPATCHTHREADS);
    }
    nazuna->network = RTEST(network) ? 1 : 0;
    nazuna->removable = RTEST(removable) ? 1 : 0;
    nazuna->safelevel = NIL_P(safelevel) ? rb_safe_level() : NUM2UINT(safelevel);
    if (nazuna->safelevel > 4) { rb_raise(rb_eArgError, "wrong safelevel (%d for 0 .. 4)", nazuna->safelevel); }

    pthread_mutex_init(&nazuna->mutex, NULL);
    pthread_cond_init(&nazuna->cond, NULL);
}


struct wait_fusemain_state
{
    pthread_mutex_t *mutex;
    pthread_cond_t *cond;
    struct timespec waittime;
};
#if 0
static int
wait_fusemain_timedout(struct wait_fusemain_state *state)
{
    return pthread_cond_timedwait(state->cond, state->mutex, &state->waittime);
}

static inline void
wait_fusemain(pthread_mutex_t *mutex, pthread_cond_t *cond)
{
    static int counter;

    struct wait_fusemain_state state = {
        .mutex = mutex,
        .cond = cond,
        .waittime = { 0, 0 },
    };

    int status = 0;
    do {
        clock_gettime(CLOCK_REALTIME, &state.waittime);
        state.waittime.tv_nsec += 200000000;
        if (state.waittime.tv_nsec >= 1000000000) {
            state.waittime.tv_sec ++;
            state.waittime.tv_nsec %= 1000000000;
        }
        status = rb_thread_blocking_region((rb_blocking_function_t *)wait_fusemain_timedout,
                                           &state, RUBY_UBF_IO, NULL);
__LOG("%u:%d", counter ++, status);
    } while (status == ETIMEDOUT);
}
#endif
static VALUE
cleanup_dispatch_join(VALUE th)
{
__LOG0();
    rb_thread_kill(th);
__LOG0();
    thread_join(th);
__LOG0();
    return Qnil;
}

static inline VALUE
cleanup_dispatch(struct nazuna *nazuna)
{
    VALUE err = Qnil;

    VALUE *p = RARRAY_PTR(nazuna->threads);
    VALUE *end = p + RARRAY_LEN(nazuna->threads);
    for (; p < end; p ++) {
        clear_queue(nazuna);

        VALUE err2 = rb_rescue2(cleanup_dispatch_join, *p,
                                rb_errinfo, Qnil,
                                rb_cObject, NULL);
__LOG0();
        if (!NIL_P(err2) && NIL_P(err)) {
            err = err2;
        }
    }

__LOG0();
    return err;
}


struct mount_main_state
{
    struct nazuna *nazuna;
    struct fuse *fuse;
    struct fuse_chan *ch;
    const char *mountpoint;
    VALUE fuseloop;
};
#if 0
static VALUE
mount_main_fuse_loop_nogvl(struct mount_main_state *state)
{
__LOG0();
    fuse_loop(state->fuse);
__LOG0();
    return Qnil;
}
#endif
static VALUE
mount_main_fuse_loop(struct mount_main_state *state)
{
    state->ch = fuse_mount(state->mountpoint, NULL);
    if (!state->ch) {
        rb_raise(eMountError, "failed fuse_mount");
    }
    state->fuse = fuse_new(state->ch, NULL, &nazuna_dispatch, sizeof(nazuna_dispatch), state->nazuna);
    if (!state->fuse) {
        rb_raise(eMountError, "failed fuse_new");
    }
    //fuse_set_signal_handlers(fuse_get_session(state->fuse));

    //rb_thread_blocking_region(mount_main_fuse_loop_nogvl, state, RUBY_UBF_IO, NULL);
    rb_thread_blocking_region(RUBY_BLOCKING_FUNC(fuse_loop), state->fuse,
                              RUBY_UBF_IO, NULL);

    return Qnil;
}

static VALUE
mount_main_join(struct mount_main_state *state)
{
    {
        int i;
        for (i = state->nazuna->nthread; i > 0; i --) {
            rb_ary_push(state->nazuna->threads,
                        rb_thread_create(request_dispatch, state->nazuna));
        }
    }

    state->fuseloop = rb_thread_create(mount_main_fuse_loop, state);

    thread_join(state->fuseloop);

    return Qnil;
}

static void
kill_dispatch_threads(VALUE threadset)
{
    VALUE *p = RARRAY_PTR(threadset);
    VALUE *end = p + RARRAY_LEN(threadset);
    for (; p < end; p ++) {
        rb_thread_kill(*p);
    }
}

static VALUE
mount_main_cleanup(struct mount_main_state *state)
{
__LOG0();
    state->nazuna->cleanup = 1;

__LOG0();
    kill_dispatch_threads(state->nazuna->threads);

__LOG0();
    clear_queue(state->nazuna);

__LOG0();
    if (state->ch) { fuse_unmount(state->mountpoint, state->ch); }
    if (state->fuse) { fuse_exit(state->fuse); fuse_destroy(state->fuse); }
__LOG0();
    if (!NIL_P(state->fuseloop)) {
        thread_join(state->fuseloop);
    }

__LOG0();
    VALUE except = cleanup_dispatch(state->nazuna);

__LOG0();
    pthread_cond_destroy(&state->nazuna->cond);
    pthread_mutex_destroy(&state->nazuna->mutex);

__LOG0();
    if (!NIL_P(except)) { rb_exc_raise(except); }

    return Qnil;
}

static VALUE
mount_main(struct nazuna *nazuna)
{
    struct mount_main_state state = {
        .nazuna = nazuna,
        .fuse = NULL,
        .ch = NULL,
        .mountpoint = StringValueCStr(nazuna->mountpoint),
        .fuseloop = Qnil,
    };

    rb_ensure(mount_main_join, (VALUE)&state,
              mount_main_cleanup, (VALUE)&state);

    return Qnil;
}


static VALUE
nazuna_i_mount_cleanup(struct nazuna *nazuna)
{
__LOG0();
    if (RTEST(nazuna->mounter)) {
__LOG0();
        rb_thread_kill(nazuna->mounter);
__LOG0();
        thread_join(nazuna->mounter);
    }
__LOG0();
    return Qnil;
}

/*
 * call-seq:
 * Nazuna.mount(dispatch, mointpoint, nthread: 5, removable: false, network: false, safelevel: nil, logger: nil)
 *
 * ※ このメソッドは Nazuna 特異クラスの特異メソッドとして定義されます。直接呼び出すことは出来ません。
 *
 * 指定 mountpoint へマウントします。I/O要求は dispatch の各メソッドを呼び出すことで受け渡されます。
 *
 * [mountpoint] ファイルシステム接続先パス位置を指定します。
 *
 *              汚染オブジェクトであれば $SAFE > 0 でセキュリティ例外が発生します。
 *
 * [dispatch]   ファイルシステムとしてのI/O要求を処理するオブジェクトを指定します。
 *
 * [nthread]    I/O要求処理のスレッド数を指定します。
 *
 * [removable]  着脱式デバイスとして認識させます。 FDDやCD/DVDドライブのようなものです。
 *
 * [network]    ネットワークストレージとして認識させます。nfsやsmbfsのようなものです。
 *
 * [safelevel]  I/O要求処理スレッドのセーフレベルを指定します。既定値は #mount 呼び出し時の $SAFE。
 *
 * [logger]     I/O要求や、処理完了時の戻り値を確認できます。デバッグ用途を想定しています。
 *
 *              <tt>yield(event, request, *args)</tt> のかたちでメソッドが呼ばれます。
 *
 * [RETURN]     常に nil が返ります。
 *
 * [EXCEPTION]  モジュール Nazuna::Exceptions 内で定義されているすべての例外と SystemCallError が発生します。
 *
 *              Nazuna の例外は ''rescue Nazuna::Exceptions'' でまとめて受けることが出来ます。
 *
 *              個々の例外についてはモジュール Nazuna::Exceptions 内にまとめてあります。
 *
 *              その他には、dispatch オブジェクトの I/O 処理メソッドで発生した例外もそのまま通知されます ($DEBUG の有効・無効は問いません)。
 */
static VALUE
nazuna_i_mount(int argc, VALUE argv[], VALUE klass)
{
    rb_secure(2);

    struct nazuna nazuna;
    setup_context(&nazuna, argc, argv);
    nazuna.mounter = rb_thread_create(mount_main, &nazuna);

    rb_ensure(thread_join, nazuna.mounter,
              nazuna_i_mount_cleanup, (VALUE)&nazuna);

    return Qnil;
}

static VALUE
nazuna_i_get_debug_log(VALUE obj)
{
    return nazuna_debug_log;
}

static VALUE
nazuna_i_set_debug_log(VALUE obj, VALUE log)
{
    return nazuna_debug_log = log;
}


// SECTION: nazuna


static VALUE mNazuna;       // module Nazuna
static VALUE mConstants;    // module Nazuna::Constants
static VALUE mExceptions;   // module Nazuna::Exceptions
static VALUE iNazuna;       // class << Nazuna
static VALUE iUtils;        // module Nazuna~::Utils
static VALUE iConstants;    // module Nazuna~::Constants


static inline VALUE
nazuna_define_exception(const char typename[], VALUE superclass)
{
    VALUE type = rb_define_class_under(mNazuna, typename, superclass);
    rb_include_module(type, mExceptions);
    return type;
}

void
Init_nazuna(void)
{
    mNazuna = rb_define_module("Nazuna");
    rb_define_const(mNazuna, "Nazuna", mNazuna);

    iNazuna = rb_singleton_class(mNazuna);
    rb_define_const(iNazuna, "Nazuna", mNazuna);

    mConstants = rb_define_module_under(mNazuna, "Constants");
    rb_include_module(mNazuna, mConstants);
    rb_define_const(mConstants, "BLOCK_SIZE", INT2FIX(512));

    mExceptions = rb_define_module_under(mNazuna, "Exceptions");
    rb_include_module(mNazuna, mExceptions);
    eMountError = nazuna_define_exception("MountError", rb_eStandardError);
    eReferenceError = nazuna_define_exception("ReferenceError", rb_eStandardError);
    eStatusError = nazuna_define_exception("StatusError", rb_eStandardError);
    eReadTooLarge = nazuna_define_exception("ReadTooLarge", eStatusError);
    eStatusType = nazuna_define_exception("StatusTypeError", eStatusError);

    rb_define_singleton_method(iNazuna, "mount", RUBY_METHOD_FUNC(nazuna_i_mount), -1);

    nazuna_debug_log = Qnil;
    rb_define_singleton_method(iNazuna, "debug_log", RUBY_METHOD_FUNC(nazuna_i_get_debug_log), 0);
    rb_define_singleton_method(iNazuna, "debug_log=", RUBY_METHOD_FUNC(nazuna_i_set_debug_log), 1);

    iUtils = rb_define_module_under(iNazuna, "Utils");
    rb_include_module(iNazuna, iUtils);
    rb_define_module_function(iUtils, "error_code", RUBY_METHOD_FUNC(utils_i_error_code), 1);
    rb_define_module_function(iUtils, "error?", RUBY_METHOD_FUNC(utils_i_is_error), 1);

    iConstants = rb_define_module_under(iNazuna, "Constants");
    rb_include_module(iNazuna, iConstants);
    rb_define_const(iConstants, "IMPLEMENT", rb_str_new_cstr("nazuna-fuse"));

    cRefInfo = rb_define_class_under(iNazuna, "FileInfo", rb_cObject);
    rb_undef_alloc_func(cRefInfo);
    rb_define_method(cRefInfo, "path", RUBY_METHOD_FUNC(refinfo_getpath), 0);
    rb_define_method(cRefInfo, "fd", RUBY_METHOD_FUNC(refinfo_getfd), 0);
    rb_define_method(cRefInfo, "flags", RUBY_METHOD_FUNC(refinfo_getflags), 0);
    rb_define_method(cRefInfo, "direct_io?", RUBY_METHOD_FUNC(refinfo_getdirect_io), 0);
    rb_define_method(cRefInfo, "keep_cache?", RUBY_METHOD_FUNC(refinfo_getkeep_cache), 0);
    rb_define_method(cRefInfo, "inspect", RUBY_METHOD_FUNC(refinfo_inspect), 0);
    rb_define_method(cRefInfo, "to_s", RUBY_METHOD_FUNC(refinfo_to_s), 0);
    rb_define_alias(cRefInfo, "to_str", "to_s");


    IDgetattr = rb_intern("getattr");
    IDreadlink = rb_intern("readlink");
    IDmknod = rb_intern("mknod");
    IDmkdir = rb_intern("mkdir");
    IDunlink = rb_intern("unlink");
    IDrmdir = rb_intern("rmdir");
    IDsymlink = rb_intern("symlink");
    IDrename = rb_intern("rename");
    IDlink = rb_intern("link");
    IDchmod = rb_intern("chmod");
    IDchown = rb_intern("chown");
    IDtruncate = rb_intern("truncate");
    IDutime = rb_intern("utime");
    IDopen = rb_intern("open");
    IDread = rb_intern("read");
    IDwrite = rb_intern("write");
    IDstatfs = rb_intern("statfs");
    IDflush = rb_intern("flush");
    IDrelease = rb_intern("release");
    IDfsync = rb_intern("fsync");
    IDopendir = rb_intern("opendir");
    IDreaddir = rb_intern("readdir");
    IDreleasedir = rb_intern("releasedir");
    IDfsyncdir = rb_intern("fsyncdir");
    IDaccess = rb_intern("access");
    IDcreate = rb_intern("create");
    IDftruncate = rb_intern("ftruncate");
    IDfgetattr = rb_intern("fgetattr");
    IDlock = rb_intern("lock");
    IDutime = rb_intern("utime");
    IDbmap = rb_intern("bmap");
    IDioctl = rb_intern("ioctl");
    IDpoll = rb_intern("poll");

    IDbavail = rb_intern("bavail");
    IDbfree = rb_intern("bfree");
    IDblocks = rb_intern("blocks");
    IDfavail = rb_intern("favail");
    IDffree = rb_intern("ffree");
    IDfiles = rb_intern("files");
    IDbsize = rb_intern("bsize");
    IDflag = rb_intern("flag");
    IDfrsize = rb_intern("frsize");
    IDfsid = rb_intern("fsid");
    IDnamemax = rb_intern("namemax");

    IDname = rb_intern("name");

    IDnthread = rb_intern("nthread");
    IDnetwork = rb_intern("network");
    IDremovable = rb_intern("removable");
    IDlogger = rb_intern("logger");
}

#endif /* defined(HAVE_FUSE_H) && !defined(HAVE_DOKAN_H) */
