// encoding:utf-8

/*
 * nazuna.c - Filesystem in userspace for ruby
 *
 * - AUTHOR : dearblue <dearblue@users.sourceforge.jp>
 * - PROJECTPAGE : http://sourceforge.jp/projects/rutsubo/
 * - LICENSE : 2-clause BSD License
 */

#ifndef NAZUNA_H
#define NAZUNA_H 1


#include <stdint.h>
#include <locale.h>
#include <ctype.h>
#include <wchar.h>
#include <fcntl.h>
#include <stdarg.h>
#include <pthread.h>
#include <errno.h>

#include <ruby.h>
#include <ruby/encoding.h>


#ifndef NAZUNA_DEBUG_OUT
#   define NAZUNA_DEBUG_OUT 0
#endif

#ifndef NAZUNA_PTHREAD_TRACE
#   define NAZUNA_PTHREAD_TRACE 0
#endif


#define ELEMENTOF(ARRAY) (sizeof(ARRAY) / sizeof((ARRAY)[0]))


#if NAZUNA_DEBUG_OUT && NAZUNA_PTHREAD_TRACE
#warning delete me after debug
#define pthread_mutex_lock(mutex) ({ __LOGS("try mutex"); int status = pthread_mutex_lock((mutex)); __LOGS("locked mutex"); status; })
#define pthread_mutex_unlock(mutex) ({ __LOGS("unlocked mutex"); int status = pthread_mutex_unlock((mutex)); status; })
#define pthread_cond_wait(cond, mutex) ({ __LOGS("wait cond"); int status = pthread_cond_wait((cond), (mutex)); __LOGS("signaled cond"); status; })
#define pthread_cond_signal(cond) ({ __LOGS("signal cond"); int status = pthread_cond_signal((cond)); status; })
#endif


#if NAZUNA_DEBUG_OUT

// FIXME: strncat の対策するべきかな・・・
static inline void
__logf(const char file[], int line, const char func[], const char format[], ...)
{
    char buf[1024] = "";
    char *ptr = buf;
    pthread_t thself = pthread_self();
    ptr += snprintf(buf, sizeof(buf), "[%08x] %s:%d:%s", *(size_t*)&thself, file, line, func);
    if (format) {
        va_list args;
        va_start(args, format);
        strncat(ptr, ": ", sizeof(buf) - (ptr - buf));
        ptr += 2;
        ptr += vsnprintf(ptr, sizeof(buf) - (ptr - buf), format, args);
    }
    strncat(ptr, ".\n", sizeof(buf) - (ptr - buf));
    ptr += 2;

    fwrite(buf, ptr - buf, 1, stderr);
    fflush(stderr);
}

#define __LOG0() __logf(__FILE__, __LINE__, __func__, NULL)
#define __LOG(...) __logf(__FILE__, __LINE__, __func__, __VA_ARGS__)
#define __LOGS(MESG) __logf(__FILE__, __LINE__, __func__, "%s", (MESG))

#else
#   define __LOG0()
#   define __LOG(...)
#   define __LOGS(MESG)
#endif


static inline VALUE
getattr(VALUE obj, ID id)
{
    return rb_funcall3(obj, id, 0, NULL);
}


static inline VALUE
frozen_str(const char mesg[])
{
    VALUE m = rb_str_new_cstr(mesg);
    OBJ_FREEZE(m);
    return m;
}


static inline VALUE
thread_join(VALUE thread)
{
    static ID IDjoin;
    if (!IDjoin) {
        IDjoin = rb_intern_const("join");
    }

    return rb_funcall2(thread, IDjoin, 0, NULL);
}


static ID IDcall;
static VALUE SYMenter, SYMleave, SYMbreak;

static inline void
calldispatch_loggerinit(void)
{
    if (!IDcall) {
        IDcall = rb_intern_const("call");
        SYMenter = ID2SYM(rb_intern("enter"));
        SYMleave = ID2SYM(rb_intern("leave"));
        SYMbreak = ID2SYM(rb_intern("break"));
        rb_gc_register_mark_object(SYMenter);
        rb_gc_register_mark_object(SYMleave);
        rb_gc_register_mark_object(SYMbreak);
    }
}

static void
calldispatch_enter(VALUE logger, ID method, int argc, VALUE argv[])
{
    VALUE argv1[argc + 2];
    argv1[0] = SYMenter;
    argv1[1] = ID2SYM(method);
    int i;
    for (i = 0; i < argc; i ++) {
        argv1[i + 2] = argv[i];
    }
    rb_funcall3(logger, IDcall, argc + 2, argv1);
}

struct calldispatch_args
{
    VALUE status;
    VALUE logger;
    VALUE recv;
    ID method;
    int argc;
    VALUE *argv;
    VALUE (*block)(ANYARGS);
    void *blockargs;
};

static inline VALUE
calldispatch_try(struct calldispatch_args *args)
{
    if (args->block) {
        args->status = rb_block_call(args->recv, args->method, args->argc, args->argv,
                                     args->block, (VALUE)(args->blockargs));
    } else {
        args->status = rb_funcall3(args->recv, args->method, args->argc, args->argv);
    }

    return args->status;
}

static VALUE
calldispatch_ensure(struct calldispatch_args *args)
{
    VALUE argv1[3];
    if (args->status == Qundef) {
        argv1[0] = SYMbreak;
        argv1[1] = ID2SYM(args->method);
        argv1[2] = rb_errinfo();
    } else {
        argv1[0] = SYMleave;
        argv1[1] = ID2SYM(args->method);
        argv1[2] = args->status;
    }
    rb_funcall3(args->logger, IDcall, 3, argv1);

    return Qnil;
}

static VALUE
calldispatch(VALUE logger, VALUE recv, ID method, int argc, VALUE argv[],
             VALUE (*block)(ANYARGS), void *blockargs)
{
    struct calldispatch_args args = {
        .status = Qundef,
        .logger = logger,
        .recv = recv,
        .method = method,
        .argc = argc,
        .argv = argv,
        .block = block,
        .blockargs = blockargs,
    };

    if (NIL_P(logger)) {
        return calldispatch_try(&args);
    } else {
        calldispatch_loggerinit();
        calldispatch_enter(logger, method, argc, argv);

        return rb_ensure(RUBY_METHOD_FUNC(calldispatch_try), (VALUE)(&args),
                         RUBY_METHOD_FUNC(calldispatch_ensure), (VALUE)(&args));
    }
}


// 可変引数部分の評価順を左から右に固定する
#define CALLDISPATCH(METHOD, NAZUNA, ...)                                \
    ({                                                                   \
        VALUE __args[] = { __VA_ARGS__ };                                \
        calldispatch((NAZUNA)->logger, (NAZUNA)->dispatch, ID ## METHOD, \
                     ELEMENTOF(__args), __args, NULL, NULL);             \
    })                                                                   \

#define CALLDISPATCH_WITHBLOCK_XXXX(LOGGER, RECV, METHOD, ARGV, BLOCK, BLOCKARGS) \
    calldispatch((LOGGER), (RECV), ID ## METHOD,                             \
                 ELEMENTOF(ARGV), (ARGV), (BLOCK), (BLOCKARGS))              \

#define CALLDISPATCH_WITHBLOCK(METHOD, NAZUNA, BLOCK, BLOCKARGS, ...)    \
    ({                                                                   \
        VALUE __args[] = { __VA_ARGS__ };                                \
        calldispatch((NAZUNA)->logger, (NAZUNA)->dispatch, ID ## METHOD, \
                     ELEMENTOF(__args), __args, (BLOCK), (BLOCKARGS));   \
    })                                                                   \


#define FUNCALL3(OBJ, ID, ...)                                  \
    ({                                                          \
        VALUE __args[] = { __VA_ARGS__ };                       \
        rb_funcall3((OBJ), (ID), ELEMENTOF(__args), __args);    \
    })                                                          \




/*
 * call-seq:
 * Nazuna~::Utils.error?(status)
 *
 * エラーステータスかどうかを確認する。
 *
 * エラーステータスとみなすオブジェクトは以下の通り:
 * - nil: ENOSYS とみなされる
 * - 負の整数: EXXX を2の補数で表現したものとみなされる
 * - Errno::EXXX クラス
 * - Errno::EXXX インスタンス 
 *
 * RETURN:: エラーであれば負の整数を、違うならば nil を返す
 * status:: ステータスコード
 */
static inline VALUE
utils_i_is_error(VALUE obj, VALUE status)
{
    if (NIL_P(status)) {
        return Qtrue;
    } else if (rb_obj_is_kind_of(status, rb_cInteger) && FUNCALL3(status, rb_intern("<"), INT2FIX(0))) {
        return Qtrue;
    } else if (rb_obj_is_kind_of(status, rb_eSystemCallError)) {
        return Qtrue;
    } else if (rb_obj_is_kind_of(status, rb_cClass) && rb_class_inherited_p(status, rb_eSystemCallError)) {
        return Qtrue;
    } else {
        return Qfalse;
    }
}


static inline VALUE
utils_i_error_code(VALUE obj, VALUE status)
{
    // static ID IDlt, IDerrno, IDErrno;
    static ID IDlt;

    if (!IDlt) {
        IDlt = rb_intern_const("<");
    }

    if (NIL_P(status)) {
        return INT2FIX(-ENOSYS);
    } else if (rb_obj_is_kind_of(status, rb_cInteger) && FUNCALL3(status, IDlt, INT2FIX(0))) {
        return status;
    } else if (rb_obj_is_kind_of(status, rb_eSystemCallError)) {
        return INT2NUM(-NUM2INT(FUNCALL3(status, rb_intern("errno"))));
    } else if (rb_obj_is_kind_of(status, rb_cClass) && RTEST(rb_class_inherited_p(status, rb_eSystemCallError))) {
        return INT2NUM(-NUM2INT(rb_const_get(status, rb_intern("Errno"))));
    } else {
        return Qnil;
    }
}


#endif /* !defined(NAZUNA_H) */
