#vim: set fileencoding:utf-8

class << Nazuna # :nodoc: all
    module Dokan
        class FileProxy < Struct.new(:obj)
            BasicStruct = superclass
        end

        module Utils
            module_function
            def map_error(err)
                err = Utils.error_code(err) or return nil
                -Utils.map_winerr(-err)
            end
        end

        module STATMODE2FLAGS
            include Dokan::Constants

            def flags
                flags = 0
                flags |= FILE_ATTRIBUTE_DIRECTORY if stat.directory?
#                flags |= STANDARD_RIGHTS_ALL
                flags
            end
        end

        class STAT2BYHANDLEFILEINFORMATION < Struct.new(:stat)
            BasicStruct = superclass

            include STATMODE2FLAGS

            def ctime; stat.ctime; end
            def atime; stat.atime; end
            def mtime; stat.mtime; end
            def dev; stat.dev; end
            def size; stat.size; end
            def nlink; stat.nlink; end
            def ino; stat.ino; end
        end

        class STAT2FINDDATA < Struct.new(:stat)
            BasicStruct = superclass

            include STATMODE2FLAGS

            def ctime; stat.ctime; end
            def atime; stat.atime; end
            def mtime; stat.mtime; end
            def size; stat.size; end
        end

        class Bridge < Struct.new(:userfs,     # 実際にIO要求を処理するオブジェクト
                                  :contextmap) # contextmap: { integer => object, ... }
            BasicStruct = superclass

            include Dokan::Constants
            include Errno

            def initialize(userfs)
                super userfs, ContextMap.new
            end

            def CreateFile(dokan, access, share, disposition, flags)
                return Utils.map_error(ENOMEM) if contextmap.full?
                path = dokan.path
                status = Utils.map_error(st = userfs.stat(path)) and st = nil
                case
                when st && st.directory?
                    return Utils.map_error(ENOSYS) unless dir = userfs.opendir(path)
                    status = Utils.map_error(dir) and return status
                    contextmap.regist(FileProxy.new(dir))
                else
                    mode = Utils.map_file_mode(access, share, disposition, flags)
                    return Utils.map_error(ENOSYS) unless file = userfs.open(path, mode)
                    status = Utils.map_error(file) and return status
                    contextmap.regist(file)
                end
            end

            def OpenDirectory(dokan)
                return Utils.map_error(ENOMEM) if contextmap.full?
                path = dokan.path
                return Utils.map_error(ENOSYS) unless dir = userfs.opendir(path)
                status = Utils.map_error(dir) and return status
                contextmap.regist(dir)
            end

            def Cleanup(dokan)
                Utils.map_error(ENOSYS)
            end

            def CloseFile(dokan)
                return 0 unless fd = contextmap.delete(dokan.context)
                fd = fd.obj if fd.kind_of?(FileProxy)
                Utils.map_error(userfs.close(fd)) or 0
            end

            def ReadFile(dokan, offset, size)
                return Utils.map_error(EBADF) unless fd = contextmap[dokan.context]
                return Utils.map_error(ENOSYS) if fd.kind_of?(FileProxy)
                Utils.map_error(buf = userfs.read(fd, offset, size) || "") or buf
            end

            def WriteFile(dokan, offset, buf)
                return Utils.map_error(EBADF) unless fd = contextmap[dokan.context]
                return Utils.map_error(ENOSYS) if fd.kind_of?(FileProxy)
                Utils.map_error(size = userfs.write(fd, offset, buf)) or size
            end

            def FlushFileBuffers(dokan)
                return Utils.map_error(EBADF) unless fd = contextmap[dokan.context]
                return Utils.map_error(ENOSYS) if fd.kind_of?(FileProxy)
                Utils.map_error(userfs.flush(fd)) or 0
            end

            def GetFileInformation(dokan)
                return Utils.map_error(EBADF) unless fd = contextmap[dokan.context]
                if fd && !fd.kind_of?(FileProxy)
                    st = userfs.fstat(fd)
                else
                    st = userfs.stat(dokan.path)
                end
                return -1 unless st
                status = Utils.map_error(st) and return status
                STAT2BYHANDLEFILEINFORMATION.new(st)
            end

            def FindFiles(dokan)
                path = dokan.path
                return Utils.map_error(EBADF) unless dir = contextmap[dokan.context]
                if dir
                    dir = dir.obj if dir.kind_of?(FileProxy)
                else
                    dir = tmp = userfs.opendir(path)
                    status = Utils.map_error(dir) and return - status
                end
                path += "/" unless path[-1] == "/"
                userfs.readdir(dir, 0) do |name, stat|
                    if stat.nil?
                        stat = userfs.stat(path + name)
                        Utils.error?(stat) and next
                    end
                    yield(name, STAT2FINDDATA.new(stat))
                end

                0
            ensure
                userfs.close(tmp) if tmp
            end

            def SetFileAttributes(dokan, attr)
                - Utils.map_error(ENOSYS)
            end

            def SetFileTime(dokan)
                - Utils.map_error(ENOSYS)
            end

            def DeleteFile(dokan, path)
                - Utils.map_error(userfs.unlink(String(path))) || 0
            end

            def CreateDirectory(dokan, path)
                - Utils.map_error(userfs.mkdir(String(path), 0)) || 0
            end

            def DeleteDirectory(dokan, path)
                - Utils.map_error(userfs.rmdir(String(path))) || 0
            end

            def MoveFile(dokan, oldpath, newpath, overwrite)
                - Utils.map_error(userfs.rename(String(oldpath), String(newpath))) || 0
            end

            def SetEndOfFile(dokan, length)
                file = contextmap[dokan.context]
                path = dokan.path
                case file
                when nil
                    - Utils.map_error(userfs.truncate(path, length)) || 0
                when FileProxy
                    - Utils.map_error(EBADF)
                else
                    - Utils.map_error(userfs.ftruncate(file, length)) || 0
                end
            end

            def SetAllocationSize(dokan, length)
                file = contextmap[dokan.context]
                path = dokan.path
                case file
                when nil
                    - Utils.map_error(userfs.truncate(path, length)) || 0
                when FileProxy
                    - Utils.map_error(EBADF)
                else
                    - Utils.map_error(userfs.ftruncate(file, length)) || 0
                end
            end

            def LockFile(dokan, offset, length)
                - Utils.map_error(ENOSYS)
                # userfs.lock
            end

            def UnlockFile(dokan, offset, length)
                - Utils.map_error(ENOSYS)
                # userfs.lock
            end

            DEFAULT_FREEBYTESAVAILABLE = 1 << 30
            DEFAULT_TOTALNUMBEROFBYTES = 1 << 30
            DEFAULT_TOTALNUMBEROFFREEBYTES = 1 << 30
            def GetDiskFreeSpace(dokan)
                # TODO: userfsを通して情報を得るようにする
                [
                    DEFAULT_FREEBYTESAVAILABLE,
                    DEFAULT_TOTALNUMBEROFBYTES,
                    DEFAULT_TOTALNUMBEROFFREEBYTES,
                ]
            end

            DEFAULT_FILESYSTEMNAME = "nazuna".freeze
            DEFAULT_VOLUMELABEL = "レナはフランカーに乗れていいよな".freeze
            DEFAULT_SERIAL = 3141592653
            DEFAULT_FILEPATHLENGTH = 2048
            DEFAULT_FILESYSTEMFLAGS = FS_CASE_IS_PRESERVED | FS_CASE_SENSITIVE | FS_UNICODE_STORED_ON_DISK

            def GetVolumeInformation(dokan)
                # TODO: userfsを通して情報を得るようにする
                [
                    DEFAULT_VOLUMELABEL,
                    DEFAULT_SERIAL,
                    DEFAULT_FILEPATHLENGTH,
                    DEFAULT_FILESYSTEMFLAGS,
                    DEFAULT_FILESYSTEMNAME,
                ]
            end

            def Unmount(dokan)
                except = cleanupcontext!
                status = userfs.unmount
                raise except if except
                status
            end

            def GetFileSecurity(dokan, info)
                - Utils.map_error(ENOSYS)
            end

            def SetFileSecurity(dokan, security)
                - Utils.map_error(ENOSYS)
            end

            private
            def cleanupcontext
                except = cleanupcontext!
                raise except if except
                nil
            end

            private
            def cleanupcontext!
                except = nil
                contextmap.each_value do |fd|
                    begin
                        fd.respond_to?(:close) and fd.close
                    rescue Object
                        except ||= $!
                    end
                end
                contextmap.clear
                except
            end

            def self.mount(userfs, mountpoint, *opts)
                bridge = new(userfs)
                Dokan.mount(bridge, mountpoint, *opts)
            ensure
                bridge.instance_eval { cleanupcontext }
            end
        end

        class NodeBridge < Struct.new(:fs, :contextmap)
            BasicStruct = superclass

            def CreateFile(dokan, access, share, disposition, flags)
                return Utils.map_error(ENOMEM) if contextmap.full?
                path = dokan.path.split(?/)
                path.shift if !path.empty? && path[0].empty?
                node = fs.root
                path.each_with_index do |name, i|
                    node = fs.lookup(node, name)
                end

                status = Utils.map_error(st = userfs.stat(path)) and st = nil
                case
                when st && st.directory?
                    return Utils.map_error(ENOSYS) unless dir = userfs.opendir(path)
                    status = Utils.map_error(dir) and return status
                    contextmap.regist(FileProxy.new(dir))
                else
                    mode = Utils.map_file_mode(access, share, disposition, flags)
                    return Utils.map_error(ENOSYS) unless file = userfs.open(path, mode)
                    status = Utils.map_error(file) and return status
                    contextmap.regist(file)
                end
            end

            def self.mount(fs, mountpoint, *opts)
                Dokan.mount(bridge = new(fs), mountpoint, *opts)
            ensure
                bridge.instance_eval { cleanup_context } if bridge
            end
        end
    end
end
