#vim: set encoding:utf-8

#:nodoc: all
class << Nazuna
    class Bridge < Struct.new(:userfs,     # 実際にIO要求を処理するインスタンス
                              :contextmap) # contextmap: Integer => Object
        BasicStruct = superclass

        include Constants
        include ::Errno

        def initialize(userfs)
            super userfs, ContextMap.new
        end

        def status?(status)
            case
            when status.nil?
                ENOSYS
            when status.kind_of?(Integer) && status < 0,
                 SystemCallError === status,
                 Class === status && SystemCallError > status
                status
            else
                nil
            end
        end


        # 戻り値: File::Stat インスタンスか、それに似たオブジェクト
        def getattr(path)
            userfs.stat(path)
        end

        # 戻り値: 文字列
        def readlink(path, size)
            userfs.readlink(path, size)
        end

        # 戻り値: 0
        def mknod(path, mode, dev)
            userfs.mknod(path, mode, dev)
        end

        def mkdir(path, mode)
            userfs.mkdir(path, mode)
        end

        def unlink(path)
            userfs.unlink(path)
        end

        def rmdir(path)
            userfs.rmdir(path)
        end

        def symlink(target, linkname)
            userfs.symlink(target, linkname)
        end

        def rename(oldpath, newpath)
            userfs.rename(oldpath, newpath)
        end

        def link(target, linkname)
            userfs.link(target, linkname)
        end

        def chmod(path, mode)
            userfs.chmod(path, mode)
        end

        def chown(path, uid, gid)
            userfs.chown(path, uid, gid)
        end

        def truncate(path, size)
            userfs.truncate(path, size)
        end

        def utime(path, actime, modtime)
            userfs.utime(path, actime, modtime)
        end

        def open(info)
            return ENOMEM if contextmap.full?
            status?(file = userfs.open(info.path, info.flags, 0)) and return file
            contextmap.regist(file)
        end

        def read(info, offset, size)
            file = contextmap[info.fd] or return EBADF
            userfs.read(file, offset, size)
        end

        def write(info, offset, buf)
            file = contextmap[info.fd] or return EBADF
            userfs.write(file, offset, buf)
        end

        def statfs(path)
            StatVFS.new(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1024)
        end

        def flush(info)
            file = contextmap[info.fd] or return EBADF
            userfs.flush(file)
        end

        # 戻り値: 0
        def release(info)
            return EBADF unless file = contextmap.delete(info.fd)
            userfs.close(file)
        end

        # 戻り値: 0
        def fsync(info, isdatasync)
            file = contextmap[info.fd] or return EBADF
            userfs.fsync(file)
        end

        # 戻り値: 0
        def opendir(info)
            return ENOMEM if contextmap.full?
            status?(dir = userfs.opendir(info.path)) and return dir
            contextmap.regist(dir)
        end

        # yield(name, stat)
        # 戻り値: 0
        def readdir(info, offset, &block)
            dir = contextmap[info.fd] or return EBADF
            path = info.path
            userfs.readdir(dir, offset) do |name, stat|
                unless stat
                    stat = userfs.stat(File.join(path, name))
                    next unless stat
                end
                yield(name, stat)
            end

            0
        end

        # 戻り値: File::Stat
        def releasedir(info)
            return EBADF unless dir = contextmap.delete(info.fd)
            userfs.close(dir)
        end

        # 戻り値: 0
        def fsyncdir(info, datasync)
            dir = contextmap[info.fd] or return EBADF
            userfs.fsyncdir(dir, datasync)
        end

        # 戻り値: 0
        def access(path, mask)
            # userfs.access(path, mask)
            0
        end

        # 戻り値: 0
        def create(info, mode)
            file = contextmap[info.fd] or return EBADF
            userfs.create(file, mode)
        end

        # 戻り値: 0
        def ftruncate(info, offset)
            file = contextmap[info.fd] or return EBADF
            userfs.ftruncate(file, offset)
        end

        # 戻り値: stat
        def fgetattr(info)
            file = contextmap[info.fd] or return EBADF
            userfs.fstat(file)
        end

        # NOT READY
        def lock(info, cmd, flock)
            file = contextmap[info.fd] or return EBADF
            userfs.lock(file, cmd, flock)
        end

        # NOT READY
        def bmap(path, blocksize, idx)
            file = contextmap[info.fd] or return EBADF
            userfs.bmap(file, blocksize, idx)
        end


        private
        def cleanupcontext
            except = cleanupcontext!
            raise except if except
            nil
        end

        private
        def cleanupcontext!
            except = nil
            contextmap.each_value do |fd|
                begin
                    fd.respond_to?(:close) and fd.close
                rescue Object
                    except ||= $!
                end
            end
            contextmap.clear
            except
        end

        # userfs: IO要求を処理するインスタンス
        # mountpoint: マウントポイント (String)
        # opts:
        #   nthread: 5       # IO要求処理スレッドの数 (最大20くらい?)
        #   removable: false # 着脱式ストレージ
        #   network: false   # ネットワークストレージ
        def self.mount(userfs, mountpoint, *opts)
            bridge = new(userfs)
            Nazuna.singleton_class.mount(bridge, mountpoint, *opts)
        ensure
            bridge.instance_eval { cleanupcontext }
        end
    end

    class StatVFS < Struct.new(:bavail,  # fsblkcnt_t    f_bavail;  /* Number of blocks */
                               :bfree,   # fsblkcnt_t    f_bfree;
                               :blocks,  # fsblkcnt_t    f_blocks;
                               :favail,  # fsfilcnt_t    f_favail;  /* Number of files (e.g., inodes) */
                               :ffree,   # fsfilcnt_t    f_ffree;
                               :files,   # fsfilcnt_t    f_files;
                               :bsize,   # unsigned long f_bsize;   /* Size of blocks counted above */
                               :flag,    # unsigned long f_flag;
                               :frsize,  # unsigned long f_frsize;  /* Size of fragments */
                               :fsid,    # unsigned long f_fsid;    /* Not meaningful */
                               :namemax) # unsigned long f_namemax; /* Same as pathconf(_PC_NAME_MAX) */
    end
end
