#vim: set fileencoding=utf-8

# nazuna.rb -
#
# * Author : dearblue <dearblue@users.sourceforge.jp>
# * License : 2-clause BSD License
# * Project Page : http://sourceforge.jp/projects/rutsubo/


require "nazuna.so"


#---
# this blank comment is recognized not binary file for rdoc!
#                                                                     # 
#                                                                     # 
#                                                                     # 
#                                                                     # 
#                                                                     # 
#                                                                     # 
#                                                                     # 
#                                                                     # 
#                                                                     # 
#                                                                     # 
#+++


# Rubyでファイルシステムを構築するためのフレームワークです。
#
# あなたが用意するファイルシステムクラスに <tt>include Nazuna::Template</tt> して、インスタンスの #mount によってマウントします。
#
# もしくは <tt>Nazuna.mount</tt> をブロック付き呼び出しで行えば、それでもファイルシステムが構築できます。
#
# <tt>require "nazuna"</tt> は、$SAFE が 1 以上だと *SecurityError* 冷害が発生します。
#
# 実際にどう使うかは、samples ディレクトリのそれぞれのコードを見てください。
module Nazuna
    # call-seq:
    # Nazuna.mount(mountpoint, *opts) { |io_request| ... } -> nil
    #
    # mount を呼び出す時に与えるブロックが IO 要求を処理します。
    #
    # ブロック引数には Nazuna::IORequests モジュール内で定義されている各クラスのインスタンスが与えられます。
    #
    # Nazuna.mount を呼び出したスレッドはその後の処理を停止させます。
    # begin - ensure - end の ensure 節は ruby のルール内で実行されます。
    #
    # アンマウントするには以下の方法があります:
    # - 他のスレッドから Nazuna.mount したスレッドを Thread#kill する
    # - Nazuna.mount に与えたディスパッチのメソッド内で大域脱出する (例外の発生やブロック内での return など)
    # - マウントポイントをアンマウントする (Windows + dokan であれば dokanctl.exe /u ??? 、FreeBSD + fusefs であれば umount ???)
    #
    # <tt>$SAFE</tt> が 2 以上のスレッドが Nazuna.mount を呼び出すと +SecurityError+ 例外が発生します。
    #
    # [RETURN]      常に nil です。
    # [mountpoint]  ファイルシステムとして結合先となるパスを指定します。文字列を与えてください。
    # [nthread]     IO 要求処理を行うスレッド数。既定値は +5+ です。
    # [safelevel]   IO 要求処理スレッドのセーフレベルを指定します。既定値は +nil+ で、+#mount+ したスレッドと同じセーフレベルです。
    # [logger]      デバッグ用途です。既定値は +nil+ で、利用しないことを意味します。
    #
    #               Proc インスタンスか、#call を持ち同じ挙動を行うオブジェクトを指定します。
    #                   proc { |event, request, *args_or_status| ... }
    #               [RETURN]    ブロックの戻り値は無視されます。
    #               [event]     IO 要求のタイミング
    #                           - Ruby で処理が開始される時は +:enter+
    #                           - 処理が終わってシステムに返答するときは +:leave+
    #                           - Ruby で処理中に例外によって強制的にシステムに戻る時は +:break+
    #                           がそれぞれ与えられます。
    #               [request]   IO 要求名がシンボルで与えられます。環境依存です。
    #               [args_or_status]    - eventが +:enter+ の場合は、IO 要求の生の引数が与えられます。環境依存です。
    #                                     引数のすり替えは出来ませんが、各要素をいじると不具合の原因となります。
    #                                   - eventが +:leave+ の場合は、IO 要求処理後の返却値が与えられます。環境依存です。
    #                                     引数のすり替えは出来ませんが、各要素をいじると不具合の原因となります。
    #                                   - eventが +:break+ の場合は、例外発生時ならば例外のインスタンスが、大域脱出時ならば nil が与えられます。
    #
    # 例えばこんな感じで呼び出します (不完全なコードでまともに動作しません):
    #   Nazuna.mount("/mnt/nazuna", nthread: 2) do |io_request|
    #       case io_request
    #       when Nazuna::Ropen
    #           なんたらかんたら
    #           next ファイルオブジェクト
    #       when Nazuna::Ropendir
    #           なんたらかんたら
    #           next ディレクトリオブジェクト
    #       else
    #           next Errno::ENOSYS
    #       end
    #   end
    def self.mount(mountpoint, *opts, &block)
        LambdaFileSystem.new(block).mount(mountpoint, *opts)
    end


    # 常に Errno::ENOSYS を返す、雛形としてのファイルシステムモジュール。
    #
    # +include+ して利用します (同時に +Nazuna+ モジュールも +include+ されます)。
    #
    # IO 要求処理メソッドは正常に処理を完了した場合、要求によって返すべき値が決まっており多くが 0 を、それ以外はそれぞれのメソッドの説明を見てください。
    # また、すべてのメソッドは Errno::EXXX クラスかそのインスタンスを返すことで、失敗を報告することが出来ます。nil を返した場合は Errno::ENOSYS と同じ意味となります。
    #
    # 必要な例外はメソッド内で捕捉するべきです。nazuna は例外が IO 要求処理メソッドを飛び出した場合、スレッドをまたいで #mount したスレッドに搬送するためです。
    # ただしあらゆる例外を補足するべきだと言っているわけではありません。誤った構文や存在しないメソッドを呼び出すようなスクリプト上のエラーはそのまま通したほうがいいでしょう。
    # その一方で IO 要求によって存在しないファイルを開いた場合、例外を捕捉してエラーとして return するべきです。
    # あくまで IO 要求に従って発生した例外は捕捉してエラーとして返す、ということです。
    module Template
        include Nazuna

        # call-seq:
        # Nazuna::Template#mount(mountpoint, nthread: 5, safelevel: 0, logger: nil) -> nil
        #
        # <tt>include Nazuna::Template</tt> されたクラスのインスタンスをシステムと結びつけます。
        # ファイルシステム IO 要求は、そのインスタンスに対する各メソッドを呼び出すことで転送されます。
        #
        # IO要求に対するそれぞれのインスタンスメソッドを再定義することで、ファイルシステムが構築できます
        # (必ずしも全てのメソッドを定義する必要はありません)。
        #
        # Nazuna::Template#mount の引数については、Nazuna.mount の引数と同じです。そちらを参照してください。
        def mount(mountpoint, *opts)
            Nazuna.singleton_class::Bridge.mount(self, mountpoint, *opts)
        end

        def __NOT_IMPLEMENTED_IO_OPERATION__(id) # :nodoc:
            @__NOT_IMPLEMENTED_IO_OPERATION__ ||= {}
            unless @__NOT_IMPLEMENTED_IO_OPERATION__[id]
                @__NOT_IMPLEMENTED_IO_OPERATION__[id] = true
                $stderr.puts "#$0: file system has not operation - #{self.class}\##{id}.\n"
            end
            Errno::ENOSYS
        end

        # ファイルの情報を得ます。
        #
        # [RETURN]  File::Stat インスタンスか、同じ振る舞いをするオブジェクトを返します。
        #
        # [path]    ファイルパス
        def stat(path)
            __NOT_IMPLEMENTED_IO_OPERATION__(:stat)
        end

        # シンボリックリンクのリンク先を返します。
        #
        # [RETURN]  文字列
        #
        # [path]    ファイルパス
        #
        # [size]    バッファの最大バイト数
        def readlink(path, size)
            __NOT_IMPLEMENTED_IO_OPERATION__(:readlink)
        end

        # 戻り値: 0
        def mknod(path, mode, dev)
            __NOT_IMPLEMENTED_IO_OPERATION__(:mknod)
        end

        # 戻り値: 0
        def mkdir(path, mode)
            __NOT_IMPLEMENTED_IO_OPERATION__(:mkdir)
        end

        # 戻り値: 0
        def unlink(path)
            __NOT_IMPLEMENTED_IO_OPERATION__(:unlink)
        end

        # 戻り値: 0
        def rmdir(path)
            __NOT_IMPLEMENTED_IO_OPERATION__(:rmdir)
        end

        # 戻り値: 0
        def symlink(target, linkname)
            __NOT_IMPLEMENTED_IO_OPERATION__(:symlink)
        end

        # 戻り値: 0
        def rename(oldpath, newpath)
            __NOT_IMPLEMENTED_IO_OPERATION__(:rename)
        end

        # 戻り値: 0
        def link(target, linkname)
            __NOT_IMPLEMENTED_IO_OPERATION__(:link)
        end

        # 戻り値: 0
        def chmod(path, mode)
            __NOT_IMPLEMENTED_IO_OPERATION__(:chmod)
        end

        # 戻り値: 0
        def chown(path, uid, gid)
            __NOT_IMPLEMENTED_IO_OPERATION__(:chown)
        end

        # 戻り値: 0
        def truncate(path, offset)
            __NOT_IMPLEMENTED_IO_OPERATION__(:truncate)
        end

        # 戻り値: 0
        def utime(path, actime, modtime)
            __NOT_IMPLEMENTED_IO_OPERATION__(:utime)
        end

        # 戻り値: object
        def open(path, mode)
            __NOT_IMPLEMENTED_IO_OPERATION__(:open)
        end

        # 戻り値: 文字列
        def read(file, offset, size)
            __NOT_IMPLEMENTED_IO_OPERATION__(:read)
        end

        # 戻り値: 書き込んだバイト数
        def write(file, offset, buf)
            __NOT_IMPLEMENTED_IO_OPERATION__(:write)
        end

        # 戻り値: 0
        def flush(file)
            __NOT_IMPLEMENTED_IO_OPERATION__(:flush)
        end

        # 戻り値: 0
        def close(file)
            release(file)
        end

        # 戻り値: 0
        def release(file)
            __NOT_IMPLEMENTED_IO_OPERATION__(:release)
        end

        # 戻り値: 0
        def fsync(file)
            __NOT_IMPLEMENTED_IO_OPERATION__(:fsync)
        end

        # 戻り値: object
        def opendir(path)
            __NOT_IMPLEMENTED_IO_OPERATION__(:opendir)
        end

        # yield(name, stat = nil)
        # 戻り値: 0
        def readdir(dir, offset, &block)
            __NOT_IMPLEMENTED_IO_OPERATION__(:readdir)
        end

        # 戻り値: 0
        def fsyncdir(dir, datasync)
            __NOT_IMPLEMENTED_IO_OPERATION__(:fsyncdir)
        end

        # 戻り値: 0
        def access(path, mask)
            __NOT_IMPLEMENTED_IO_OPERATION__(:access)
        end

        # 戻り値: 0
        def create(file, mode)
            __NOT_IMPLEMENTED_IO_OPERATION__(:create)
        end

        # 戻り値: 0
        def ftruncate(file, offset)
            __NOT_IMPLEMENTED_IO_OPERATION__(:ftruncate)
        end

        # 戻り値: stat
        def fstat(file)
            __NOT_IMPLEMENTED_IO_OPERATION__(:fstat)
        end

        # 戻り値: 0
        #
        # TODO: IMPLEMENT ME
        def lock(file, cmd, flock)
            __NOT_IMPLEMENTED_IO_OPERATION__(:lock)
        end
    end

    # 検証のための擬似接続モジュール
    #
    # ここで言う擬似接続というのは、実際にmountすることはしないという意味。検証目的での利用を想定。
    #
    # クラスに include してインスタンスを作成し #mount するか、Nazuna::Test.mount する。
    module Test
        include Template

        def mount(mountpoint, *opts)
            Try.mount(self, mountpoint, *opts)
        end

        def self.mount(mountpoint, *opts, &block)
            LambdaFileSystem.new(block).extend(self).mount(mountpoint, *opts)
        end


        # 実際の検証を実行するためのモジュール
        #
        # 利用者が直接利用することは無いでしょう。
        module Try
            def self.status?(status)
                case
                when status.nil?
                    -Errno::ENOSYS::Errno
                when status.kind_of?(SystemCallError)
                    -status.errno
                when status.kind_of?(Integer) && status <= 0
                    status
                when status.kind_of?(Module) && SystemCallError > status
                    -status::Errno
                else
                    nil
                end
            end

            def self.mount(fs, mountpoint, *opts)
                # TODO: 引数についても正しいかを検証する。

                if err = status?(file = fs.open("/", 0, 0))
                    $stderr.puts "#$0: failed open - status: #{SystemCallError.new(-err)} (#{file.inspect}).\n"
                else
                    begin
                        0
                    ensure
                        file.close
                    end
                end
            end
        end
    end


    # Rubyのブロックを用いたファイルシステムにおいて、IO要求を受け渡す場合に使われるクラスをまとめたモジュール。
    #
    #
    module IORequests
        # ファイルを開くIO要求です。
        #
        # [path]    開くファイルパスです。
        # [mode]    開くファイルのモードです。File.open の mode 引数としてそのまま渡せます。
        # [RETURN]  開いたファイルオブジェクトを返してください。
        Ropen    = Struct.new(:path, :mode)

        # ディレクトリを開くIO要求です。
        #
        # [path]    開くディレクトリのパスです。
        # [RETURN]  開いたディレクトリオブジェクトを返してください。
        Ropendir = Struct.new(:path)

        # ディレクトリ内容を読み込むIO要求です。
        #
        # [dir]     opendir で返したディレクトリオブジェクトです。
        # [offset]  ディレクトリの読み込み位置です。
        # [yield]   エントリを報告するためのメソッドです。
        #
        #           <tt>req.yield(name)</tt> もしくは <tt>req.yield(name, stat)</tt> のように指定します。
        #
        #           yield が真を返したらそれ以上の報告は打ち切ってください (内部ライブラリ中のバッファがすでにいっぱいです)。
        #
        #           [NOTE] <tt>req.yield(name, stat)</tt> で渡すほうが高速となります。
        #
        # [RETURN]  0 を返してください。
        Rreaddir = Class.new(Struct.new(:dir, :offset, :yield))

        # ディレクトリを作成するIO要求です。
        #
        # [path]    作成するディレクトリのパスです。
        # [RETURN]  0 を返してください。
        Rmkdir   = Struct.new(:path)

        # ディレクトリを破棄するIO要求です。
        #
        # [path]    破棄するディレクトリのパスです。
        # [RETURN]  0 を返してください。
        Rrmdir   = Struct.new(:path)

        # ファイルを削除するIO要求です。
        #
        # [path]    破棄するファイルのパスです。
        # [RETURN]  0 を返してください。
        Runlink  = Struct.new(:path)

        # ファイル情報を取得するIO要求です。
        #
        # [path]    破棄するディレクトリのパスです。
        # [RETURN]  File::Stat インスタンスか、同じ挙動をするオブジェクトを返してください。
        Rstat    = Struct.new(:path)

        # 開いてあるファイル情報を取得するIO要求です。
        #
        # [file]    open もしくは opendir で返したオブジェクトです。
        # [RETURN]  File::Stat インスタンスか、同じ挙動をするオブジェクトを返してください。
        Rfstat   = Struct.new(:file)

        # ファイルデータの読み込みIO要求です。
        #
        # [file]    open で返したオブジェクトです。
        # [offset]  バイト単位のファイル読み込み位置です。0 を含めた正の整数です。
        # [size]    バイト単位の読み込み量です。
        # [RETURN]  読み込んだデータを String インスタンスにして返してください。EOF のばあいは空文字を返してください。
        Rread    = Struct.new(:file, :offset, :size)

        # ファイルデータの書き込みIO要求です。
        #
        # [file]    open で返したオブジェクトです。
        # [offset]  バイト単位のファイル読み込み位置です。0 を含めた正の整数です。
        # [buf]     書き込みデータです。
        # [RETURN]  実際に書き込んだ (処理した) データ量をバイト単位で返してください。
        Rwrite   = Struct.new(:file, :offset, :buf)

        # ファイルもしくはディレクトリを閉じるIO要求です。
        #
        # [file]    open もしくは opendir で返したオブジェクトです。
        # [RETURN]  0 を返してください。
        Rclose   = Struct.new(:file)

        # ファイルのロック・ロック解除のIO要求です。
        #
        # [RETURN]  0 を返してください。
        # [file]    open で返したオブジェクトです。
        # [type]    ロックモードもしくはロック解除です。
        # [range]   操作範囲を示します。range.end が 0 であればファイルの末尾を意味します。
        Rlock    = Struct.new(:file, :type, :range)


        class Rreaddir # :nodoc:
            BasicStruct = superclass

            def yield(*args)
                super().call(*args)
            end
        end
    end
    include IORequests


    # Rubyのブロックを用いてファイルシステムを構築するためのクラス。
    #
    # Nazuna.mount が表面上の手続きを行うため、利用者が直接利用する必要はありません。
    class LambdaFileSystem < Struct.new(:lambda) # :nodoc:
        BasicStruct = superclass

        include IORequests
        include Template

        def stat(path)
            lambda.(Rstat.new(path))
        end

        def close(file)
            lambda.(Rclose.new(file))
        end

        def open(path, mode)
            lambda.(Ropen.new(path, mode))
        end

        def fstat(file)
            lambda.(Rfstat.new(file))
        end

        def read(file, offset, size)
            lambda.(Rread.new(file, offset, size))
        end

        def write(file, offset, buf)
            lambda.(Rwrite.new(file, offset, buf))
        end

        def opendir(path)
            lambda.(Ropendir.new(path))
        end

        def readdir(dir, offset, &block)
            lambda.(Rreaddir.new(dir, offset, block))
        end

        def mkdir(path)
            lambda.(Rmkdir.new(path))
        end

        def rmdir(path)
            lambda.(Rrmdir.new(path))
        end

        def unlink(path)
            lambda.(Runlink.new(path))
        end
    end


    # ファイル属性を返す場合に利用できるファイル情報クラスで、File::Stat に似せてあります。
    #
    # File.stat が利用できない場合 (あなたの独自ファイルシステムではそうなるでしょう)、これを代わりとすることが出来ます。
    #
    # 初期化は Nazuna::Stat.new が基本ですが、簡単に作成するための Nazuna::Stat.file および Nazuna::Stat.directory があります。
    class Stat < Struct.new(:dev, :ino, :mode, :nlink, :uid, :gid, :rdev,
                            :size, :blksize, :blocks, :atime, :mtime, :ctime)
        # Struct の生のクラスです。
        BasicStruct = superclass

        module Constants
            IFIFO  = 0x1000
            IFCHR  = 0x2000
            IFBLK  = 0x3000
            IFDIR  = 0x4000
            IFREG  = 0x8000
            IFMT   = 0xF000

            OX  = 0001
            OW  = 0002
            OR  = 0004
            GX  = 0010
            GW  = 0020
            GR  = 0040
            IEXEC  = 0100
            IWRITE = 0200
            IREAD  = 0400

            NORMAL = IFREG
            DIRECTORY = IFDIR
        end
        include Constants

        # 引数が一つで File::Stat であれば、それを元に初期化します。
        def initialize(*args)
            case args.length
            when 1
                if args[0].kind_of?(File::Stat)
                    st = args[0]
                    super(st.dev, st.ino, st.mode, st.nlink, st.uid, st.gid, st.rdev,
                          st.size, st.blksize, st.blocks, st.atime, st.mtime, st.ctime)
                else
                    super
                end
            else
                super
            end
        end

        # 情報がファイルを表すかを真偽値で返します。
        def file?
            (mode & IFMT) == IFREG ? true : false
        end

        # 情報がディレクトリを表すかを真偽値で返します。
        def directory?
            (mode & IFMT) == IFDIR ? true : false
        end

        # ファイルサイズと日時(作成日時、更新日時、参照日時の区別はありません)を指定するだけでファイル情報として初期化されます。
        def self.file(size, time)
            new(0, 0, IFREG | 0555, 1, Process.euid, Process.egid, 0, size.to_i, 0, 0, time, time, time)
        end

        # ファイルサイズと日時(作成日時、更新日時、参照日時の区別はありません)を指定するだけでディレクトリ情報として初期化されます。
        def self.directory(size, time)
            new(0, 0, IFDIR | 0755, 1, Process.euid, Process.egid, 0, size.to_i, 0, 0, time, time, time)
        end
    end
end


module Nazuna # :nodoc: all
    class << self
        require IMPLEMENT

        class ContextMap < Hash
            CONTEXT_MAX = 2000
            CONTEXT_BASE = 0x1000
            CONTEXT_RANGE = 0xffe000

            def regist(fd)
                raise "context is nil" unless fd
                raise OutOfContext if full?
                fdi = rand(CONTEXT_RANGE) + CONTEXT_BASE
                fdi += 1 while key?(fdi)
                store(fdi, fd)
                fdi
            end

            def full?
                length >= CONTEXT_MAX ? true : false
            end
        end
    end
end
