#!/usr/bin/env ruby
#vim: set fileencoding:utf-8

# - AUTHOR : dearblue <dearblue@users.sourceforge.jp>
# - PROJECT PAGE : http://sourceforge.jp/projects/rutsubo/
# - LICENSE : Public Domain
# - SUMMARY : 読み込み専用ミラーファイルシステムを実現する、Rubyによるユーザー空間ファイルシステムのサンプルコード
# - DESCRIPT : I/O要求処理にブロックを用いたサンプルコードで、読み込み専用ミラーファイルシステムを構築する。
#   大まかな処理の流れとしては:
#   1. <tt>require "nazuna"</tt> で、ライブラリの読み込む。
#   2. マウントポイントをブロック付きで (任意でオプション引数も) つけて Nazuna.mount を呼び出す。
#   3. ブロック内で IO 要求を内包したインスタンスが渡されるので、これを捌く!
#   4. IO 要求に対して値を返す。多くの場合正常処理であれば 0 である。
#   これでセカイであなただけのファイルシステムを構築できる。

# このサンプルコードの著作者であるdearblueは、このサンプルコードに関するすべての権利と義務を漏れなく放棄することを宣言する。
# (要約: このサンプルコードは Public Domain にするよ)


require "nazuna"
require "ioposrw"


if RUBY_PLATFORM =~ /mswin|mingw/
    DEFAULT_ROOTDIR =  "c:/windows"
    DEFAULT_MOUNTPOINT = "w:/"
else
    DEFAULT_ROOTDIR =  "/usr/local"
    DEFAULT_MOUNTPOINT = File.join(ENV["HOME"], "nazuna-test")
end

# 定数名から想像してね
ROOTDIR = (ARGV[0] || DEFAULT_ROOTDIR).encode("utf-8")
ROOTDIR.gsub!(File::ALT_SEPARATOR, File::SEPARATOR) if File::ALT_SEPARATOR
MOUNTPOINT = (ARGV[1] || DEFAULT_MOUNTPOINT).encode("utf-8")
MOUNTPOINT.gsub!(File::ALT_SEPARATOR, File::SEPARATOR) if File::ALT_SEPARATOR


# いきなり実行してしまう御仁のための端末への表示
$stderr.print <<-__END__.chomp
#$0 (PID: #$$):
    ROOTDIR: #{ROOTDIR}.
    MOUNTPOINT: #{MOUNTPOINT}.
** [Ctrl]+[C] to terminate filesystem **
sleeping 5 seconds for not flowing out this message
__END__

5.times do
    $stderr.print "."
    sleep 1
end

$stderr.puts


# 今回のサンプルコードは SIGINT 受け取るのが簡単な終了手段 (他にも umount あるいは dokanctl で unmount する)。
#
# 実際の利用時には Intterupt 例外を捕捉してその後の処理を行わせることも可能です。
#
# また、Nazuna.mount を呼び出す部分をスレッドで囲み Thread#kill を利用することも出来ます。
Signal.trap(:SIGINT) { exit 0 }


# このブロックは、常に必要となるわけではありません。デバッグ用途を想定しています。
#
# 今回は処理の流れを視覚化する為に記述してあります。
logger = proc do |ev, op, *args|
    marker = "#{Time.now.strftime("%H:%M:%S.%3N")} #{Thread.current} #{__FILE__}:#{__LINE__}"
    case ev
    when :enter
        if op == :write
            $stderr.printf("%s: %s - %s : %d octets.\n", marker, ev, op, args[0].bytesize)
        elsif op == :WriteFile
            $stderr.printf("%s: %s - %s : path=%s, offset=%s, size=%d octets.\n", marker, ev, op, args[0], args[1], args[2].bytesize)
        else
            $stderr.printf("%s: %s - %s : %s\n", marker, ev, op, args.inspect)
        end
    when :leave
        if op == :read || op == :ReadFile
            if args[0].kind_of?(String)
                $stderr.printf("%s: %s - %s : %d octets.\n", marker, ev, op, args[0].bytesize)
            else
                $stderr.printf("%s: %s - %s : %s\n", marker, ev, op, args.inspect)
            end
        else
            $stderr.printf("%s: %s - %s : %s\n", marker, ev, op, args[0].inspect)
        end
    else # ev == :break
        $stderr.printf("%s: %s - %s (%s)\n", marker, ev, op, $!)
    end
end


# Nazuna.mount は内部の Thread#join によって、その後の処理を停止させます。
#
# 異なるスレッドから Thread#kill して ensure 節を実行させることが可能です。
#
# +logger+ 引数は、IO 要求およびその返却した値がその都度渡されるようになります (デバッグ用途を想定)。
Nazuna.mount(MOUNTPOINT, nthread: 1, logger: logger) do |ioreq|
    # このブロック内は、Nazuna.mount したスレッドとは異なります。
    #
    # nazuna が作成したスレッドから呼ばれています。

    begin
        case ioreq
        when Nazuna::Ropen
            # open 要求!!
            # 今回は読み書きその他の指示子をまるっきり無視して、節操なく開きます!
            File.open(File.join(ROOTDIR, ioreq.path), "rb:binary")
        when Nazuna::Ropendir
            # opendir 要求!!
            Dir.open(File.join(ROOTDIR, ioreq.path), encoding: "utf-8")
        when Nazuna::Rstat
            # stat 要求!!
            File.stat(File.join(ROOTDIR, ioreq.path))
        when Nazuna::Rfstat
            File.stat(ioreq.file)
        when Nazuna::Rclose
            ioreq.file.close
        when Nazuna::Rread
            ioreq.file.pread(ioreq.offset, ioreq.size)
        when Nazuna::Rreaddir
            ioreq.dir.entries.each do |n|
                begin
                    ioreq.yield(n, File.stat(File.join(ioreq.dir, n)))
                rescue SystemCallError
                    nil
                end
            end
        else
            # 今回の実装で最低限必要とする IO 要求処理は上記七つです。
            Errno::ENOSYS
        end
    rescue SystemCallError
        # SystemCallError 例外を発生しがちなコードでして・・・
        $!.class
    end
end
