#include <boost/test/unit_test.hpp>
using namespace boost::unit_test_framework;

#include "graph.hpp"
#include "debug_macro.hpp"

using namespace std;

namespace {
BOOST_AUTO_TEST_SUITE(test_graph);

BOOST_AUTO_TEST_CASE(test_empty)
{
	Graph<int,int> g;
	
	BOOST_CHECK_EQUAL(g.empty(), true);
	BOOST_CHECK_EQUAL(g.size(), 0);
	BOOST_CHECK_EQUAL(g.edgeSize(), 0);
}
BOOST_AUTO_TEST_CASE(test_add_vertex)
{
	Graph<int,int> g;
	g.addVertex(10);
	BOOST_CHECK_EQUAL(g.size(), 1);
	BOOST_CHECK_EQUAL(g.empty(), false);
	BOOST_CHECK_EQUAL(g.edgeSize(), 0);
	g.addVertex(20);
	BOOST_CHECK_EQUAL(g.size(), 2);
	g.addVertex(10);
	BOOST_CHECK_EQUAL(g.size(), 2);
	BOOST_CHECK_EQUAL(g.hasVertex(10), true);
	BOOST_CHECK_EQUAL(g.hasVertex(20), true);
	BOOST_CHECK_EQUAL(g.hasVertex(100), false);
	
}
BOOST_AUTO_TEST_CASE(test_remove_vertex)
{
	Graph<int,int> g;
	g.addVertex(10);
	g.addVertex(20);
	BOOST_CHECK_EQUAL(g.size(), 2);
	BOOST_CHECK_EQUAL(g.hasVertex(10), true);
	g.removeVertex(100);
	BOOST_CHECK_EQUAL(g.size(), 2);
	BOOST_CHECK_EQUAL(g.hasVertex(10), true);
	g.removeVertex(10);
	BOOST_CHECK_EQUAL(g.size(), 1);
	BOOST_CHECK_EQUAL(g.hasVertex(10), false);
	g.removeVertex(10);
	BOOST_CHECK_EQUAL(g.size(), 1);
	BOOST_CHECK_EQUAL(g.hasVertex(10), false);
}
BOOST_AUTO_TEST_CASE(test_add_edge)
{
	Graph<int,int> g;
	
	g.addEdge(1,3);
	BOOST_CHECK_EQUAL(g.size(), 2);
	BOOST_CHECK_EQUAL(g.edgeSize(), 1);
	
	g.addEdge(2,3);
	BOOST_CHECK_EQUAL(g.size(), 3);
	BOOST_CHECK_EQUAL(g.edgeSize(), 2);
	BOOST_CHECK_EQUAL(g.hasEdge(2,3), true);
	BOOST_CHECK_EQUAL(g.hasEdge(1,3), true);
	BOOST_CHECK_EQUAL(g.hasEdge(3,1), false);
	BOOST_CHECK_EQUAL(g.hasEdge(5,1), false);
	
	g.addEdge(3,3);
	BOOST_CHECK_EQUAL(g.size(), 3);
	BOOST_CHECK_EQUAL(g.edgeSize(), 3);
	
	g.addEdge(4,5);
	BOOST_CHECK_EQUAL(g.size(), 5);
	BOOST_CHECK_EQUAL(g.edgeSize(), 4);
	
	g.addEdge(4,5);
	BOOST_CHECK_EQUAL(g.size(), 5);
	BOOST_CHECK_EQUAL(g.edgeSize(), 4);
	
	g.addEdge(5,4);
	BOOST_CHECK_EQUAL(g.size(), 5);
	BOOST_CHECK_EQUAL(g.edgeSize(), 5);
	
	g.addEdge(6,6);
	BOOST_CHECK_EQUAL(g.size(), 6);
	BOOST_CHECK_EQUAL(g.edgeSize(), 6);
	
}	
BOOST_AUTO_TEST_CASE(test_remove_edge)
{
	Graph<int,int> g;
	
	g.addEdge(1,3); g.addEdge(2,3); g.addEdge(3,3);
	g.addEdge(4,5); g.addEdge(5,4); g.addEdge(6,6);
	BOOST_CHECK_EQUAL(g.size(), 6);
	BOOST_CHECK_EQUAL(g.edgeSize(), 6);
	g.removeEdge(2,4);
	BOOST_CHECK_EQUAL(g.size(), 6);
	BOOST_CHECK_EQUAL(g.edgeSize(), 6);
	g.removeEdge(2,3);
	BOOST_CHECK_EQUAL(g.size(), 6);
	BOOST_CHECK_EQUAL(g.edgeSize(), 5);
	g.removeEdge(3,1);
	BOOST_CHECK_EQUAL(g.size(), 6);
	BOOST_CHECK_EQUAL(g.edgeSize(), 5);
	
}	
BOOST_AUTO_TEST_CASE(test_remove_vertex2)
{
	Graph<int,int> g;
	
	g.addEdge(1,3); g.addEdge(2,3); g.addEdge(3,3);
	g.addEdge(4,5); g.addEdge(5,4); g.addEdge(6,6);
	BOOST_CHECK_EQUAL(g.size(), 6);
	BOOST_CHECK_EQUAL(g.edgeSize(), 6);
	
	g.removeVertex(3);	
	BOOST_CHECK_EQUAL(g.size(), 5);
	BOOST_CHECK_EQUAL(g.edgeSize(), 3);
}	
BOOST_AUTO_TEST_CASE(test_successors)
{
	Graph<int,int> g;
	
	g.addEdge(1,2); g.addEdge(1,3); g.addEdge(1,4);
	BOOST_CHECK_EQUAL(g.successors(1).size(), 3);
	BOOST_CHECK_EQUAL(g.predecessors(1).size(), 0);
	BOOST_CHECK_EQUAL(g.successors(2).size(), 0);
	BOOST_CHECK_EQUAL(g.predecessors(2).size(), 1);
	BOOST_CHECK_EQUAL(g.edgeSize(), 3);
	
	g.addEdge(1,3);
	BOOST_CHECK_EQUAL(g.successors(1).size(), 3);
	BOOST_CHECK_EQUAL(g.predecessors(3).size(), 1);
	BOOST_CHECK_EQUAL(g.edgeSize(), 3);
	
	g.addEdge(1,1);
	BOOST_CHECK_EQUAL(g.edgeSize(), 4);
	BOOST_CHECK_EQUAL(g.successors(1).size(), 4);
	BOOST_CHECK_EQUAL(g.predecessors(1).size(), 1);
}	
BOOST_AUTO_TEST_CASE(test_begin)
{
	Graph<int,int> g;

	unsigned count=0;
	for (Graph<int,int>::iterator i=g.begin();
		 i != g.end(); ++i) {
		count++;
	}
	
	g.addEdge(1,4);
	vector<int> v;
	v.push_back(1);
	v.push_back(4);
	
	count=0;
	for (Graph<int,int>::iterator i=g.begin();
		 i != g.end(); ++i) {
		BOOST_CHECK_EQUAL(v[count], i->first);
		count++;
	}
	BOOST_CHECK_EQUAL(count, g.size());
	
	
	g.addEdge(1,3); g.addEdge(1,2); g.addEdge(2,4);
	g.addEdge(5,6);
	
	count=0;
	for (Graph<int,int>::iterator i=g.begin();
		 i != g.end(); ++i) {
		BOOST_CHECK_EQUAL(0, i->second);
		count++;
	}
	
	BOOST_CHECK_EQUAL(count, 6);
	BOOST_CHECK_EQUAL(count, g.size());
}
struct Sum
{
	int sum;
	Sum() { sum = 0; }
	void operator()(int i) { sum+=i; }
};
BOOST_AUTO_TEST_CASE(test_foreach)
{
	
	Graph<int,int> g;
	g.addEdge(1,2); g.addEdge(1,3); g.addEdge(1,4);
	g.addEdge(2,4); g.addEdge(5,6);

	Sum f;
	g.foreach(f);
	
	BOOST_CHECK_EQUAL(f.sum, 1+2+3+4+5+6);
}
struct SumEdge
{
	int sum;
	SumEdge() { sum = 0; }
	void operator()(int i, int j) { sum++; }
};
BOOST_AUTO_TEST_CASE(test_foreachEdge)
{
	
	Graph<int,int> g;
	g.addEdge(1,2); g.addEdge(1,3); g.addEdge(1,4);
	g.addEdge(2,4); g.addEdge(5,6);

	SumEdge f;
	g.foreachEdge(f);
	
	BOOST_CHECK_EQUAL(f.sum, 5);	
}	
BOOST_AUTO_TEST_CASE(test_successors_part2)
{
	
	Graph<int,bool> g;
	g.addEdge(1,2); g.addEdge(1,3); g.addEdge(1,4);
	g.addEdge(2,4); g.addEdge(5,6); g.addEdge(7,7);
	
	
	BOOST_CHECK_EQUAL(g.successors(1).size(),3);
	BOOST_CHECK_EQUAL(g.successors(2).size(),1);
	BOOST_CHECK_EQUAL(g.successors(3).size(),0);
	BOOST_CHECK_EQUAL(g.successors(7).size(),1);
	BOOST_CHECK_EQUAL(g.successors(8).size(),0);
	
	BOOST_CHECK_EQUAL(g.predecessors(1).size(),0);
	BOOST_CHECK_EQUAL(g.predecessors(2).size(),1);
	BOOST_CHECK_EQUAL(g.predecessors(3).size(),1);
	BOOST_CHECK_EQUAL(g.predecessors(7).size(),1);
	BOOST_CHECK_EQUAL(g.predecessors(8).size(),0);
}	
BOOST_AUTO_TEST_CASE(test_traversal)
{
	
	Graph<int,bool> g;
	int expect_pre[]={4,2,3,1,0};
	int expect_post[]={1,2,4,3,0};
	int expect_rpost[]={3,4,2,1,0};
	int expect_breadth[]={1,2,4,3,0};
	
	g.addEdge(1,2); g.addEdge(1,4); g.addEdge(1,3); 
	g.addEdge(2,4); g.addEdge(5,6); g.addEdge(7,7);
	int count=0;
	for (Graph<int,bool>::iterator i=g.pre(1); i!=g.end(); ++i) {
		BOOST_CHECK_EQUAL(i->first, expect_pre[count]);
		count++;
	}
	BOOST_CHECK_EQUAL(count, 4);
	count=0;
	for (Graph<int,bool>::iterator i=g.post(1); i!=g.end(); ++i) {
		BOOST_CHECK_EQUAL(i->first, expect_post[count]);
		count++;
	}
	BOOST_CHECK_EQUAL(count, 4);
	count=0;
	for (Graph<int,bool>::iterator i=g.rpost(1); i!=g.end(); ++i) {
		BOOST_CHECK_EQUAL(i->first, expect_rpost[count]);
		count++;
	}
	BOOST_CHECK_EQUAL(count, 4);
	count=0;
	for (Graph<int,bool>::iterator i=g.breadthFirst(1); i!=g.end(); ++i) {
		BOOST_CHECK_EQUAL(i->first, expect_breadth[count]);
		count++;
	}
	BOOST_CHECK_EQUAL(count, 4);
}

BOOST_AUTO_TEST_CASE(test_copy)
{
	
	Graph<int,bool> g;
	int expect_pre[]={4,2,3,1,0};
    
	g.addEdge(1,2); g.addEdge(1,4); g.addEdge(1,3); 
	g.addEdge(2,4); g.addEdge(5,6); g.addEdge(7,7);
    
    Graph<int,bool> g_dash(g);
	int count=0;
    Graph<int,bool>::iterator i;
	for (i=g_dash.pre(1); i!=g_dash.end(); ++i) {
		BOOST_CHECK_EQUAL(i->first, expect_pre[count]);
		count++;
	}
	BOOST_CHECK_EQUAL(count, 4);
}
BOOST_AUTO_TEST_CASE(test_label)
{
	
	Graph<int,int> g;
    
	g.addLabeledEdge(1,2, 4);
    g.addLabeledEdge(1,4, 1);
    g.addLabeledEdge(1,3, 0); 
    
	BOOST_CHECK_EQUAL(g.label(1,2), 4);
	BOOST_CHECK_EQUAL(g.label(2,1), 0); // default value.
	BOOST_CHECK_EQUAL(g.label(1,4), 1);
	BOOST_CHECK_EQUAL(g.label(1,3), 0);
}

BOOST_AUTO_TEST_CASE(test_remove_edge2)
{
	Graph<int,int> g;
	
	g.addEdge(1,2); g.addEdge(1,3); g.addEdge(1,4);
	BOOST_CHECK_EQUAL(g.successors(1).size(), 3);
	BOOST_CHECK_EQUAL(g.predecessors(2).size(), 1);
	BOOST_CHECK_EQUAL(g.hasEdge(1,2), true);
    
	g.removeEdge(1,2); 
	BOOST_CHECK_EQUAL(g.successors(1).size(), 2);
	BOOST_CHECK_EQUAL(g.predecessors(2).size(), 0);
	BOOST_CHECK_EQUAL(g.hasEdge(1,2), false);
    
	g.removeVertex(3); 
	BOOST_CHECK_EQUAL(g.successors(1).size(), 1);
	BOOST_CHECK_EQUAL(g.hasEdge(1,3), false);
}	

BOOST_AUTO_TEST_SUITE_END();
}


