#include "reference.hpp"


std::string ReferenceHelper::name(ReferenceType type)
{
    switch (type) {
    case reference::WRITE: return "byte";
    case reference::IDX_WRITE: return "[]byte";
    case reference::PRE_IND_WRITE: return "[]^byte"; // (q,x)
    case reference::POST_IND_WRITE: return "^[]byte"; // (q),y
        
    case reference::READ: return "const byte";
    case reference::IDX_READ: return "const []byte";
    case reference::PRE_IND_READ: return "const []^byte"; // (q,x)
    case reference::POST_IND_READ: return "const ^[]byte"; // (q),y
    case reference::JUMP: return "label";
    case reference::IND_JUMP: return "^label";
    case reference::JUMP_TABLE: return "[]^label";
    case reference::CALL: return "sub";
    case reference::IND_CALL: return "^sub";
    case reference::RETURN: return "return";
    case reference::STK_JUMP: return "stack jump";
    case reference::BREAK: return "break";
    case reference::JUNCTION: return "junction";
    case reference::BRANCH: return "branch";
    default: return "???";
    }
}

bool ReferenceHelper::isDataAccess(ReferenceType type)
{
    switch (type){
    case reference::JUMP: case reference::JUNCTION:
    case reference::CALL: case reference::RETURN: 
    case reference::STK_JUMP:
    case reference::BREAK: case reference::BRANCH:
        return false;
    case reference::WRITE: case reference::IDX_WRITE:
    case reference::PRE_IND_WRITE:  case reference::POST_IND_WRITE: 
        
    case reference::READ: case reference::IDX_READ: 
    case reference::PRE_IND_READ: case reference::POST_IND_READ:
        
    case reference::JUMP_TABLE: 
    case reference::IND_JUMP: case reference::IND_CALL: 
        return true;
        
    default:
        return false;
    }
}
bool
ReferenceHelper::isControlChange(ReferenceType type)
{
    switch (type){
    case reference::JUMP: case reference::JUNCTION:
    case reference::CALL: case reference::RETURN: 
    case reference::STK_JUMP: case reference::BREAK:  
    case reference::BRANCH:
        return true;
    case reference::WRITE: case reference::IDX_WRITE:
    case reference::PRE_IND_WRITE:  case reference::POST_IND_WRITE: 
        
    case reference::READ: case reference::IDX_READ: 
    case reference::PRE_IND_READ: case reference::POST_IND_READ:
        
    case reference::JUMP_TABLE: 
    case reference::IND_JUMP: case reference::IND_CALL: 
        return false;
        
    default:
        return false;
    }
}

ReferenceType
ReferenceHelper::mapWithAddressingMode(OpcodeType op_type,
                                           AddressingMode mode)
{
    ReferenceType type = ReferenceHelper::map(op_type);
    switch (type) {
    case reference::READ:
        switch (mode) {
        case opcode::ZERO_X: case opcode::ABS_X:
        case opcode::ZERO_Y: case opcode::ABS_Y:
            return reference::IDX_READ;
        case opcode::PRE_IND:
            return reference::PRE_IND_READ;
        case opcode::POST_IND:
            return reference::POST_IND_READ;
        case opcode::ZERO: case opcode::ABS:
        default:
            return reference::READ;
        }
    case reference::WRITE:
        switch (mode) {
        case opcode::ZERO_X: case opcode::ABS_X:
        case opcode::ZERO_Y: case opcode::ABS_Y:
            return reference::IDX_WRITE;
        case opcode::PRE_IND:
            return reference::PRE_IND_WRITE;
        case opcode::POST_IND:
            return reference::POST_IND_WRITE;
        case opcode::ZERO: case opcode::ABS:
        default:
            return reference::WRITE;
        }
    default:
        return type;
    }
}

ReferenceType ReferenceHelper::map(OpcodeType op_type)
{
	switch (op_type) {
	case opcode::LDA: case opcode::LDX: case opcode::LDY:
	case opcode::ADC: case opcode::AND:
	case opcode::SBC: case opcode::EOR: case opcode::ORA:
	case opcode::BIT:
    case opcode::CMP: case opcode::CPX: case opcode::CPY:
		return reference::READ;
		
	case opcode::STA: case opcode::STX: case opcode::STY:
	case opcode::INC: case opcode::INX: case opcode::INY:
	case opcode::DEC: case opcode::DEX: case opcode::DEY:
	case opcode::ASL: case opcode::LSR:
	case opcode::ROL: case opcode::ROR:
		return reference::WRITE;
		
	case opcode::JMP:
		return reference::JUMP;
		
	case opcode::JSR:
		return reference::CALL;
        
	case opcode::BCC: case opcode::BCS: case opcode::BEQ:
	case opcode::BMI: case opcode::BNE: case opcode::BPL:
	case opcode::BVC: case opcode::BVS:
		return reference::BRANCH;
		
	case opcode::RTS: case opcode::RTI:
		return reference::RETURN;
		
	case opcode::BRK:
		return reference::BREAK;

	case opcode::NOP:
		
	case opcode::PHA: case opcode::PHP: case opcode::PLA: case opcode::PLP:
	case opcode::CLC: case opcode::CLD: case opcode::CLI: case opcode::CLV:
	case opcode::SEC: case opcode::SED: case opcode::SEI:
	case opcode::TAX: case opcode::TAY: case opcode::TSX:
	case opcode::TXA: case opcode::TXS: case opcode::TYA:
	default:
		return reference::REFERENCE_UND;
	}	
}




std::set<MemoryID>
ReferenceSelector::select(const ReferenceGraph& graph,
                          ReferenceType condition)
{
        
    std::set<MemoryID> result;
    ReferenceGraph::iterator i;
    for (i=graph.begin(); i!=graph.end(); ++i) {
        MemoryID id = i->first;
        const std::vector<MemoryID>& pred
            =graph.predecessors(id);
        for (unsigned j=0; j<pred.size(); ++j) {
            MemoryID referer_id = pred[j];
            ReferenceType type = graph.label(referer_id, id);
            if (type == condition) {
                result.insert(id);
            }
        }
    }
    return result;
}
std::set<MemoryID>
ReferenceSelector::entries(const ReferenceGraph& graph)
{
    std::set<MemoryID> result;
    ReferenceGraph::iterator i;
    for (i=graph.begin(); i!=graph.end(); ++i) {
        MemoryID id = i->first;
        const std::vector<MemoryID>& pred
            =graph.predecessors(id);
        for (unsigned j=0; j<pred.size(); ++j) {
            MemoryID referer_id = pred[j];
            ReferenceType type = graph.label(referer_id, id);
            if (type == reference::CALL
                || type == reference::JUNCTION) {
                result.insert(id);
            }
        }
    }
    return result;
}
std::set<MemoryID>
ReferenceSelector::topLevelEntries(const ReferenceGraph& graph)
{
    std::set<MemoryID> result;
    MemoryID id = memory_id::ENTRY;
        
    const std::vector<MemoryID>& succ =graph.successors(id);

    for (unsigned j=0; j<succ.size(); ++j) {
        MemoryID dest_id = succ[j];
        ReferenceType type = graph.label(id, dest_id);
            
        if (type == reference::CALL
            || type == reference::JUNCTION) {
            result.insert(dest_id);
        }
    }
    return result;
}
std::set<MemoryID>
ReferenceSelector::anyLabel(const ReferenceGraph& graph)
{
    std::set<MemoryID> result;
    ReferenceGraph::iterator i;
    for (i=graph.begin(); i!=graph.end(); ++i) {
        MemoryID id = i->first;
        if (!graph.predecessors(id).empty()){
            result.insert(id);
        }
    }
    return result;
}
std::set<MemoryID>
ReferenceSelector::codeLabel(const ReferenceGraph& graph)
{
    std::set<MemoryID> result;
    ReferenceGraph::iterator i;
    for (i=graph.begin(); i!=graph.end(); ++i) {
        MemoryID id = i->first;
        const std::vector<MemoryID>& pred
            =graph.predecessors(id);
        for (unsigned j=0; j<pred.size(); ++j) {
            MemoryID referer_id = pred[j];
            ReferenceType type = graph.label(referer_id, id);
            if (ReferenceHelper::isControlChange(type)) {
                result.insert(id);
            }
        }
    }
    return result;
}
std::set<MemoryID>
ReferenceSelector::dataLabel(const ReferenceGraph& graph)
{
    std::set<MemoryID> result;
    ReferenceGraph::iterator i;
    for (i=graph.begin(); i!=graph.end(); ++i) {
        MemoryID id = i->first;
        const std::vector<MemoryID>& pred
            =graph.predecessors(id);
        for (unsigned j=0; j<pred.size(); ++j) {
            MemoryID referer_id = pred[j];
            ReferenceType type = graph.label(referer_id, id);
            if (ReferenceHelper::isDataAccess(type)) {
                result.insert(id);
            }
        }
    }
    return result;
}


static void displayOneLine(std::ostream& os,
                           unsigned type, unsigned mode)
{
    switch (mode) {
    case opcode::ZERO: os << "$"; break;
    case opcode::ZERO_X: os << "[x]$"; break;
    case opcode::ZERO_Y: os << "[y]$"; break;
    case opcode::ABS: break;
    case opcode::ABS_X: os << "[]"; break;
    case opcode::ABS_Y: os << "[y]"; break;
    case opcode::IND: os << "^"; break;
    case opcode::PRE_IND: os << "^[]"; break;
    case opcode::POST_IND: os << "[]^"; break;
    default: break;
    }
    switch (type) {
    case reference::READ: os << "const"; break;
    case reference::WRITE: os << "var"; break;
    case reference::JUMP: os << "label"; break;
    case reference::CALL: os << "sub"; break;
    case reference::BREAK: os << "break"; break;
    case reference::BRANCH: os << "branch"; break;
    case reference::JUNCTION: os << "junc"; break;
    case reference::UNCALL: os << "uncalled"; break;
    case reference::JUMP_TABLE: os << "[]^sub"; break;
    case reference::SUSPECIOUS_CALL: os << "suspecious"; break;
    default: os << "? "; break;
    }
}

ReferenceCode::ReferenceCode(const Reference& reference)
{
    init(reference.statistics());
}

void ReferenceCode::display(std::ostream& os)const
{
    unsigned used_type = 0, variation_num = variation();
    bool is_union = variation_num > 1;
    if (is_union) {
        os << "union {";
    }
    for (unsigned i=0; i<reference::REFERENCE_NUM; ++i) {
        for (unsigned j=0; j<opcode::ADDRESSING_NUM; ++j) {
            if (m_n_times[i][j]) {
                displayOneLine(os,i,j);
                if (is_union && used_type < variation_num - 1) {
                    os << ", ";
                }
                used_type++;
            }
        }
    }
    if (is_union) {
        os << "}";
    }
}
void ReferenceCode::init(const Reference::Statistic& n_times)
{
    m_n_times.resize(reference::REFERENCE_NUM);
		
    for (unsigned i=0; i<reference::REFERENCE_NUM; ++i) {
        m_n_times[i] = n_times[i];
    }
    // ZERO <= ABS, _X <= _Y
    for (unsigned i=0; i<reference::REFERENCE_NUM; ++i) {
        m_n_times[i][opcode::ABS_X]
            = m_n_times[i][opcode::ABS_X]
            | m_n_times[i][opcode::ABS_Y];
        m_n_times[i][opcode::ABS_Y] = false;
        m_n_times[i][opcode::ZERO_X]
            = m_n_times[i][opcode::ZERO_X]
            | m_n_times[i][opcode::ZERO_Y];
        m_n_times[i][opcode::ZERO_Y] = false;

        if (m_n_times[i][opcode::ZERO] | m_n_times[i][opcode::ZERO_X]) {
            m_n_times[i][opcode::ZERO]
                = m_n_times[i][opcode::ZERO]
                | m_n_times[i][opcode::ABS];
            m_n_times[i][opcode::ABS] = false;
        }
    }
    // WRITE(>0) <= READ
    for (unsigned j=0; j<opcode::ADDRESSING_NUM; ++j) {
			
        if (m_n_times[reference::WRITE][j]) {
            m_n_times[reference::WRITE][j]
                = m_n_times[reference::WRITE][j]
                | m_n_times[reference::READ][j];
            m_n_times[reference::READ][j] = false;
        }

        if (m_n_times[reference::JUNCTION][j]) {
            m_n_times[reference::JUNCTION][j]
                = m_n_times[reference::JUNCTION][j]
                | m_n_times[reference::SUSPECIOUS_CALL][j];
            m_n_times[reference::SUSPECIOUS_CALL][j] = false;
        }
        if (m_n_times[reference::CALL][j]) {
            m_n_times[reference::CALL][j]
                = m_n_times[reference::CALL][j]
                | m_n_times[reference::SUSPECIOUS_CALL][j];
            m_n_times[reference::SUSPECIOUS_CALL][j] = false;
        }
        if (m_n_times[reference::UNCALL][j]) {
            m_n_times[reference::UNCALL][j]
                = m_n_times[reference::UNCALL][j]
                | m_n_times[reference::SUSPECIOUS_CALL][j];
            m_n_times[reference::SUSPECIOUS_CALL][j] = false;
				
            if (m_n_times[reference::READ][opcode::ABS]) {
                m_n_times[reference::READ][j]
                    = m_n_times[reference::READ][j]
                    | m_n_times[reference::UNCALL][j]; 
                m_n_times[reference::UNCALL][j] = false; 
            }
        }
    }
}
unsigned ReferenceCode::variation() const
{
    unsigned used_type = 0;
    for (unsigned i=0; i<reference::REFERENCE_NUM; ++i) {
        for (unsigned j=0; j<opcode::ADDRESSING_NUM; ++j) {
            if (m_n_times[i][j]) {
                used_type++;
            }
        }
    }
    return used_type;
}

