#pragma once
#include "basic_type.hpp"
#include <vector>
#include <iostream>
#include "utils/hex_form.hpp"
#include "instruction.hpp"
#include "utils/string_aligner.hpp"
#include "ines_file.hpp"
#include "code_server.hpp"

/// Under construction. So don't split now.
class Disassembler
{
	std::ostream& m_os;
    CodeServer m_coder;
	enum Constants {
		LABEL_WIDTH = 8, MNEMONIC_WIDTH = 24,
	};
public:
	Disassembler(std::ostream& out)
            :m_os(out) {}
	void writeINesHeader(const INesHeader& header)
	{
		std::string indent = col1(" ");
		
		m_os << indent << col2(".inesprg " + num(header.prg()))
             << std::endl
			 << indent << col2(".ineschr " + num(header.chr()))
             << std::endl
			 << indent << col2(".inesmir " + num(header.mir()))
             << std::endl
			 << indent << col2(".inesmap " + num(header.map())) 
			 << "; " << m_coder.mapper(header.map())
             << std::endl;
	}
	void writeInterruptVector(const std::vector<Word>& interrupts)
	{
		assert(interrupts.size() == 3);
		
		std::string indent = col1(" ");
		m_os << indent << col2(".dw   " + hex(interrupts[0]))
			 << "; NMI interrupt" << std::endl;
		m_os << indent << col2(".dw   " + hex(interrupts[1]))
			 << "; RESET interrupt" << std::endl;
		m_os << indent << col2(".dw   " + hex(interrupts[2]))
			 << "; IRQ interrupt" << std::endl;
	}
	void writeINesFile(const INesFile& nes)
	{
		m_os << "; iNes header infomation" << std::endl;
		writeINesHeader(nes.header());
		
		if (!nes.rest().empty()) {
			m_os << "; trainer/FEE: $" <<  hex(nes.rest().size())
				 << " bytes remained" << std::endl;
		}
		std::string indent = col1(" ");
	}
	void writeBankOrigin(unsigned bank_num, Word origin)
	{
		m_os << "; PRG-ROM Bank " << bank_num << std::endl
			 << col1(" ") << ".bank " << bank_num << std::endl
			 << col1(" ") << ".org  $"
			 << hex(origin) << std::endl;
	}
	void writeMnemonic(const Instruction& instr)
	{
		m_os << col2(std::string(instr.mnemonic()) + "   "
					 + m_coder.operand(
                             instr.operand(), instr.mode()));
	}
	void writeAddressLabel(Word addr)
	{
		m_os << col1(HexForm(addr).to_s() + ":");
	}
	void writeMemoryIDLabel(MemoryID id, Word bank_origin)
	{
        Word addr = (id & 0x1FFF) + bank_origin;
		m_os << col1(HexForm(addr).to_s() + ":");
	}
	/** To literal translate the byte vector.*/
	void writeBankRegeon(Word origin,
                         const ByteSequence& bank,
                         unsigned begin_index,
                         unsigned end_index)
	{
        assert(end_index < bank.size());
        
		for (unsigned i=begin_index; i < end_index; ) {
			Word addr = i+origin;
			Instruction instr(addr, bank, i);
			writeAddressLabel(addr);
			writeMnemonic(instr);
            m_os << "; ";
			writeByteCode(instr.bytes());
			i+=instr.bytelength();
			m_os << std::endl;
		}
	}
	void writeBankLiterally(Word origin, const ByteSequence& bank)
	{
        writeBankRegeon(origin, bank,0,bank.size());
	}
    void writeByteCode(const ByteSequence& bytes)
    {
        for (unsigned i=0; i<bytes.size(); i++) {
            m_os << byteHex(bytes[i]) + " ";
        }
    }
	void dump(std::ostream& os, const std::vector<Byte>* bytes)
	{
		std::stringstream ss;
		for (unsigned i=0; i<bytes->size(); ++i) {
			ss << (*bytes)[i];
		}
		ss >> BinaryDump(16, BinaryDump::WRITE_HEX | BinaryDump::WRITE_CHAR)
		   >> os;
	}
private:
	std::string col1(std::string str)
	{
		StringAligner align(LABEL_WIDTH, StringAligner::LEFT);
		return align(str);
	}
	std::string col2(std::string str)
	{
		StringAligner align(MNEMONIC_WIDTH, StringAligner::LEFT);
		return align(str);
	}
	std::string byteHex(unsigned val) 
	{
		HexForm h(val);
		return h.width(2).to_s();
	}
	std::string hex(unsigned val)
	{
		HexForm h(val);
		return h.to_s();
	}
	std::string num(unsigned val)
	{
		DecimalFormat n(val);
		return n.to_s();
	}
};
