#pragma once
#include "basic_type.hpp"
#include <vector>
#include <string>
#include <iostream>
#include <fstream>
#include <stdexcept>
#include <sstream>

#include "ines_header.hpp"

#include "utils/number_format.hpp"
#include "utils/stream_measure.hpp"
#include "utils/file_path.hpp"
#include "utils/binary_dump.hpp"
#include "debug_macro.hpp"

#include "nes_memory_map.hpp"

class InvalidNesFormatException : public std::runtime_error
{
public:
	InvalidNesFormatException(std::string message)
			:std::runtime_error(
                    "Error: invalid nes format: " +message){}
    
};

/** load file and separate banks.*/
class INesFile
{
private:
	INesHeader m_header;
	std::vector<PByteSequence> m_prg;
	std::vector<PByteSequence> m_chr;
	std::vector<Byte> m_rest;
	unsigned m_size;
public:
	INesFile(){};
	const std::vector<PByteSequence> prgBanks()const{ return m_prg; }
	const std::vector<PByteSequence> chrBanks()const{ return m_chr; }
	const std::vector<Byte> rest()const{ return m_rest; }
	const INesHeader& header()const { return m_header; }
	
	unsigned map()const { return m_header.map(); }
	unsigned mir()const { return m_header.mir(); }
	
	unsigned fileSize()const {
		return INesHeader::HEADER_LENGTH
				+ m_header.prg() * nes::PRG_PAGE_SIZE
				+ m_header.chr() * nes::CHR_PAGE_SIZE;
	}
    unsigned prgSize()
    {
        return m_header.prg() * nes::PRG_PAGE_SIZE;
    }
	void load(FilePath file_name)
	{
		std::ifstream fin(file_name.c_str(),
						  std::ifstream::in | std::ifstream::binary);
		read(fin);
	}
	void read(std::istream& is)
	{
		StreamMeasure measure(is);
		m_size = measure.size();
		
		m_header.read(is);
		if (! m_header.isValid()) {
			throw InvalidNesFormatException("iNes header missing");
		} else if (m_size < fileSize()) {
			throw InvalidNesFormatException("file imcompete");
		}
		
		for (unsigned i=0; i < m_header.prg() * 2; ++i) {
			// 2 times prg page number; 1 page is 2 banks. 
			PByteSequence p_bytes(new ByteSequence(nes::BANK_SIZE));
			m_prg.push_back(p_bytes);
			for (unsigned j=0; j< p_bytes->size(); ++j) {
				(*p_bytes)[j] = is.get();
			}
		}
		for (unsigned i=0; i<m_header.chr();++i) {
			PByteSequence p_bytes(new ByteSequence(nes::BANK_SIZE));
			m_chr.push_back(p_bytes);
			for (unsigned j=0; j< p_bytes->size(); ++j) {
				(*p_bytes)[j] = is.get();
			}
		}
		for (char ch; is.get(ch); ) {
			m_rest.push_back(Byte(ch));
		}
		ensure(m_prg.size() == m_header.prg()*2);
		ensure(m_chr.size() == m_header.chr());
	}
};
