
#include "front_end.hpp"

#include "reference.hpp"
#include "ines_file.hpp"
#include "instruction.hpp"
#include <algorithm>

/** The reference table is corresponding to each PRG-ROM bank.
 * CONCEPT: scan all jump instructions in banks.
 * INPUT:   INesFile, threshold
 * OUTPUT:  Word vector.
 */
class OriginScannerImpl
{
	enum BankRefType{
		REF_JUMP=0, REF_CALL, REF_READ,
		REF_TYPE_MAX=3,
	};
	/** threshold is lower, it will be confused by irregular jumps.*/
	unsigned m_threshold;
	std::vector<Word> m_origins;
    
	// tables[bank_num][page_idx][type] <- count
	typedef std::vector<std::vector<unsigned> > AccessCountTable;
	std::vector<AccessCountTable> m_tables;
public:
	// if threshold is higher, the bank with few jump code is assumed data.
	OriginScannerImpl(unsigned threshold) :m_threshold(threshold) { }
    
	// origin of each banks. it get nes for future use of mapper.
    std::vector<Word> scan(const INesFile& nes)
    {
        const std::vector<PByteSequence>& banks = nes.prgBanks();
        
        initTables(banks.size());
        for (unsigned i=0; i<banks.size(); ++i) {
            scanBank(m_tables[i], *(banks[i]));
        }
        assumeOriginByJumpDestination();
        assumeOriginByDataAccess();
        return m_origins;
    }
    
private:
    void initTables(unsigned banks_size)
    {
        m_origins.resize(banks_size);
        
        m_tables.resize(banks_size);
        for (unsigned i=0; i<banks_size; ++i) {
            AccessCountTable& sub_table = m_tables[i];
            
            sub_table.resize(nes::PAGE_NUM);
            for (unsigned i=0; i<nes::PAGE_NUM; ++i) {
                sub_table[i].resize(REF_TYPE_MAX);
                std::fill(sub_table[i].begin(),
                          sub_table[i].end(), 0);
            }
        }
    }
	// [$8000[JUMP, CALL, READ], $A000[],...
    void scanBank(AccessCountTable& sub_table,
                  const ByteSequence& bank)
    {
        for (unsigned i=0; i < bank.size(); ) {
            Instruction instr(0, bank, i);
            if (InstructionHelper::isRomAccess(instr)) {
                Word addr = instr.operand();
                unsigned page_idx
                    = NesMemoryMapHelper::pageIndexByAddress(addr);
                
                unsigned type=0;
                switch (ReferenceHelper::map(instr.type())) {
                case reference::JUMP: type=REF_JUMP; break;
                case reference::CALL: type=REF_CALL; break;
                case reference::READ: type=REF_READ; break;
                default: type=2; break;
                }
                if (page_idx < nes::PAGE_NUM) {
                    sub_table[page_idx][type] += 1;
                }
            }
            i+=instr.bytelength();
        }
    }

	// if jump addressing memory area may be the banks origin. 
    void assumeOriginByJumpDestination()
    {
        for (unsigned i=0; i<m_origins.size(); ++i) {
            m_origins[i]=0;
            unsigned max_jump_count = m_threshold;
            for (unsigned j=0; j<nes::PAGE_NUM; ++j) {
                unsigned count = m_tables[i][j][REF_JUMP]; 
                if (count > max_jump_count) {
                    max_jump_count = count;
                    m_origins[i]
                        = NesMemoryMapHelper::pageOrigin(j);
                }
            }
        }
    }
	// if jump is not so many, it may be data section.
    void assumeOriginByDataAccess()
    {
        // possibility count; if code[n] is higher, page n is code
        Word code[nes::PAGE_NUM]; 
        Word data[nes::PAGE_NUM];
        
        for (unsigned i=0; i<nes::PAGE_NUM; ++i){
            code[i]=data[i]=0;
        }
		
        for (unsigned i=0; i<m_origins.size(); ++i) {
            if (m_origins[i] == 0) {
                continue;
            }
            for (unsigned j=0; j<nes::PAGE_NUM; ++j) {
                code[j] += m_tables[i][j][REF_CALL]
                    + m_tables[i][j][REF_JUMP];
                data[j] += m_tables[i][j][REF_READ];
            }
        }
        Word data_bank_origin[2];
        data_bank_origin[0] = (data[0]+code[2]-code[0]-data[2] >= 0)?
			0x8000:0xC000;
        data_bank_origin[1] = (data[1]+code[3]-code[1]-data[3] >= 0)?
			0xA000:0xE000;
		
        for (unsigned i=0; i<m_origins.size(); ++i) {
            if (m_origins[i]==0) {
                m_origins[i]=data_bank_origin[i%2];
            }
        }
    }
};

std::vector<Word>
OriginScanner::scanOrigin(
        const INesFile& cartridge, unsigned threshold)
{
    OriginScannerImpl impl(threshold);
    return impl.scan(cartridge);
}

