package jp.sourceforge.tamanegisoul.sa.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Calendar;
import java.util.HashSet;
import java.util.Set;


import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import android.accounts.Account;
import android.accounts.AccountManager;
import android.accounts.AuthenticatorException;
import android.accounts.OperationCanceledException;
import android.content.ContentValues;
import android.content.Context;
import android.database.sqlite.SQLiteDatabase;
import android.util.Xml;

/**
 * アカウント管理クラス。
 */
public class AppAccountManager {

    /** アカウント、開始日付、終了日付をパラメータとして予定を取得するURL */
    private static final String URL_CALENDAR = "https://www.google.com/calendar/feeds/%s/private/full?fields=entry(title,gd:when)&orderby=starttime&start-min=%tFT00:00:00%%2b09:00&start-max=%tFT00:00:00%%2b09:00&q=alarm";

    private Context mContext;
    private AccountManager mAccountManager;

    public AppAccountManager(Context context) {
        super();
        mContext = context;
        mAccountManager = AccountManager.get(mContext);
    }

    public String[] getAccountNames() {
        Account[] accounts = mAccountManager.getAccountsByType("com.google");
        String[] accountNames = new String[accounts.length];
        for (int i = 0; i < accounts.length; i++) {
            accountNames[i] = accounts[i].name;
        }
        return accountNames;
    }

    public void refreshCalendar() {
        String account = PreferenceUtils.getString(mContext, PreferenceUtils.KEY_ACCOUNT);
        if (account == null) {
            LogUtil.w("No account is selected.");
        } else {
            String token = getAuthToken(account);
            if (token == null) {
                LogUtil.e("Getting AuthToken failed.");
            } else {
                Calendar startDay = Calendar.getInstance();
                Calendar endDay = (Calendar) startDay.clone();
                endDay.add(Calendar.DATE, 7);
                String url = String.format(URL_CALENDAR, new Object[] { account, startDay, endDay });
                LogUtil.d("Calendar query url->%s", url);

                BufferedReader reader = null;
                SQLiteDatabase db = null;
                try {
                    HttpGet request = new HttpGet(url);
                    request.setHeader("Authorization", "GoogleLogin auth=" + token);
                    HttpClient httpclient = new DefaultHttpClient();
                    HttpResponse response = (HttpResponse) httpclient.execute(request);
                    reader = new BufferedReader(new InputStreamReader(new BufferedHttpEntity(response.getEntity()).getContent()));
                    XmlPullParser parser = Xml.newPullParser();
                    parser.setInput(reader);

                    // カレンダーをすべて削除
                    db = new DBHelper(mContext).getWritableDatabase();
                    db.beginTransaction();
                    db.delete(DBHelper.T_CALENDAR, null, null);
                    LogUtil.d("Calendar deleted.");
                    // XMLパース
                    Set<String[]> addDates = new HashSet<String[]>(); // +alarmの日時リスト
                    Set<String[]> cancelDates = new HashSet<String[]>(); // -alarmの日時リスト
                    String currentElement = "";
                    String[] currentValue = null;
                    for (int eventType = parser.getEventType(); eventType != XmlPullParser.END_DOCUMENT; eventType = parser.next()) {
                        switch (eventType) {
                        case XmlPullParser.START_TAG:
                            currentElement = parser.getName();
                            if (currentElement.equals("when")) {
                                currentValue[1] = parser.getAttributeValue(null, "startTime");
                                if(currentValue[0].contains("-alarm")){
                                    cancelDates.add(currentValue);
                                }else if(currentValue[0].contains("+alarm")){
                                    addDates.add(currentValue);
                                }
                            }
                            break;
                        case XmlPullParser.TEXT:
                            if (currentElement.equals("title")) {
                                currentValue = new String[]{parser.getText(), null};
                            }
                            break;
                        }
                    }
                    // 先に+alarmをinsertしてから-alarmの日のものを削除（同日の場合-alarm優先）
                    ContentValues values = new ContentValues();
                    for(String[] addDate : addDates) {
                        values.clear();
                        values.put(DBHelper.C_CALENDAR_DATE, addDate[1].substring(0, 10));
                        values.put(DBHelper.C_CALENDAR_TIME, addDate[1].substring(11, 16));
                        values.put(DBHelper.C_CALENDAR_IS_CANCEL, false);
                        values.put(DBHelper.C_CALENDAR_NAME, addDate[0]);
                        db.insert(DBHelper.T_CALENDAR, null, values);
                        LogUtil.d("Calendar inserted. date->%s, time->%s, is_cancel->%s", values.get(DBHelper.C_CALENDAR_DATE), values
                                .get(DBHelper.C_CALENDAR_TIME), values.get(DBHelper.C_CALENDAR_IS_CANCEL));
                    }
                    for(String[] cancelDate : cancelDates) {
                        values.clear();
                        values.put(DBHelper.C_CALENDAR_DATE, cancelDate[1].substring(0, 10));
                        values.put(DBHelper.C_CALENDAR_IS_CANCEL, true);
                        values.put(DBHelper.C_CALENDAR_NAME, cancelDate[0]);
                        db.insert(DBHelper.T_CALENDAR, null, values);
                        LogUtil.d("Calendar inserted. date->%s, time->%s, is_cancel->%s", values.get(DBHelper.C_CALENDAR_DATE), values
                                .get(DBHelper.C_CALENDAR_TIME), values.get(DBHelper.C_CALENDAR_IS_CANCEL));
                        db.delete(DBHelper.T_CALENDAR, DBHelper.C_CALENDAR_DATE + "=? and " + DBHelper.C_CALENDAR_IS_CANCEL + "=0", new String[]{values.getAsString(DBHelper.C_CALENDAR_DATE)});
                        DBHelper.dump(db.query(DBHelper.T_CALENDAR, null, null, null, null, null, null));
                    }
                    
                    db.setTransactionSuccessful();
                } catch (ClientProtocolException e) {
                    e.printStackTrace();
                } catch (IOException e) {
                    e.printStackTrace();
                } catch (XmlPullParserException e) {
                    e.printStackTrace();
                } finally {
                    if (db != null)
                        db.endTransaction();
                    if (reader != null) {
                        try {
                            reader.close();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    }
                }
            }
        }
    }

    private String getAuthToken(String account) {
        String token = null;
        try {
            token = mAccountManager.blockingGetAuthToken(getAccount(account), "cl", false);
        } catch (OperationCanceledException e) {
            e.printStackTrace();
        } catch (AuthenticatorException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return token;
    }

    private Account getAccount(String accountName) {
        for (Account item : mAccountManager.getAccountsByType("com.google")) {
            if (item.name.equals(accountName)) {
                return item;
            }
        }
        return null;
    }
}
