#include <stdio.h>
#include <string.h>

#include "ublox.h"
#include "myuart.h"
#include "byte2var.h"

// UBX packet内のデータ位置
#define UBX_CLASS   2
#define UBX_PAYLOAD 6
// payload長さからパケット全体の長さ(checksum含む)をget
#define UBX_PKTLEN(len_payload) (len_payload + 8)



// ublox_rcv_pkt()用
#define UBX_STS_HEAD1   0
#define UBX_STS_HEAD2   1
#define UBX_STS_CLASS   2
#define UBX_STS_ID      3
#define UBX_STS_LEN1    4
#define UBX_STS_LEN2    5
#define UBX_STS_PAYLOAD 6
#define UBX_STS_SUMA    7
#define UBX_STS_SUMB    8

/*
 * lenp: payload length
 */
static void ublox_make_sum(unsigned char *pkt, unsigned int lenp, unsigned char *suma, unsigned char *sumb)
{
    unsigned int    i;
    *suma = 0;
    *sumb = 0;
    for(i = 0; i < lenp+4; i++) {
        *suma += pkt[UBX_CLASS + i];
        *sumb += *suma;
    }
}
/*
 * GPSに送信するコマンドパケットの作成
 * unsgined char lenp: payloadのバイト数
 * return: 作成したパケットpktのバイト数
 */
int ublox_make_cmd(unsigned char id1, unsigned char id2, unsigned int lenp, unsigned char *payload, unsigned char pkt[])
{
    unsigned char   suma, sumb;

    pkt[0] = 0xb5;
    pkt[1] = 0x62;
    pkt[2] = id1;
    pkt[3] = id2;
    pkt[4] = (0xFF & lenp);
    pkt[5] = (0xFF & (lenp >> 8));
    memcpy(&pkt[UBX_PAYLOAD], payload, lenp);
    // Check Sum
    ublox_make_sum(pkt, lenp, &suma, &sumb);
    pkt[UBX_PAYLOAD + lenp] = suma;
    pkt[UBX_PAYLOAD + lenp + 1] = sumb;
    return UBX_PKTLEN(lenp);
}
/*
 * Debug用
 * GPSから受信したパケットのモニタ出力
 */
void ublox_debug_pkt_out(unsigned char *pkt, int len)
{
    unsigned int i;
    char    buf[16];

    for(i = 0; i < len; i++) {
        sprintf(buf, "%02X ", (unsigned char)(pkt[i] & 0xFF));
        PDEBUG(buf);
    }
    PDEBUG("\r\n");
}
/*
 * ubloxにデータ送信
 * polling
 */
void ublox_send(unsigned char *pkt, unsigned int len_pkt)
{
    unsigned int i;

    for(i = 0; i < len_pkt; i++) {
//        while(U1STAbits.UTXBF==1); /* Wait until TX buf read for new data */
        while(uart1_tx_is_full());
        uart1_putc(pkt[i]);

    }
}
/**** コマンド送信 *************************/
/*
 * CFG-PRT
 *  プロトコル選択
*/
void ublox_send_cfg_prt(unsigned long bps)
{
    const unsigned char   lenp = 20;
    unsigned char   payload[lenp];
    unsigned char   pkt[32];
    unsigned int i;
    unsigned int len_pkt;

    payload[0] = 0x01;  // Port
    payload[1] = 0x00;
    payload[2] = 0x00;  // TxReady
    payload[3] = 0x00;
    // mode
    payload[4] = 0b11010000;    // 8bit
    payload[5] = 0b00001000;    // 1stop non-parity
    payload[6] = 0x00;
    payload[7] = 0x00;
    // baudrate
    payload[8] = (bps & 0xFF);
    payload[9] = ((bps >> 8)& 0xFF);
    payload[10] = ((bps >> 16)& 0xFF);
    payload[11] = ((bps >> 24)& 0xFF);
    payload[12] = 0x01;     // inProtoMask
    payload[13] = 0x00;
    payload[14] = 0x01;     // outProtoMask
    payload[15] = 0x00;
    payload[16] = 0x00;     // reserved
    payload[17] = 0x00;
    payload[18] = 0x00;
    payload[19] = 0x00;

    len_pkt = ublox_make_cmd(UBX_CLS_CFG, UBX_ID_CFG_PRT, lenp, payload, pkt);
    ublox_send(pkt, len_pkt);

}
/*
 CFG-NMEA
 */
void ublox_send_cfg_nmea(void)
{
    const unsigned char   lenp = 4;
    unsigned char   payload[lenp];
    unsigned char   pkt[32];
    unsigned int len_pkt;

    payload[0] = 0x00;
    payload[1] = 0x23;
    payload[2] = 0xFF;
    payload[3] = 0x02;

    len_pkt = ublox_make_cmd(UBX_CLS_CFG, 0x17, lenp, payload, pkt);
    ublox_send(pkt, len_pkt);
}
/*
 * CFG-MSG
 * パケットの出力レートを指定
 */
void ublox_send_cfg_msg(void)
{
    const unsigned char   lenp = 3;
    unsigned char   payload[lenp];
    unsigned char   pkt[32];
    unsigned int len_pkt;

    payload[0] = UBX_CLS_NAV;
    payload[1] = UBX_ID_NAV_TIMEUTC;
    payload[2] = 1; // Rate[Hz]

    len_pkt = ublox_make_cmd(UBX_CLS_CFG, UBX_ID_CFG_MSG, lenp, payload, pkt);
    ublox_send(pkt, len_pkt);
}
/*
 * CFG_TP5
 * TIMEPULSE2
 * AD用CLOCK出力
 */
void ublox_send_cfg_tp5_timepulse2(void)
{
    const unsigned char   lenp = 32;
    unsigned char   payload[lenp];
    unsigned char   pkt[64];
    unsigned int len_pkt;

    memset(payload, 0, lenp);

    payload[0] = 0x01;  // Timepulse2

    payload[4] = 50;    // cable delay ns

    payload[8] = 0x00;  // Freq 7.372800MHz
    payload[9] = 0x80;
    payload[10] = 0x70;
    payload[11] = 0x00;

    payload[12] = 0x00;  // Freq locked 7.372800MHz
    payload[13] = 0x80;
    payload[14] = 0x70;
    payload[15] = 0x00;

    payload[16] = 00;   // Pulse Duty
    payload[17] = 00;
    payload[18] = 00;
    payload[19] = 0x80;

    payload[20] = 00;   // Pulse Duty locked
    payload[21] = 00;
    payload[22] = 00;
    payload[23] = 0x80;

    // gridUtcGps,ploarity,alignToTow,isLength,isFreq,lockedOtherSet,LockGpsFreq,Active
    payload[28] = 0b11101111;   // Grid=GPS,Pol=Rise

    len_pkt = ublox_make_cmd(UBX_CLS_CFG, UBX_ID_CFG_TP5, lenp, payload, pkt);
    ublox_send(pkt, len_pkt);
}

/*
 * CFG TP5
 * TIMEPULSE1
 * 1Hz OUT
 */
void ublox_send_cfg_tp5_timepulse1(void)
{
    const unsigned char   lenp = 32;
    unsigned char   payload[lenp];
    unsigned char   pkt[64];
    unsigned int len_pkt;

    memset(payload, 0, lenp);

    payload[0] = 0x00; //Timepulse1

    payload[4] = 50;    // cable delay ns

    payload[8] = 0x40;  // Period 1,000,000us
    payload[9] = 0x42;
    payload[10] = 0x0F;
    payload[11] = 0x00;

    payload[12] = 0x40;  // Period Locked us
    payload[13] = 0x42;
    payload[14] = 0x0F;
    payload[15] = 0x00;

    payload[16] = 0x20;   // Pulse width 500,000us
    payload[17] = 0xA1;
    payload[18] = 0x07;
    payload[19] = 0x00;

    payload[20] = 0xA0;   // Pulse width locked 100,000us
    payload[21] = 0x86;
    payload[22] = 0x01;
    payload[23] = 0x00;

    payload[28] = 0b11110111;   // Grid=GPS,Pol=Rise

    len_pkt = ublox_make_cmd(UBX_CLS_CFG, UBX_ID_CFG_TP5, lenp, payload, pkt);
    ublox_send(pkt, len_pkt);
}
/**** DEBUG ************************************************/
#if 0
void ublox_rcv_poll(void)
{
    unsigned int i, data;
    char    buf[16];
    unsigned char    buf_rcv[48];

    // Header wait
    for(i = 0; i < 1000; i++) {
//        while(DataRdyUART1() == 0);
//        data = ReadUART1();
        while(U1STAbits.URXDA == 0);
        data = U1RXREG;
        if ((data & 0xFF) == 0xB5) break;
    }
    // rcv
    for(i = 0; i < 48; i++) {
        if (U1STAbits.OERR == 1) {
            U1STAbits.OERR = 0;
        }
        buf_rcv[i]= U1RXREG;
        while(U1STAbits.URXDA == 0);
    }
    // debug out
    for(i = 0; i < 48; i++) {
//        while(DataRdyUART1() == 0);
//        data = ReadUART1();
        sprintf(buf, "%02X ", buf_rcv[i]);
        PDEBUG(buf);
    }

    PDEBUG("\r\n");
}
#endif
/*
 * polling 受信
 */
void ublox_rcv_poll(void)
{
    unsigned int i, data;
    char    buf[16];

    // Header wait
    for(i = 0; i < 1000; i++) {
//        while(DataRdyUART1() == 0);
//        data = ReadUART1();
        while(uart1_rcvbuf_is_data() == 0);
        data = uart1_rcvbuf_getc();
        sprintf(buf, "%02X ", data);
        PDEBUG(buf);
    }
}
/**** パケット受信 *************************************/
/*
 * NAV-TIMEUTCデコード
 * return
 *  0=OK
 *  -1=エラー
 */

int ublox_decode_nav_timeutc(unsigned char *payload, UbloxNavTimeUtc *g)
{
    g->tow = byte4_to_ulong(payload);   // ms
    g->tacc = byte4_to_ulong(payload+4);    // ns
    g->nano = byte4_to_long(payload+8);     // ns
    g->year = byte2_to_uint(payload+12);
    g->month = byte1_to_uchar(payload+14);
    g->day = byte1_to_uchar(payload+15);
    g->hour = byte1_to_uchar(payload+16);
    g->min = byte1_to_uchar(payload+17);
    g->sec = byte1_to_uchar(payload+18);
    g->valid = byte1_to_uchar(payload+19);
    return 0;
}
/*
 * 受信バッファから1packet取り出す
 * UART割り込み使用
 * return
 *  0=受信OK
 *  -1=受信エラー or タイムアウト
 */
//TODO: timeout処理, timeout指定
//TODO: check sum check
int ublox_rcv_pkt(unsigned char *class, unsigned char *id, unsigned int *len, unsigned char *payload)
{
    unsigned int i, data;
    char    sts = UBX_STS_HEAD1;
    unsigned int ptr;
    unsigned char   suma, sumb;

    while(1) {
        while(uart1_rcvbuf_is_data() == 0);
        data = uart1_rcvbuf_getc();
        switch(sts) {
            case UBX_STS_HEAD1:
                if (data == 0xB5) sts = UBX_STS_HEAD2;
                break;
            case UBX_STS_HEAD2:
                if (data == 0x62) sts = UBX_STS_CLASS;
                break;
            case UBX_STS_CLASS:
                *class = data;
                sts = UBX_STS_ID;
                break;
            case UBX_STS_ID:
                *id = data;
                sts = UBX_STS_LEN1;
                break;
            case UBX_STS_LEN1:
                *len = data;
                sts = UBX_STS_LEN2;
                break;
            case UBX_STS_LEN2:
                *len += ((data << 8) & 0xFF00);
                ptr = 0;
                sts = UBX_STS_PAYLOAD;
                break;
            case UBX_STS_PAYLOAD:
                payload[ptr++] = data;
                if (ptr >= *len) sts = UBX_STS_SUMA;
                break;
            case UBX_STS_SUMA:
                suma = data;
                sts = UBX_STS_SUMB;
                break;
            case UBX_STS_SUMB:
                sumb = data;
                return 0;
        }
    }
}
/*
 * 受信バッファからACKまたはNAKを取り出す
 * return
 * 　1=ACK受信
 * 　0=NACK受信
 * 　-1=timeout
 */
// TODO: timeout処理,timeoutを引数で指定
int ublox_rcv_ack(void)
{
    unsigned char class, id;
    unsigned int    len;
    unsigned char   payload[128];

    while(1) {
        ublox_rcv_pkt(&class, &id, &len, payload);
        if (class == 0x05 && id == 0x01) {
            // ACK
//            PDEBUG("ACK\r\n");
            return 1;
        } else if (class == 0x05 && id == 0x00) {
            // NACK
//            PDEBUG("NACK\r\n");
            return 0;
        }
    }
}
