/**
 * 
 */
package jp.mydns.masahase.sdes.common;

/**
 * FEMでよく用いられる定数および関数
 * 
 * @author MASA.H
 * 
 */
public final class FemFunc {
	/**
	 * ガウス・ルジャンドルの積分公式におけるサンプリング点が2である場合の係数
	 */
	public static double GaussLegendreFactor2 = 0.577350269189626;

	/**
	 * 4ノード四辺形要素の補間関数
	 * 
	 * @param u
	 *            正規化された座標1
	 * @param v
	 *            正規化された座標2
	 * @param value
	 *            各ノードでの値（double[4]）
	 * @return 指定された座標での値
	 */
	static public double formFunction4Square(double u, double v, double[] value) {
		double N1 = ((1 + u) * (1 + v)) / 4;
		double N2 = ((1 - u) * (1 + v)) / 4;
		double N3 = ((1 + u) * (1 - v)) / 4;
		double N4 = ((1 + u) * (1 - v)) / 4;
		return N1 * value[0] + N2 * value[1] + N3 * value[2] + N4 * value[3];
	}

	/**
	 * 8ノードレンガ型要素の補間関数
	 * 
	 * @param u
	 *            正規化された座標
	 * @param v
	 *            正規化された座標
	 * @param w
	 *            正規化された座標
	 * @param value
	 *            各ノードでの値(double[8])
	 * @return 指定された座標での値
	 */
	static public double formFunction8Blick(double u, double v, double w,
			double[] value) {
		double N[] = new double[8];
		double ret = 0;
		N[0] = (u - 1) * (v - 1) * (w - 1) / -8;
		N[1] = (u + 1) * (v - 1) * (w - 1) / 8;
		N[2] = (u + 1) * (v + 1) * (w - 1) / -8;
		N[3] = (u - 1) * (v + 1) * (w - 1) / 8;
		N[4] = (u - 1) * (v - 1) * (w + 1) / 8;
		N[5] = (u + 1) * (v - 1) * (w + 1) / -8;
		N[6] = (u + 1) * (v + 1) * (w + 1) / 8;
		N[7] = (u - 1) * (v + 1) * (w + 1) / -8;
		for (int i = 0; i < 8; i++) {
			ret += N[i] * value[i];
		}
		return ret;
	}

	/**
	 * 3ノード三角形の補間関数
	 * 
	 * @param u
	 *            　 正規化された座標
	 * @param v
	 *            正規化された座標
	 * @param value
	 *            各ノードでの値(double[3])
	 * @return 指定された座標での値
	 */
	static public double formFunction3Triangle(double u, double v,
			double value[]) {
		double N[] = new double[3];
		double ret = 0;
		N[0] = u;
		N[1] = v;
		N[2] = 1 - u - v;
		for (int i = 0; i < 3; i++) {
			ret += N[i] * value[i];
		}
		return ret;
	}

	/**
	 * 8ノードレンガ型要素の補間関数の逆
	 * 
	 * @param x
	 *            デカルト座標系での座標(double[3])
	 * @param value
	 *            各ノードの座標(double[8][3])
	 * @return 正規化座標系での座標(double[3])
	 */
	static public double[] invertFormFunction8Blick(double[] x, double[][] value) {
		double[] ret = null;
		double u, v, w;
		double ru, rv, rw;
		double x0, y0, z0, x9, y9, z9;
		double a, b, c, d, e, f, g, h, j;
		double[] tmp_x, tmp_y, tmp_z;
		double eps = 1e-6;
		u = 0;
		v = 0;
		w = 0;
		x0 = x[0];
		y0 = x[1];
		z0 = x[2];
		tmp_x = new double[8];
		tmp_y = new double[8];
		tmp_z = new double[8];
		for (int i = 0; i < tmp_x.length; i++) {
			tmp_x[i] = value[i][0];
			tmp_y[i] = value[i][1];
			tmp_z[i] = value[i][2];
		}
		for (int i = 0; i < 1.0e5; i++) {
			x9 = formFunction8Blick(u, v, w, tmp_x);
			y9 = formFunction8Blick(u, v, w, tmp_y);
			z9 = formFunction8Blick(u, v, w, tmp_z);
			// Jf=matrix([a,b,c],[d,e,f],[g,h,i]);
			a = -((v + 1) * (w + 1) * tmp_x[7]) / 8
					+ ((v + 1) * (w + 1) * tmp_x[6]) / 8
					- ((v - 1) * (w + 1) * tmp_x[5]) / 8
					+ ((v - 1) * (w + 1) * tmp_x[4]) / 8
					+ ((v + 1) * (w - 1) * tmp_x[3]) / 8
					- ((v + 1) * (w - 1) * tmp_x[2]) / 8
					+ ((v - 1) * (w - 1) * tmp_x[1]) / 8
					- ((v - 1) * (w - 1) * tmp_x[0]) / 8;
			b = -((u - 1) * (w + 1) * tmp_x[7]) / 8
					+ ((u + 1) * (w + 1) * tmp_x[6]) / 8
					- ((u + 1) * (w + 1) * tmp_x[5]) / 8
					+ ((u - 1) * (w + 1) * tmp_x[4]) / 8
					+ ((u - 1) * (w - 1) * tmp_x[3]) / 8
					- ((u + 1) * (w - 1) * tmp_x[2]) / 8
					+ ((u + 1) * (w - 1) * tmp_x[1]) / 8
					- ((u - 1) * (w - 1) * tmp_x[0]) / 8;
			c = -((u - 1) * (v + 1) * tmp_x[7]) / 8
					+ ((u + 1) * (v + 1) * tmp_x[6]) / 8
					- ((u + 1) * (v - 1) * tmp_x[5]) / 8
					+ ((u - 1) * (v - 1) * tmp_x[4]) / 8
					+ ((u - 1) * (v + 1) * tmp_x[3]) / 8
					- ((u + 1) * (v + 1) * tmp_x[2]) / 8
					+ ((u + 1) * (v - 1) * tmp_x[1]) / 8
					- ((u - 1) * (v - 1) * tmp_x[0]) / 8;
			d = -((v + 1) * (w + 1) * tmp_y[7]) / 8
					+ ((v + 1) * (w + 1) * tmp_y[6]) / 8
					- ((v - 1) * (w + 1) * tmp_y[5]) / 8
					+ ((v - 1) * (w + 1) * tmp_y[4]) / 8
					+ ((v + 1) * (w - 1) * tmp_y[3]) / 8
					- ((v + 1) * (w - 1) * tmp_y[2]) / 8
					+ ((v - 1) * (w - 1) * tmp_y[1]) / 8
					- ((v - 1) * (w - 1) * tmp_y[0]) / 8;
			e = -((u - 1) * (w + 1) * tmp_y[7]) / 8
					+ ((u + 1) * (w + 1) * tmp_y[6]) / 8
					- ((u + 1) * (w + 1) * tmp_y[5]) / 8
					+ ((u - 1) * (w + 1) * tmp_y[4]) / 8
					+ ((u - 1) * (w - 1) * tmp_y[3]) / 8
					- ((u + 1) * (w - 1) * tmp_y[2]) / 8
					+ ((u + 1) * (w - 1) * tmp_y[1]) / 8
					- ((u - 1) * (w - 1) * tmp_y[0]) / 8;
			f = -((u - 1) * (v + 1) * tmp_y[7]) / 8
					+ ((u + 1) * (v + 1) * tmp_y[6]) / 8
					- ((u + 1) * (v - 1) * tmp_y[5]) / 8
					+ ((u - 1) * (v - 1) * tmp_y[4]) / 8
					+ ((u - 1) * (v + 1) * tmp_y[3]) / 8
					- ((u + 1) * (v + 1) * tmp_y[2]) / 8
					+ ((u + 1) * (v - 1) * tmp_y[1]) / 8
					- ((u - 1) * (v - 1) * tmp_y[0]) / 8;
			g = -((v + 1) * (w + 1) * tmp_z[7]) / 8
					+ ((v + 1) * (w + 1) * tmp_z[6]) / 8
					- ((v - 1) * (w + 1) * tmp_z[5]) / 8
					+ ((v - 1) * (w + 1) * tmp_z[4]) / 8
					+ ((v + 1) * (w - 1) * tmp_z[3]) / 8
					- ((v + 1) * (w - 1) * tmp_z[2]) / 8
					+ ((v - 1) * (w - 1) * tmp_z[1]) / 8
					- ((v - 1) * (w - 1) * tmp_z[0]) / 8;
			h = -((u - 1) * (w + 1) * tmp_z[7]) / 8
					+ ((u + 1) * (w + 1) * tmp_z[6]) / 8
					- ((u + 1) * (w + 1) * tmp_z[5]) / 8
					+ ((u - 1) * (w + 1) * tmp_z[4]) / 8
					+ ((u - 1) * (w - 1) * tmp_z[3]) / 8
					- ((u + 1) * (w - 1) * tmp_z[2]) / 8
					+ ((u + 1) * (w - 1) * tmp_z[1]) / 8
					- ((u - 1) * (w - 1) * tmp_z[0]) / 8;
			j = -((u - 1) * (v + 1) * tmp_z[7]) / 8
					+ ((u + 1) * (v + 1) * tmp_z[6]) / 8
					- ((u + 1) * (v - 1) * tmp_z[5]) / 8
					+ ((u - 1) * (v - 1) * tmp_z[4]) / 8
					+ ((u - 1) * (v + 1) * tmp_z[3]) / 8
					- ((u + 1) * (v + 1) * tmp_z[2]) / 8
					+ ((u + 1) * (v - 1) * tmp_z[1]) / 8
					- ((u - 1) * (v - 1) * tmp_z[0]) / 8;
			// Jf.matrix([ru],[rv],[rw])=F;
			ru = (c * (e * (z9 - z0) - h * y9 + h * y0) + b
					* (f * (z0 - z9) + j * y9 - j * y0) + (f * h - e * j) * x9 + (e
					* j - f * h)
					* x0)
					/ (a * (f * h - e * j) + b * (d * j - f * g) + c
							* (e * g - d * h));
			rv = -(c * (d * (z9 - z0) - g * y9 + g * y0) + a
					* (f * (z0 - z9) + j * y9 - j * y0) + (f * g - d * j) * x9 + (d
					* j - f * g)
					* x0)
					/ (a * (f * h - e * j) + b * (d * j - f * g) + c
							* (e * g - d * h));
			rw = (b * (d * (z9 - z0) - g * y9 + g * y0) + a
					* (e * (z0 - z9) + h * y9 - h * y0) + (e * g - d * h) * x9 + (d
					* h - e * g)
					* x0)
					/ (a * (f * h - e * j) + b * (d * j - f * g) + c
							* (e * g - d * h));

			// 残差の評価
			double r = Math.sqrt(ru * ru + rv * rv + rw * rw);
			if (r <= eps)
				break;
			if (ru == java.lang.Double.NaN || rv == java.lang.Double.NaN
					|| rw == java.lang.Double.NaN) {
				ru = eps;
				rv = eps;
				rw = eps;
			}
			u = u - ru;
			v = v - rv;
			w = w - rw;
		}
		if (u == java.lang.Double.NaN || v == java.lang.Double.NaN
				|| w == java.lang.Double.NaN) {
			throw new java.lang.RuntimeException();
		}
		ret = new double[3];
		ret[0] = u;
		ret[1] = v;
		ret[2] = w;
		return ret;
	}

	/**
	 * 4ノード四辺形要素の補間関数の逆
	 * 
	 * @param x
	 *            デカルト座標系での座標(double[2])
	 * @param value
	 *            各ノードの座標(double[4][2])
	 * @return 正規化座標系での座標(double[2])
	 */
	public static double[] invertFormFunction4Square(double[] x,
			double[][] value) {
		double[] ret = null;
		double u, v;
		double ru, rv, r;
		double x0, x1, x2, x3, x4;
		double y0, y1, y2, y3, y4;
		double eps = 1.0e-6;
		u = 0;
		v = 0;
		ru = 0;
		rv = 0;
		x0 = x[0];
		y0 = x[1];
		x1 = value[0][0];
		y1 = value[0][1];
		x2 = value[1][0];
		y2 = value[1][1];
		x3 = value[2][0];
		y3 = value[2][1];
		x4 = value[3][0];
		y4 = value[3][1];
		// ニュートン法による計算
		for (int i = 0; i < 1.0e5; i++) {
			double u2, v2;
			u2 = u * u;
			v2 = v * v;
			ru = -(x1
					* (u * (2 * y4 - 2 * y0) + u2 * (y4 - y3) + y4 + y3 - 2 * y0)
					+ x2
					* (y4 + u2 * (y3 - y4) + y3 + u * (2 * y0 - 2 * y3) - 2 * y0)
					+ x0
					* (-2 * y4 + u * (-2 * y4 + 2 * y3 - 2 * y2 + 2 * y1) - 2
							* y3 + 2 * y2 + 2 * y1)
					+ x3
					* (u * (2 * y2 - 2 * y0) - y2 + u2 * (y1 - y2) - y1 + 2 * y0) + x4
					* (u2 * (y2 - y1) - y2 - y1 + u * (2 * y0 - 2 * y1) + 2 * y0))
					/ (x2 * (u * (y4 - y3) + v * (y4 - y1) + y3 - y1) + x3
							* (y4 + v * (y1 - y4) + u * (y2 - y1) - y2) + x1
							* (-y4 + u * (y3 - y4) + v * (y2 - y3) + y2) + x4
							* (v * (y3 - y2) - y3 + u * (y1 - y2) + y1));
			rv = (x3
					* (v * (2 * y4 - 2 * y0) - y4 + v2 * (y1 - y4) - y1 + 2 * y0)
					+ x2
					* (v2 * (y4 - y1) - y4 - y1 + v * (2 * y0 - 2 * y1) + 2 * y0)
					+ x0
					* (2 * y4 + v * (-2 * y4 + 2 * y3 - 2 * y2 + 2 * y1) - 2
							* y3 - 2 * y2 + 2 * y1)
					+ x4
					* (v2 * (y3 - y2) + y3 + v * (2 * y0 - 2 * y3) + y2 - 2 * y0) + x1
					* (y3 + v2 * (y2 - y3) + v * (2 * y2 - 2 * y0) + y2 - 2 * y0))
					/ (x2 * (u * (y4 - y3) + v * (y4 - y1) + y3 - y1) + x3
							* (y4 + v * (y1 - y4) + u * (y2 - y1) - y2) + x1
							* (-y4 + u * (y3 - y4) + v * (y2 - y3) + y2) + x4
							* (v * (y3 - y2) - y3 + u * (y1 - y2) + y1));
			r = Math.sqrt(ru * ru + rv * rv);// 残差の計算
			if (r < eps)
				break;
			if (r == java.lang.Double.NaN || java.lang.Double.isInfinite(r)) {
				ru = eps;
				rv = eps;
			}
			if (ru > 1000)
				ru = 1000;
			if (ru < -1000)
				ru = -1000;
			if (rv > 1000)
				rv = 1000;
			if (rv < -1000)
				rv = -1000;
			u = u - ru;
			v = v - rv;
		}
		if (u == java.lang.Double.NaN || v == java.lang.Double.NaN) {
			throw new java.lang.RuntimeException();
		}
		ret = new double[2];
		ret[0] = u;
		ret[1] = v;
		// TODO: 精度確認の追加
		return ret;
	}

	/**
	 * 3ノード3角形要素の補間関数の逆
	 * 
	 * @param x
	 *            デカルト座標系での座標(double[2])
	 * @param value
	 *            各ノードの座標(double[3][2])
	 * @return 正規化座標系での座標(double[2])
	 */
	public static double[] invertFormFunction3Triangle(double[] x,
			double[][] value) {
		double[] ret = new double[2];
		double ru, rv, tmp;
		double u0, v0, x1, x2, x3, y1, y2, y3;
		double eps = 1.0e-6;
		u0 = 0.0;
		v0 = 0.0;
		x1 = value[0][0];
		y1 = value[0][1];
		x2 = value[1][0];
		y2 = value[1][1];
		x3 = value[2][0];
		y3 = value[2][1];
		for (int i = 0; i < 1.0e5; i++) {
			tmp = x1 * (y3 - y2) + x2 * (y1 - y3) + x3 * (y2 - y1);
			if (tmp == 0.0) {
				u0 = (x[1] - y1) / (x3 - x1);
				v0 = u0;
				break;
			}
			ru = (x2 * (y3 + u0 * (y1 - y3) - x[1]) + u0 * x1 * (y3 - y2)
					+ x[0] * (y2 - y3) + x3 * (u0 * (y2 - y1) - y2 + x[1]))
					/ tmp;
			rv = (x1 * (v0 * (y3 - y2) - y3 + x[1]) + x[0] * (y3 - y1) + v0
					* x2 * (y1 - y3) + x3 * (v0 * (y2 - y1) + y1 - x[1]))
					/ tmp;
			if (Math.sqrt(ru * ru + rv * rv) < eps)
				break;
			if (ru == java.lang.Double.NaN || rv == java.lang.Double.NaN) {
				ru = eps;
				rv = eps;
			}
			u0 = u0 - ru;
			v0 = v0 - rv;
		}
		if (u0 == java.lang.Double.NaN || v0 == java.lang.Double.NaN) {
			throw new java.lang.RuntimeException();
		}
		ret[0] = u0;
		ret[1] = v0;
		return ret;
	}

	/**
	 * 相当塑性ひずみを計算。
	 * 全塑性理論に基づく相当塑性ひずみ
	 * 
	 * @param strain
	 *            ひずみ(double[6])
	 * @return 相当塑性ひずみ(double)
	 */
	public static double calcEstimateStrain(double[] strain) {
		double ret;
		ret = Math
				.sqrt((2 * (strain[0] * strain[0] + strain[1] * strain[1] + strain[2]
						* strain[2]) + (strain[3] * strain[3] + strain[4]
						* strain[4] + strain[5] * strain[5])) / 3);
		return ret;
	}
}
