/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sed;

import java.io.IOException;
import java.io.PushbackReader;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.morilib.sed.cmd.SedCommands;
import net.morilib.sed.line.SedLastLineMatcher;
import net.morilib.sed.line.SedNumberLineMatcher;
import net.morilib.sed.line.SedPatternLineMatcher;
import net.morilib.sed.parser.SedAParser;
import net.morilib.sed.parser.SedBParser;
import net.morilib.sed.parser.SedCParser;
import net.morilib.sed.parser.SedCommentParser;
import net.morilib.sed.parser.SedCompositeParser;
import net.morilib.sed.parser.SedIParser;
import net.morilib.sed.parser.SedLabelParser;
import net.morilib.sed.parser.SedQParser;
import net.morilib.sed.parser.SedRParser;
import net.morilib.sed.parser.SedSParser;
import net.morilib.sed.parser.SedSimpleCommandParser;
import net.morilib.sed.parser.SedTParser;
import net.morilib.sed.parser.SedWParser;
import net.morilib.sed.parser.SedYParser;


public class SedParser {

	private static final Map<Integer, SedCommandParser> PARSERS;

	static {
		Map<Integer, SedCommandParser> p;

		p = new HashMap<Integer, SedCommandParser>();
		p.put((int)'s', new SedSParser());
		p.put((int)'#', new SedCommentParser());
		p.put((int)'q', new SedQParser());
		p.put((int)'d', new SedSimpleCommandParser(SedCommands.D));
		p.put((int)'p', new SedSimpleCommandParser(SedCommands.P));
		p.put((int)'n', new SedSimpleCommandParser(SedCommands.N));
		p.put((int)'y', new SedYParser());
		p.put((int)'a', new SedAParser());
		p.put((int)'i', new SedIParser());
		p.put((int)'c', new SedCParser());
		p.put((int)'=', new SedSimpleCommandParser(SedCommands.EQ));
		p.put((int)'r', new SedRParser());
		p.put((int)'w', new SedWParser());
		p.put((int)'N', new SedSimpleCommandParser(SedCommands.NL));
		p.put((int)'P', new SedSimpleCommandParser(SedCommands.PL));
		p.put((int)'D', new SedSimpleCommandParser(SedCommands.DL));
		p.put((int)'h', new SedSimpleCommandParser(SedCommands.H));
		p.put((int)'H', new SedSimpleCommandParser(SedCommands.HL));
		p.put((int)'g', new SedSimpleCommandParser(SedCommands.G));
		p.put((int)'G', new SedSimpleCommandParser(SedCommands.GL));
		p.put((int)'x', new SedSimpleCommandParser(SedCommands.X));
		p.put((int)':', new SedLabelParser());
		p.put((int)'b', new SedBParser());
		p.put((int)'t', new SedTParser());
		p.put((int)'{', new SedCompositeParser());
		PARSERS = Collections.unmodifiableMap(p);
	}

	public static List<SedCommandLine> parse(
			Reader rd) throws IOException {
		List<SedCommandLine> r = new ArrayList<SedCommandLine>();
		PushbackReader re = new PushbackReader(rd);
		SedCommandLine l;
		int c;

		while((c = re.read()) >= 0) {
			re.unread(c);
			if((l = parseLine(re)) != null)  r.add(l);
			if((c = re.read()) >= 0 && c != '\n' && c != ';') {
				throw new SedSyntaxException((char)c + "");
			}
		}
		return r;
	}

	public static SedCommandLine parseLine(String rd) {
		PushbackReader re = new PushbackReader(new StringReader(rd));
		SedCommandLine l;

		try {
			l = parseLine(re);
			if(re.read() >= 0)  throw new SedSyntaxException();
			return l;
		} catch(IOException e) {
			throw new RuntimeException(e);
		}
	}

	public static SedCommandLine parseLine(
			PushbackReader rd) throws IOException {
		SedCommandLine r = new SedCommandLine();

		r.matcher = lines(rd);
		return (r.command = parsecmd(rd, r.matcher)) == null ?
				null : r;
	}

	static SedCommand parsecmd(PushbackReader rd,
			SedLineOption l) throws IOException {
		SedCommandParser p;
		int c;

		if((c = rd.read()) < 0 ||
				c == '\n' || (p = PARSERS.get(c)) == null) {
			throw new SedSyntaxException();
		}
		return p.parse(rd, l);
	}

	static SedLineOption lines(PushbackReader rd) throws IOException {
		SedLineMatcher m, n;
		SedLineOption r;
		String s;
		long a, b;
		int c;

		if((c = rd.read()) == 'g') {
			if((c = rd.read()) < 0 || c == '\n') {
				throw new SedSyntaxException();
			}
			s = getstr(rd, c);
			r = new SedLineOption(s);

			if((c = rd.read()) == '!') {
				r.negate = true;
			} else {
				rd.unread(c);
			}
		} else {
			rd.unread(c);
			if((m = line(rd)) != null) {
				if((c = rd.read()) == ',') {
					n = line(rd);
					if(m instanceof SedNumberLineMatcher &&
							n instanceof SedNumberLineMatcher) {
						a = ((SedNumberLineMatcher)m).getLineno();
						b = ((SedNumberLineMatcher)n).getLineno();
						if(b < a)  throw new SedSyntaxException();
					} else if(m instanceof SedLastLineMatcher &&
							!(n instanceof SedLastLineMatcher)) {
						throw new SedSyntaxException();
					}
					r = new SedLineOption(m, n);
					if((c = rd.read()) == '!') {
						r.negate = true;
					} else {
						rd.unread(c);
					}
				} else if(c == '!') {
					r = new SedLineOption(m);
					r.negate = true;
				} else {
					rd.unread(c);
					r = new SedLineOption(m);
				}
			} else {
				r = new SedLineOption();
			}
		}
		return r;
	}

	static SedLineMatcher line(PushbackReader rd) throws IOException {
		SedLineMatcher r;
		String s;
		int c;

		if((c = rd.read()) == '/') {
			s = getstr(rd, '/');
			r = new SedPatternLineMatcher(s);
		} else if(c == '$') {
			r = SedLastLineMatcher.INSTANCE;
		} else if(c >= '0' && c <= '9') {
			rd.unread(c);
			r = new SedNumberLineMatcher(nums(rd));
		} else {
			rd.unread(c);
			r = null;
		}
		return r;
	}

	static long nums(PushbackReader rd) throws IOException {
		long r = 0;
		int c;

		while((c = rd.read()) >= '0' && c <= '9') {
			if((r = r * 10 + (c - '0')) < 0) {
				throw new SedSyntaxException();
			}
		}
		if(r <= 0)  throw new SedSyntaxException();
		rd.unread(c);
		return r;
	}

	static String getstr(PushbackReader rd,
			int dlm) throws IOException {
		StringBuffer b = new StringBuffer();
		boolean esc = false;
		int c;

		while((c = rd.read()) != dlm || esc) {
			if(c < 0) {
				throw new SedSyntaxException();
			} else if(esc) {
				esc = false;
			} else if(!(esc = (c == '\\'))) {
				b.appendCodePoint(c);
			}
		}
		return b.toString();
	}

}
