package noguchi.SentenceLearner;

import java.util.ArrayList;
import java.util.List;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.database.sqlite.SQLiteDatabase;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.os.Bundle;
import android.os.Handler;
import android.view.GestureDetector;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.RatingBar;
import android.widget.RelativeLayout;
import android.widget.TextView;
import android.widget.Toast;

public class QuestionViewer extends Activity 
implements Button.OnClickListener,
View.OnTouchListener,
GestureDetector.OnGestureListener,
GestureDetector.OnDoubleTapListener,
MediaPlayer.OnCompletionListener,
RatingBar.OnRatingBarChangeListener {

	// DB object
	private DatabaseHelper helper;
	private SQLiteDatabase db;

	// GUI components
	Button questionButton;
	Button prevButton;
	Button answerButton;
	Button nextButton;
	TextView answerView;
	TextView soundView;
	RelativeLayout layout;
	private RatingBar ratingBar;

	// questions
	private List<QuestionItem> questions;
	private int questionIndex;

	// Media Player
	private MediaPlayer mplayer;
	private boolean isPlay;

	private Handler handler = new Handler();
	private Runnable runnable;
	private final int MONITOR_INTERVAL = 10;
	private int _endPos = -1;

	private boolean isAnswerDisplayed = false;

	// For menu
	private static final int MENU_ID_MENU1 = (Menu.FIRST + 1);
	private static final int MENU_ID_MENU2 = (Menu.FIRST + 2);

	// For automation
	private boolean isAuto = false;
	private int sleepInAuto = 1;
	private int repeatInAuto = 20;
	private int answerCountInAuto = 0;

	// Gesture
	private GestureDetector gestureDetector;

	/** Called when the activity is first created. */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.questionviewer);

		// Avoid sleep mode
		getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);

		// Set Volume Control key behavier
		setVolumeControlStream(AudioManager.STREAM_MUSIC);
		// Create the database (if not exist yet)
		helper = new DatabaseHelper(this);
		db = helper.getReadableDatabase();

		// Initialize objects for Questions
		questions = new ArrayList<QuestionItem>();

		// Initialize GUI components
		questionButton = (Button)findViewById(R.id.Question);
		prevButton = (Button)findViewById(R.id.btnPrev);
		answerButton = (Button)findViewById(R.id.btnAnswer);
		nextButton = (Button)findViewById(R.id.btnNext);

		answerView = (TextView)findViewById(R.id.Answer);
		soundView = (TextView)findViewById(R.id.SoundName);

		layout = (RelativeLayout)findViewById(R.id.Layout01);

		ratingBar = (RatingBar)findViewById(R.id.RatingBar01);

		// Set listeners
		questionButton.setOnClickListener(this);
		prevButton.setOnClickListener(this);
		answerButton.setOnClickListener(this);
		nextButton.setOnClickListener(this);
		ratingBar.setOnRatingBarChangeListener(this);

		layout.setOnTouchListener(this);
		answerView.setOnTouchListener(this);
		this.gestureDetector = new GestureDetector(this, this);

		// Get params
		Bundle extras = getIntent().getExtras();	
		String[] soundFiles = null;
		int baseRating = 0;
		
		if (extras != null)
		{
			soundFiles = extras.getStringArray("TARGETS");
			baseRating = extras.getInt("RATING");
		}

		// Get questions
		try {
			for (int i = 0 ; i < soundFiles.length ; i++)
			{
				SoundItem sound = (SoundItem) helper.getSoundItem(soundFiles[i], db);
				List<QuestionItem> lst = helper.getQuestions(db, sound.getFileName(), baseRating);
				questions.addAll(lst);
			}
		}
		catch (Exception ex)
		{
			Toast.makeText(QuestionViewer.this, getResources().getString(R.string.DBFailed), Toast.LENGTH_SHORT).show();
		}
		if (questions.size() == 0)
		{
			Toast.makeText(QuestionViewer.this, getResources().getString(R.string.NoSoundsFound), Toast.LENGTH_SHORT).show();
			setResult(RESULT_OK);
			finish();
			return;
		}
		questionIndex = 0;

		// Initialize media player
		mplayer = new MediaPlayer();
		isPlay = false;

		// Set the thread which is used in stopping the sound
		runnable = new Runnable() {
			@Override
			public void run() {            	
				stopMonitor();    
				handler.postDelayed(this, MONITOR_INTERVAL);
			}
		};

		updateScreen();
	}

	@Override
	public void onPause()
	{
		stop();
		super.onPause();
	}

	private void updateScreen()
	{
		questionButton.setText(questions.get(questionIndex).getName());
		answerView.setText(questions.get(questionIndex).getAnswer());

		ratingBar.setRating(questions.get(questionIndex).getRating());

		String soundName = questions.get(questionIndex).getSoundName();
		String qStartString = QuestionItem.GetTimeString(questions.get(questionIndex).getQuestionStartTime());
		String aStartString = QuestionItem.GetTimeString(questions.get(questionIndex).getAnswerStartTime());
		String endString = QuestionItem.GetTimeString(questions.get(questionIndex).getEndTime());
		
		soundView.setText(soundName + " " + qStartString + " / " + aStartString + " / " + endString);

		int startPos = questions.get(questionIndex).getQuestionStartTime();
		int endPos = questions.get(questionIndex).getAnswerStartTime();
		String soundFileName = questions.get(questionIndex).getSoundFileName();
		play(soundFileName, startPos, endPos);

		isAnswerDisplayed = false;
	}

	private void play(String fileName, int startPos, int endPos)
	{
		if (startPos < 0 || endPos < 0) return;

		if (isPlay)
			stop();

		try {
			mplayer.setDataSource(fileName);
			mplayer.prepare();
			mplayer.start();
			isPlay = true;

			if (startPos > 0 && startPos < mplayer.getDuration())
			{
				mplayer.seekTo(startPos);
			}
			_endPos = endPos;
			mplayer.setOnCompletionListener(this);
			handler.postDelayed(runnable, MONITOR_INTERVAL);
		}
		catch (Exception ex)
		{
			Toast.makeText(this, R.string.palyerInitializefailed, Toast.LENGTH_LONG).show();
		}
	}

	private void stop()
	{
		handler.removeCallbacks(runnable);

		if (mplayer != null)
		{
			mplayer.stop();
			mplayer.reset();
		}
		isPlay = false;
		_endPos = -1;
	}

	private void stopAndNext()
	{
		stop();

		if (isAuto)
		{			
			if (isAnswerDisplayed && answerCountInAuto >= repeatInAuto)
			{
				answerCountInAuto = 0;
				next();
			}
			else
			{
				if (answerCountInAuto == 0)
				{
					try {
						Thread.sleep(sleepInAuto*1000);
					} catch (InterruptedException e) {}
				}
				answerCountInAuto ++;
				displayAnswer();
			}
		}
	}

	private void next()
	{
		questionIndex ++;
		if (questionIndex >= questions.size())
			questionIndex = 0;
		updateScreen();
	}

	private void prev()
	{
		questionIndex --;
		if (questionIndex < 0)
			questionIndex = questions.size()-1;
		updateScreen();
	}

	private void displayQuestion()
	{
		int startPos = questions.get(questionIndex).getQuestionStartTime();
		int endPos = questions.get(questionIndex).getAnswerStartTime();
		String soundFileName = questions.get(questionIndex).getSoundFileName();
		play(soundFileName, startPos, endPos);
	}

	private void displayAnswer()
	{
		int startPos = questions.get(questionIndex).getAnswerStartTime();
		int endPos = questions.get(questionIndex).getEndTime();
		String soundFileName = questions.get(questionIndex).getSoundFileName();
		play(soundFileName, startPos, endPos);
		isAnswerDisplayed = true;
	}

	private void stopMonitor()
	{
		if (isPlay) {
			int pos = mplayer.getCurrentPosition();
			if (pos > _endPos){
				stopAndNext();
			}
		}
	}

	@Override
	public void onClick(View arg0) {		
		// Disable clicks in the auto mode
		if (isAuto) return;

		if (arg0 == nextButton)
		{
			next();
		}
		else if (arg0 == prevButton)
		{
			prev();
		}
		else if (arg0 == questionButton)
		{
			displayQuestion();
		}
		else if (arg0 == answerButton)
		{
			displayAnswer();
		}
	}

	@Override
	public void onCompletion(MediaPlayer arg0) {
		stopAndNext();
	}

	// Handling a menu
	@Override
	public boolean onPrepareOptionsMenu(Menu menu) {
		// Add menus
		menu.clear();
		if (!isAuto)
		{
			menu.add(Menu.NONE, MENU_ID_MENU1, Menu.NONE, getResources().getString(R.string.EnableAuto));
		}
		else
		{
			menu.add(Menu.NONE, MENU_ID_MENU1, Menu.NONE, getResources().getString(R.string.DisableAuto));    		
		}

		menu.add(Menu.NONE, MENU_ID_MENU2, Menu.NONE, getResources().getString(R.string.SetRating));

		return super.onPrepareOptionsMenu(menu);
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		boolean ret = true;
		AlertDialog.Builder alertDialogBuilder;

		switch (item.getItemId()) {
		default:
			ret = super.onOptionsItemSelected(item);
			break;
		case MENU_ID_MENU1:
			if (!isAuto)
			{
				final EditText edit = new EditText(this);
				final EditText edit2 = new EditText(this);
				edit.setWidth(5);
				edit2.setWidth(5);
				edit.setText(Integer.toString(sleepInAuto));
				edit2.setText(Integer.toString(repeatInAuto));

				alertDialogBuilder = new AlertDialog.Builder(this);
				alertDialogBuilder.setTitle(getResources().getString(R.string.EnableAuto));
				alertDialogBuilder.setMessage(getResources().getString(R.string.SpecifySleepTime));
				//alertDialogBuilder.setView(edit); TODO: sleep time should be specified by user here.
				alertDialogBuilder.setView(edit2);	        	
				alertDialogBuilder.setPositiveButton(getResources().getString(R.string.Ok),
						new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) {
						CharSequence t = edit.getText();
						CharSequence t2 = edit2.getText();
						try 
						{
							sleepInAuto = Integer.parseInt(t.toString());
							repeatInAuto = Integer.parseInt(t2.toString());
							isAuto = true;
							next();
						}
						catch (Exception ex)
						{
							Toast.makeText(QuestionViewer.this, getResources().getString(R.string.ParseIntFailed), Toast.LENGTH_SHORT).show();
						}
					}
				});

				alertDialogBuilder.setCancelable(true);
				AlertDialog alertDialog = alertDialogBuilder.create();
				alertDialog.show();
			}
			else
			{
				isAuto = false;
				Toast.makeText(QuestionViewer.this, getResources().getString(R.string.AutoDisabled), Toast.LENGTH_SHORT).show();
			}
			break;
		case MENU_ID_MENU2:
			final RatingBar r = new RatingBar(this);
			r.setRating(ratingBar.getRating());

			alertDialogBuilder = new AlertDialog.Builder(this);
			alertDialogBuilder.setTitle(getResources().getString(R.string.SetRating));
			alertDialogBuilder.setMessage("");
			alertDialogBuilder.setView(r);	        	
			alertDialogBuilder.setPositiveButton(getResources().getString(R.string.Ok),
					new DialogInterface.OnClickListener() {
				@Override
				public void onClick(DialogInterface dialog, int which) {
					int v = (int)r.getRating();
					ratingBar.setRating(v);

					QuestionItem item = questions.get(questionIndex);
					item.setRating((int)ratingBar.getRating());
					try {
						helper.updateQuestionItemRating(item, db);
					}
					catch (Exception e)
					{
					}
				}
			});

			alertDialogBuilder.setCancelable(true);
			AlertDialog alertDialog = alertDialogBuilder.create();
			alertDialog.show();
			break;

		}
		return ret;
	}

	@Override
	public boolean onDoubleTap(MotionEvent arg0) {
		next();
		return true;
	}

	@Override
	public boolean onDoubleTapEvent(MotionEvent e) {
		// Do Nothing
		return false;
	}

	@Override
	public boolean onSingleTapConfirmed(MotionEvent e) {
		displayAnswer();
		return true;
	}

	@Override
	public boolean onDown(MotionEvent arg0) {
		// Do Nothing
		return true;
	}

	@Override
	public boolean onFling(MotionEvent arg0, MotionEvent arg1, float arg2,
			float arg3) {
		// Do Nothing
		return false;
	}

	@Override
	public void onLongPress(MotionEvent arg0) {
		// Do Nothing		
	}

	@Override
	public boolean onScroll(MotionEvent arg0, MotionEvent arg1, float arg2,
			float arg3) {
		// Do Nothing		
		return false;
	}

	@Override
	public void onShowPress(MotionEvent arg0) {
		// Do Nothing		

	}

	@Override
	public boolean onSingleTapUp(MotionEvent e) {
		// Do Nothing		
		return false;
	}

	@Override
	public boolean onTouch(View arg0, MotionEvent event) {
		if(this.gestureDetector.onTouchEvent(event))
		{
			return true;
		} 
		else
		{
			return false;
		}
	}

	@Override
	public void onRatingChanged(RatingBar arg0, float arg1, boolean arg2) {
	}
}
