class Apk < ActiveRecord::Base
  belongs_to :task
  attr_accessible :name
  attr_protected :revision

  attr_accessible :apk, :apk_file_name
  has_attached_file :apk,
    :url => '/apks/:rails_env/:id.:extension',
    :use_timestamp => false

  validates_presence_of :task, :name, :revision
  validates_uniqueness_of :task_id
  validates_attachment_presence :apk
  validates_acceptance_of :revision, :on => :create, :accept => 1
  validate :validate_apk

  include JsonWithoutTimestamps
  def as_json(options = {})
    ignore_keys = [
      :task_id,
      :apk_content_type,
      :apk_file_name,
      :apk_file_size,
      :apk_updated_at,
    ]
    super options.merge({:except => (options[:except] || []) + ignore_keys})
  end

  after_initialize do |apk|
    if apk.revision.nil?
      apk.revision = 1
    end
  end

  before_update do |apk|
    if apk.apk_updated_at_changed?
      apk.revision = apk.revision_was + 1
    end
  end

  def validate_apk
    if name_changed? and not apk_updated_at_changed?
      errors.add :name, :only_changed
      return
    end
    return if not apk_updated_at_changed?

    # Skip apk check if dexdump is not available
    return unless Rails.configuration.dexdump_path

    path = apk.queued_for_write[:original].instance_variable_get(:@tempfile).to_path
    buf = ''
    status = Open4.popen4(Rails.configuration.dexdump_path.to_s, '-lxml', path) do |pid, stdin, stdout, stderr|
      stdin.close
      buf = stdout.read
    end
    if status.exitstatus == 0
      doc = Nokogiri::XML.parse buf
      ctor = doc.at_xpath("//class/constructor[@type='#{name}']")
      if ctor.nil?
        errors.add :apk, :no_class
      end
    else
      errors.add :apk, :invalid
    end
  end
end
