require 'test_helper'

class AchievementsControllerTest < ActionController::TestCase
  include Devise::TestHelpers

  setup do
    @user = FactoryGirl.create :user
    @achievement = FactoryGirl.create :achievement, :user_id => @user.id
    alt_user = FactoryGirl.create :user
    @alt_achievement = FactoryGirl.create :achievement, :user_id => alt_user.id
    sign_in @user
  end

  test "should get index" do
    get :index
    assert_response :success
    assert_not_nil assigns(:achievements)
  end

  test "should get index with JSON" do
    get :index, :format => :json
    assert_response :success
    assert_not_nil assigns(:achievements)
    j = JSON.parse @response.body
    assert j.is_a?(Array)
    assert_equal 1, j.size
    a = j[0]
    assert_equal @achievement.result, a['result']
    assert_equal @achievement.task.id, a['task_id']
  end

  test "should not get index without sign-in" do
    sign_out @user
    get :index
    assert_redirected_to new_user_session_path
  end

  test "should NOT get new" do
    assert_no_routes do
      get :new
    end
  end

  test "should create achievement" do
    assert_no_routes do
      post :create, achievement: { result: @achievement.result }
    end
  end

  test "should normalize result" do
    assert_difference('Achievement.count') do
      put :update, :format => :json, :task_id => @alt_achievement.task_id, :achievement => { :result => '1,    2' }
    end
    a = assigns :achievement
    assert_not_nil a
    assert_equal '1,2', a.result
  end

  test "should show achievement" do
    get :show, id: @achievement
    assert_response :success
  end

  test "should not get show without sign-in" do
    sign_out @user
    get :show, :id => @achievement
    assert_redirected_to new_user_session_path
  end

  test "should NOT get edit" do
    assert_no_routes do
      get :edit, id: @achievement
    end
  end

  test "should update achievement" do
    put :update, :format => :json, :task_id => @achievement.task_id, achievement: { result: @achievement.result }
    assert_response :success
  end

  test "should NOT destroy achievement" do
    assert_no_routes do
      delete :destroy, id: @achievement
    end
  end

  test "should not show another user's achievement" do
    get :show, :id => @alt_achievement.id
    assert_redirected_to achievements_path
    assert_not_nil flash.alert
  end

  test "should merge results" do
    xs = @achievement.results
    ys = Array.new(xs.size) { 0 }
    ys[0] = 10000
    put :update, :format => :json, :task_id => @achievement.task_id, :achievement => { :result => ys.join(',') }
    updated_achievement = assigns :achievement
    assert_not_nil updated_achievement
    assert_equal Achievement.merge(xs, ys), updated_achievement.results
  end

  test "reject invalid merge" do
    xs = @achievement.results
    ys = xs + [0]
    put :update, :format => :json, :task_id => @achievement.task, :achievement => { :result => ys.join(',') }
    errors = JSON.parse @response.body
    assert errors.has_key?('result')
    assert_match_any /merge/, errors['result']
  end

  test "reject invalid format" do
    assert_difference('Achievement.count', 0) do
      put :update, :format => :json, :task_id => @alt_achievement.task_id, :achievement => { :result => 'a' }
    end
    errors = JSON.parse @response.body
    assert errors.has_key?('result')
    assert_match_any /element/, errors['result']
  end

  test "reject empty result" do
    assert_difference('Achievement.count', 0) do
      put :update, :format => :json, :task_id => @alt_achievement.task_id, :achievement => { :result => '' }
    end
    errors = JSON.parse @response.body
    assert errors.has_key?('result')
    assert_match_any /empty/, errors['result']
  end
end
