require 'pathname'
require 'rexml/document'
require 'erb'

module Android
  class Project# {{{
    def initialize(app_name)
      @app_name = app_name
      # We need Pathname#realpath for Windows
      @base_dir = Pathname.new('.').realpath
    end

    def xml
      @xml ||= REXML::Document.new File.read('AndroidManifest.xml')
    end

    def package
      xml.root.attributes['package']
    end

    MAIN_ACTIVITY_XPATH = '/manifest/application/activity[intent-filter/action[@android:name="android.intent.action.MAIN"]]'
    def main_activity
      if e = REXML::XPath.first(xml, MAIN_ACTIVITY_XPATH)
        name = e.attributes['android:name'].gsub(/^\./, '')
        package + '.' + name
      end
    end

    API_LEVEL_XPATH = '/manifest/uses-sdk[@android:minSdkVersion]'
    def api_level
      if e = REXML::XPath.first(xml, API_LEVEL_XPATH)
        e.attributes['android:minSdkVersion'].to_i
      end
    end

    def output_dir
      @base_dir.join 'bin'
    end

    def class_output_dir
      output_dir.join 'classes'
    end

    def res_output_dir
      output_dir.join 'res'
    end

    def gen_dir
      @base_dir.join 'gen'
    end

    def src_dir
      @base_dir.join 'src'
    end

    def res_dir
      @base_dir.join 'res'
    end

    def res_apk_file
      output_dir.join "#{@app_name}.ap_"
    end

    def asset_dir
      @base_dir.join 'assets'
    end

    def manifest_output_file
      output_dir.join 'AndroidManifest.xml'
    end

    def dex_output_file
      output_dir.join 'classes.dex'
    end

    def apk_unaligned_file
      output_dir.join "#{@app_name}-debug-unaligned.apk"
    end

    def apk_aligned_file
      output_dir.join "#{@app_name}-debug.apk"
    end
  end# }}}

  class Sdk# {{{
    attr_accessor :android_home, :project

    Separator = RbConfig::CONFIG['target_os'] =~ /mswin(?!ce)|mingw|cygwin|bccwin/ ? ';' : ':'

    def initialize(home, project)
      if home.nil?
        raise "home is nil"
      end
      @android_home = Pathname.new home
      @project = project
      detect_platforms
    end

    def detect_platforms
      @platforms = {}
      platforms.entries.each do |e|
        path = platforms.join e
        if path.directory?
          prop = path.join 'source.properties'
          if prop.file? and prop.readable?
            prop.open do |f|
              f.readlines.each do |line|
                line.chomp!
                if m = line.match(/\AAndroidVersion\.ApiLevel=(\d+)\z/)
                  # $stderr.puts "Detected platform api_level=#{m[1]} at #{path.to_s}"
                  @platforms[m[1].to_i] = path
                end
              end
            end
          end
        end
      end
    end

    def platform_tools
      @android_home.join 'platform-tools'
    end

    def platforms
      @android_home.join 'platforms'
    end

    def platform
      if @platforms.has_key? @project.api_level
        @platforms[@project.api_level]
      else
        raise "Platform not found!: api_level=#{@project.api_level}"
      end
    end

    def tools
      @android_home.join 'tools'
    end

    def aapt
      platform_tools.join 'aapt'
    end

    def dx
      platform_tools.join 'dx'
    end

    def adb
      platform_tools.join 'adb'
    end

    def apkbuilder
      tools.join 'apkbuilder'
    end

    def zipalign
      tools.join 'zipalign'
    end

    def android_jar
      platform.join 'android.jar'
    end

    def annotations_jar
      tools.join 'support', 'annotations.jar'
    end

    def classpath
      lib_file = Pathname.new('.').parent.join('jar', 'CUTEnLib.jar').realpath
      [
        @project.class_output_dir.to_s,
        annotations_jar.to_s,
        lib_file.to_s,
      ].join(Separator)
    end
  end# }}}

  class BuildConfigGenerator# {{{
    attr_reader :package, :debug
    def initialize(package, debug)
      @package = package
      @debug = debug
    end

    erb = ERB.new <<-EOS
/** Automatically generated file. DO NOT MODIFY */
package <%= package %>;

public final class BuildConfig {
    public final static boolean DEBUG = <%= debug %>;
}
    EOS
    erb.def_method self, 'generate'
  end# }}}
end
